package com.ibm.oti.connection.ssl;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import javax.microedition.io.Connector;
import javax.microedition.io.StreamConnection;

import com.ibm.oti.connection.CreateConnection;
import com.ibm.j9.bluez.crypto.CL3;

import javax.microedition.io.SecureConnection;
import javax.microedition.io.SecurityInfo;
import javax.microedition.pki.Certificate;

import com.ibm.oti.security.provider.X509Certificate;

import com.ibm.oti.security.midp.PermissionManager;
import javax.microedition.pki.CertificateException;

import com.ibm.j9.ssl.CipherAlgorithm;
import com.ibm.j9.ssl.CipherSpec;
import com.ibm.j9.ssl.ConnectionState;
import com.ibm.j9.ssl.HandshakeHandler;
import com.ibm.j9.ssl.HandshakeSocket;
import com.ibm.j9.ssl.J9HandshakeException;
import com.ibm.j9.ssl.Record;
import com.ibm.j9.ssl.SSLProtocol;
import com.ibm.j9.ssl.SessionState;
import com.ibm.j9.ssl.StreamQueue;

public class Connection implements CreateConnection, StreamConnection, SecureConnection, HandshakeSocket {

	com.ibm.oti.connection.socket.Connection socket = null;
	InputStream inputStream = null;
	OutputStream outputStream = null;

	private static final int UNOPENED = 0, OPEN = 1, CLOSED = 2;
	int inputStatus = UNOPENED;
	int outputStatus = UNOPENED;

	private String host;
	private int port;

	SessionState sessionState = null;
	private ConnectionState readConnectionState = null;
	private ConnectionState writeConnectionState = null;
	private ConnectionState pendingReadConnectionState = null;
	private HandshakeHandler handshakeHandler = new HandshakeHandler(this);

	private static boolean rngInitialised = false;

	private final StreamQueue applicationDataInputQueue = new StreamQueue();

	private final class SSLOutputStream extends OutputStream {

		private SSLOutputStream() {
			super();
		}

		public void write(byte[] buffer, int offset, int count) throws IOException {
			if (outputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			writeData(buffer, offset, count, SSLProtocol.CONTENT_APPLICATION_DATA);
		}

		public void write(byte[] buffer) throws IOException {
			if (outputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			writeData(buffer, 0, buffer.length, SSLProtocol.CONTENT_APPLICATION_DATA);
		}

		public void write(int oneByte) throws IOException {
			if (outputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			write(new byte[]{(byte)oneByte});
		}

		public void close() throws IOException {
			if( outputStatus != CLOSED ){
				outputStatus = CLOSED;
				outputStream.close();
				if (inputStatus == CLOSED) closeConnection();
			}
		}
	}

	private final class SSLInputStream extends InputStream {
		private final StreamQueue applicationDataInputQueue;

		private SSLInputStream(StreamQueue applicationDataInputQueue) {
			super();
			this.applicationDataInputQueue = applicationDataInputQueue;
		}

		public int available() throws IOException {
			if (inputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			if (applicationDataInputQueue.getReadStream().available() == 0 && inputStream == null) {
				// no more queued data and the input stream has been closed
				return -1;
			}

			// if there is more data on the stream and we have none, get some
			while (applicationDataInputQueue.getReadStream().available() == 0 && inputStream != null && inputStream.available() > 0) {
				readData();
			}

			return applicationDataInputQueue.getReadStream().available();
		}

		public int read() throws IOException {
			if (inputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			// must block until we have at least some data
			while (applicationDataInputQueue.getReadStream().available() == 0 && inputStream != null) {
				readData();
			}

			if (applicationDataInputQueue.getReadStream().available() == 0 && inputStream == null) {
				// the inputStream has reached the end of the stream and there is no queued data
				return -1;
			}

			return applicationDataInputQueue.getReadStream().read();
		}

		public int read(byte[] b, int offset, int length) throws IOException {
			if (inputStatus == CLOSED)
				// K0059 = Stream is closed
				throw new IOException(com.ibm.oti.util.Msg.getString("K0059")); //$NON-NLS-1$
			// must block until we have at least some data
			while (applicationDataInputQueue.getReadStream().available() == 0 && inputStream != null) {
				readData();
			}

			// if there is more data on the stream and we need more, get it
			while (inputStream != null && inputStream.available() > 0 && applicationDataInputQueue.getReadStream().available() < length) {
				readData();
			}

			if (inputStream == null && applicationDataInputQueue.getReadStream().available() == 0) {
				// the inputStream has reached the end of the stream and there is no queued data
				return -1;
			}

			//TODO: readlen is never read.  This might indicate a bug in the return result.  See bug
//			int readlen = applicationDataInputQueue.getReadStream().available() >= length ? length : applicationDataInputQueue.getReadStream().available();
			return applicationDataInputQueue.getReadStream().read(b, offset, length);
		}

		public int read(byte[] b) throws IOException {
			return read(b, 0, b.length);
		}

		public void close() throws IOException {
			if (inputStatus != CLOSED) {
				inputStatus = CLOSED;
				if (inputStream != null) inputStream.close();
				if (outputStatus == CLOSED) closeConnection();
			}
		}
	}

	public Connection() {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap())
			throw new SecurityException();
		if (PermissionManager.getManager() != null)
			PermissionManager.getManager().checkPermission("javax.microedition.io.Connector.ssl");
	}

	public Connection(com.ibm.oti.connection.socket.Connection socket, String host) throws IOException {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap())
			throw new SecurityException();
		this.host = host;
		this.socket = socket;
		connect();
	}

	/**
	 * Method openDataInputStream.
	 * @return DataInputStream
	 * @throws IOException
	 */
	public DataInputStream openDataInputStream() throws IOException {
		return new DataInputStream(openInputStream());
	}

	/**
	 * Method openDataOutputStream.
	 * @return DataOutputStream
	 * @throws IOException
	 */
	public DataOutputStream openDataOutputStream() throws IOException {
		return new DataOutputStream(openOutputStream());
	}

	/**
	 * Method openInputStream.
	 * @return InputStream
	 * @throws IOException
	 */
	public InputStream openInputStream() throws IOException {
		// K0192 = Stream already opened
		if (inputStatus != UNOPENED) throw new IOException(com.ibm.oti.util.Msg.getString("K0192")); //$NON-NLS-1$
		inputStatus = OPEN;
		return new SSLInputStream(applicationDataInputQueue);
	}

	/**
	 * Method openOutputStream.
	 * @return OutputStream
	 * @throws IOException
	 */
	public OutputStream openOutputStream() throws IOException {
		// K0192 = Stream already opened
		if (outputStatus != UNOPENED) throw new IOException(com.ibm.oti.util.Msg.getString("K0192")); //$NON-NLS-1$
		outputStatus = OPEN;
		return new SSLOutputStream();
	}

	/**
	 * Passes the parameters from the Connector.open() method to this
	 * object. Protocol used by MIDP 2.0
	 *
	 * @author		OTI
	 * @version		initial
	 *
	 * @param		spec String
	 *					The address passed to Connector.open()
	 * @param		access int
	 *					The type of access this Connection is
	 *					granted (READ, WRITE, READ_WRITE)
	 * @param		timeout boolean
	 *					A boolean indicating whether or not the
	 *					caller to Connector.open() wants timeout
	 *					exceptions
	 * @exception	IOException
	 *					If an error occured opening and configuring
	 *					serial port.
	 *
	 * @see javax.microedition.io.Connector
	 */
	public javax.microedition.io.Connection setParameters2(String spec, int access, boolean timeout) throws IOException {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap())
			throw new SecurityException();

		setParameters(spec, access, timeout);
		return this;
	}

	/**
	 * Passes the parameters from the Connector.open() method to this
	 * object. Protocol used by MIDP 1.0
	 *
	 * @author		OTI
	 * @version		initial
	 *
	 * @param		spec String
	 *					The address passed to Connector.open()
	 * @param		access int
	 *					The type of access this Connection is
	 *					granted (READ, WRITE, READ_WRITE)
	 * @param		timeout boolean
	 *					A boolean indicating whether or not the
	 *					caller to Connector.open() wants timeout
	 *					exceptions
	 * @exception	IOException
	 *					If an error occured opening and configuring
	 *					serial port.
	 *
	 * @see javax.microedition.io.Connector
	 */
	public void setParameters(String spec, int access, boolean timeout) throws IOException {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap())
			throw new SecurityException();

		String hostPart = null;
		if (!spec.startsWith("//")) { //$NON-NLS-1$
			throw new IllegalArgumentException();
		}
		spec = spec.substring(2);
		int index = spec.indexOf('/');
		if (index == -1) {
			hostPart = spec;
		} else {
			hostPart = spec.substring(0, index);
		}
		if ((index = hostPart.indexOf(':')) != -1) {
			host = hostPart.substring(0, index);
			String portString = hostPart.substring(index + 1);
			try {
				port = Integer.parseInt(portString);
			} catch (NumberFormatException e) {
				// K00b1 = Invalid port number
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00b1")); //$NON-NLS-1$
			}
			if (port < 0 || port > 65535)
				// K0325 = Port out of range: {0}
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0325", port)); //$NON-NLS-1$
		} else {
			// K03a1 = No port specified
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K03a1")); //$NON-NLS-1$
		}
		String socketSpec = "//" + host + ":" + port;  //$NON-NLS-1$//$NON-NLS-2$
		this.socket = new com.ibm.oti.connection.socket.Connection();
		((CreateConnection)socket).setParameters(socketSpec, Connector.READ_WRITE, timeout);

		connect();
	}

	/**
	 * Method close.
	 * @throws IOException
	 */
	public void close() throws IOException {
		closeConnection();
	}

	void closeConnection() throws IOException {
		inputStatus = CLOSED;
		if (outputStatus != CLOSED) {
			sendAlert(SSLProtocol.ALERT_LEVEL_WARNING, SSLProtocol.ALERT_CLOSE_NOTIFY);
			outputStatus = CLOSED;
		}
		if (socket != null) {

			if( inputStream != null ){
				inputStream.close();
				inputStream = null;
			}
			if( outputStream != null ){
				outputStream.close();
				outputStream = null;
			}

			socket.close();
			socket = null;
		}
	}

	private void connect() throws IOException {
		if (socket == null) throw new IOException();

		this.inputStream = socket.openInputStream();
		this.outputStream = socket.openOutputStream();

		SessionState nullSession = new SessionState(getHostName());

		ConnectionState nullConnection = new ConnectionState();
		nullConnection.setSessionState(nullSession);
		this.readConnectionState = nullConnection;
		this.writeConnectionState = nullConnection;

		//Set the RNG prior to starting handshake to avoid timeout on connection
		if( rngInitialised == false ){
			CL3.rng( null, new byte[1], 0, 1 );
			rngInitialised = true;
		}
		try{
            handshakeHandler.run();
        }
		catch (J9HandshakeException e) {
			throw new javax.microedition.pki.CertificateException(e.getMessage(), null, javax.microedition.pki.CertificateException.VERIFICATION_FAILED);
		}
		catch( IOException e ){
            this.closeSocketStreams();
            throw e;
        }

	}

	private SecurityInfo securityInfo = new SecurityInfo() {
		public String getCipherSuite() {
			return "TLS_" + sessionState.getCipherSpec().toString(); //$NON-NLS-1$
		}
		public String getProtocolName() {
			return "SSL"; //$NON-NLS-1$
		}
		public String getProtocolVersion() {
			return "3.0"; //$NON-NLS-1$
		}
		public Certificate getServerCertificate() {
			return getCertificate();
		}
	};

	public String getServerCertSubject(){
		if (sessionState==null || sessionState.getPeerCertificates()==null || sessionState.getPeerCertificates().length==0)
			return null;
		try {
			X509Certificate cert = X509Certificate.certificateFromASN1Object(sessionState.getPeerCertificates()[0].getEncoded());
			return cert.getSubject();
		} catch (CertificateException e) {
			return null;
		}
	}

	public X509Certificate getCertificate() {
		if (sessionState==null || sessionState.getPeerCertificates()==null || sessionState.getPeerCertificates().length==0)
			return null;
		X509Certificate result;
		try {
			result = X509Certificate.certificateFromASN1Object(sessionState.getPeerCertificates()[0].getEncoded());
		} catch (CertificateException e) {
			return null;
		}
		return result;
	}

	/**
	 * Answers the security information associated with this connection.
	 *
	 * @return the security information for this connection
	 * @throws IOException if a connection failure occurs
	 */
	public SecurityInfo getSecurityInfo() throws IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return securityInfo;
	}

	/**
	 * Writes a record to the ssl socket.
	 *
	 * @param data			The data to write.
	 * @param contentType	The content type of the data (see Record for constants).
	 *
	 * @throws IOException
	 */
	private void writeRecord(byte[] data, byte contentType) throws IOException {
		// encrypt the compressed data
		byte[] cipherData;
		if (writeConnectionState.getCipherAlgorithm() != null)
			cipherData = writeConnectionState.getCipherAlgorithm().encipher(data, contentType);
		else
			cipherData = data;

		Record record = new Record( contentType, SSLProtocol.SSL_PROTOCOL_VERSION, cipherData );
		record.writeRecord( outputStream );

		writeConnectionState.transmitSequenceNumber++;
	}

	/**
	 * Reads the next record from the ssl socket.  This operation blocks until
	 * enough data is read to fill the record.
	 *
	 * @throws IOException
	 */
	protected Record readRecord() throws IOException {
		if (inputStream==null) {
			return null;
		}

		// read the data from the input stream
		Record record = Record.readRecord(inputStream);
		if( record == null ){
			try{inputStream.close();}catch(IOException e ){/*Ignore*/}
			inputStream = null;
		}

		return record;
	}

	/**
	 * Writes the data to the ssl socket.
	 *
	 * @param data			The data to write.
	 * @param contentType	The content type of the data (see Record for constants).
	 *
	 * @throws IOException
	 */
	public void writeData(byte[] data, int offset, int length, byte contentType) throws IOException {
		if (contentType==SSLProtocol.CONTENT_APPLICATION_DATA && handshakeHandler.shouldHandshake()) {
			handshakeHandler.run();
		}
		int i = 0;
		while(i < length) {
			byte[] writedata = new byte[length - i < Record.MAX_RECORD_LENGTH ? length - i : Record.MAX_RECORD_LENGTH];
			System.arraycopy(data, offset+i, writedata, 0, writedata.length);
			writeRecord(writedata, contentType);
			i += Record.MAX_RECORD_LENGTH;
		}
	}

	/**
	 * Reads an SSL record from the underlying socket and processes it.
 	 * @return boolean - true if stream is open, false if stream was closed (i.e. read() returns -1)
	 *
	 * @throws IOException
	 */
	 /* [PR 119578] Sametime SSL socket impl usage creates loop-forever hang */
	 public boolean readData() throws IOException {

		// Read a record
		Record record = readRecord();
		if (record==null){
			/* [PR 119578] Sametime SSL socket impl usage creates loop-forever hang */
			return false;
		}

		// Decrypt the record data
		byte[] recordData = null;
		CipherAlgorithm cipher = this.readConnectionState.getCipherAlgorithm();
		try {
			if (cipher != null){
				recordData = cipher.decipher(record.getData(), record.getContentType());
			} else {
				recordData = record.getData();
			}
		} catch (IOException e) {
			sendAlert( SSLProtocol.ALERT_LEVEL_FATAL, SSLProtocol.ALERT_BAD_RECORD_MAC );
			throw e;
		}
		// the receive sequence number isn't updated until after the data is decrypted because the sequence
		// number is checked in the decryption method
		readConnectionState.receiveSequenceNumber++;

		// Process the record
		switch(record.getContentType()) {
			case SSLProtocol.CONTENT_APPLICATION_DATA:
				applicationDataInputQueue.getWriteStream().write(recordData);
				break;
			case SSLProtocol.CONTENT_CHANGE_CIPHER_SPEC:
				// process all queued handshake messages before changing the cipher spec
				// this ensures all necessary computation has been done
				handshakeHandler.processQueuedMessages();
				handleChangeCipherSpec(recordData);
				break;
			case SSLProtocol.CONTENT_ALERT:
				handleAlert(recordData);
				break;
			case SSLProtocol.CONTENT_HANDSHAKE:
				handshakeHandler.notifyReceived(recordData);
				break;
		}
		/* [PR 119578] Sametime SSL socket impl usage creates loop-forever hang */
		return true;
	}

	/**
	 * Handles the receipt of a change_cipher_spec message.
	 */
	private void handleChangeCipherSpec(byte[] message) throws IOException {
		// switch to the pending cipher spec for subsequent reads
		if (pendingReadConnectionState == null){
			// K01d4 = Unexpected SSL message: change_cipher_spec
			throw new IOException(com.ibm.oti.util.Msg.getString("K01d4")); //$NON-NLS-1$
		}
		this.readConnectionState = pendingReadConnectionState;
	}

	void sendAlert(byte level, byte description) throws IOException {
		byte[] data = new byte[2];
		data[0] = level;
		data[1] = description;
		writeRecord(data, SSLProtocol.CONTENT_ALERT);
	}

	public void sendSocketAlert( byte level, byte description ){
		try {
			sendAlert( level, description );
		} catch (IOException e) {
			// No handling, the API requires that this is done on a best-effort basis only
		}
	}

	public String getHostName(){
		return host;
	}

	/**
	 * Handles the receipt of a alert message.
	 */
	private void handleAlert(byte[] message) throws IOException {
		if (message[0] == SSLProtocol.ALERT_LEVEL_FATAL) {

			String msgstr;
			if (message[1]==SSLProtocol.ALERT_BAD_CERTIFICATE)
				// K01d5 = bad certificate
				msgstr = com.ibm.oti.util.Msg.getString("K01d5"); //$NON-NLS-1$
			else if (message[1]== SSLProtocol.ALERT_BAD_RECORD_MAC)
				// K01d6 = bad record mac
				msgstr = com.ibm.oti.util.Msg.getString("K01d6"); //$NON-NLS-1$
			else if (message[1]==SSLProtocol.ALERT_DECOMPRESSION_FAILURE)
				// K01d7 = decompression failure
				msgstr = com.ibm.oti.util.Msg.getString("K01d7"); //$NON-NLS-1$
			else if (message[1]==SSLProtocol.ALERT_HANDSHAKE_FAILURE)
				// K01d8 = handshake failure
				msgstr = com.ibm.oti.util.Msg.getString("K01d8"); //$NON-NLS-1$
			else if (message[1]==SSLProtocol.ALERT_ILLEGAL_PARAMETER)
				// K01d9 = illegal parameter
				msgstr = com.ibm.oti.util.Msg.getString("K01d9"); //$NON-NLS-1$
			else if (message[1]==SSLProtocol.ALERT_NO_CERTIFICATE)
				// K01da = no client certificate
				msgstr = com.ibm.oti.util.Msg.getString("K01da"); //$NON-NLS-1$
			else if (message[1]==SSLProtocol.ALERT_UNEXPECTED_MESSAGE)
				// K01db = unexpected message
				msgstr = com.ibm.oti.util.Msg.getString("K01db"); //$NON-NLS-1$
			else
				// K01dc = invalid SSL error code {0}
				msgstr = com.ibm.oti.util.Msg.getString("K01dc", message[1]); //$NON-NLS-1$

			// K01dd = SSL error received: {0}
			throw new IOException(com.ibm.oti.util.Msg.getString("K01dd", msgstr)); //$NON-NLS-1$
		}
	}

	/**
	 * Sets the session state.
	 */
	public void setSessionState(SessionState state) {
		this.sessionState = state;
	}

	/**
	 * Sets the connection state for messages written to the socket.  This causes a
	 * change_cipher_spec message to be transmitted.
	 */
	public void setWriteConnectionState(ConnectionState state) throws IOException {
		// send the change cipher spec message
		writeData(new byte[]{0x01}, 0, 1, SSLProtocol.CONTENT_CHANGE_CIPHER_SPEC);

		// switch to the new state
		this.writeConnectionState = state;
	}

	/**
	 * Sets the pending connection state for messages read from the socket.  The pending
	 * state will be copied to the current state when a change_cipher_spec message is received
	 * from the server.
	 */
	public void setReadPendingConnectionState(ConnectionState state) throws IOException {
		this.pendingReadConnectionState = state;
	}

	/**
	 * Answers the local address for the connection.
	 * @return String	the local address
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public String getLocalAddress() throws IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return socket.getLocalAddress();
	}

    private void closeSocketStreams(){
        try{
			// The following wasn't really the problem, but was discovered when investigating
			// to be sure that it had already been solved.
			if( this.inputStream != null ){
				this.inputStream.close();
			}
			if( this.outputStream != null ){
				this.outputStream.close();
			}
        } catch( IOException e ) {
            // No handling - we tried to close it
        }
    }

	public long getTimeoutLength() {
		return this.socket.getTimeout() * 2;
	}

	/**
	 * Answers the address of the remote host that the connection accesses.
	 * @return String	the address of the remote host
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public String getAddress() throws IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return socket.getAddress();
	}

	/**
	 * Answers the port number of the connection on the remote host.
	 *
	 * @return int	the port number
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getPort() throws IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return socket.getPort();
	}

	/**
	 * Answers the number of the local port that the connection is on.
	 *
	 * @return int	the local port number
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getLocalPort() throws IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return socket.getLocalPort();
	}

	/**
	 * Answers the value of the selected option.  The option passed to the function
	 * must be one of the constants DELAY, KEEPALIVE, LINGER, RCVBUF, SNDBUF.
	 *
	 * @param optname
	 * @return int	the option value to retrieve
	 * @throws IllegalArgumentException	if the option requested is not one of the legal constants
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getSocketOption(byte optname) throws IllegalArgumentException, IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		return socket.getSocketOption(optname);
	}

	/**
	 * Sets the value of the selected option.  The choice of option passed to the function
	 * must be one of the constants DELAY, KEEPALIVE, LINGER, RCVBUF, SNDBUF.
	 *
	 * @param optname	the option option to return
	 * @param optval	the value to set
	 *
	 * @throws IllegalArgumentException	if the new option value is invalid
	 * or the option is not one of the legal constants
	 * @throws IOException	if the connection has already been closed
	 */
	public void setSocketOption(byte optname, int optval) throws IllegalArgumentException, IOException {
		if (inputStatus == CLOSED && outputStatus == CLOSED) throw new IOException();
		socket.setSocketOption(optname, optval);
	}

	public CipherSpec[] getEnabledCipherSpecs() {
		return CipherSpec.SUPPORTED_SPECS;
	}

	public String[] getEnabledProtocols() {
		return new String[] {SSLProtocol.SSL_PROTOCOL_NAME}; //$NON-NLS-1$
	}

}
