package com.ibm.oti.crypto;

import java.io.IOException;

import com.ibm.j9.bluez.crypto.CL3;

/**
 * <p>
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 * </p>
  */
public abstract class CL3BasedProvider extends Provider {

	int mode;
	byte[] iv;

	protected CL3BasedProvider(int algorithm, int effectiveKeyBitLength) {
		super(algorithm, effectiveKeyBitLength);
	}

	/**
	 * @see com.ibm.oti.crypto.Provider#destroyKey(Key)
	 */
	final void destroyKey(Key key) {
		// It is necessary to destroy the CryptoLite representation of the key
		if (key instanceof CL3BasedKey) {
			CL3BasedKey cl3Key = (CL3BasedKey) key;
			CL3.dispose(cl3Key.workingKey);
			cl3Key.workingKey = null;
		}
	}

	/**
	 * @see com.ibm.oti.crypto.Provider#cryptInit(Key, int, int, byte[])
	 */
	final void cryptInit(Key key, int operation, int padtype, byte[] iv) throws IOException {

		// Verify key is a CL3BasedKey - this is needed for CryptoLite interactions
		if(!(key instanceof CL3BasedKey)) {
			/* [MSG "K01fa", "The key could not be imported"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K01fa" ) ); //$NON-NLS-1$
		}

		this.mode = operation;
		if (!isValidPadType(padtype)) {
			/* [MSG "K01f7", "invalid padding"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K01f7" ) ); //$NON-NLS-1$
		}

		// Validate IV - it must be provided (CBC requires it, ECB not supported)
		if ((iv == null ) || (iv.length != getIVLength())) {
			/* [MSG "K01f6", "invalid IV"] */
			throw new IOException(com.ibm.oti.util.Msg.getString( "K01f6" ) ); //$NON-NLS-1$
		} else {
			this.iv = new byte[getIVLength()];
			System.arraycopy( iv, 0, this.iv, 0, getIVLength() );
		}
	}

	byte[] cryptUpdate(Key key, byte[] bytes, int offset, int length, boolean finished) throws IOException {
		// Verify key is a valid CL3Based key
		if (!(key instanceof CL3BasedKey)) {
			/* [MSG "K01fa", "The key could not be imported"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K01fa" ) ); //$NON-NLS-1$
		}

		// Perform operation - error checking on operation type performed by cryptInit
		if (mode == Key.OPERATION_ENCRYPT) {
			return encryptImpl((CL3BasedKey) key, bytes, offset, length, finished);
		} else {
			return decryptImpl((CL3BasedKey) key, bytes, offset, length, finished);
		}
	}

	/**
	 * Validate pad type for provider.  Subclasses may have to over ride.
	 */
	boolean isValidPadType(int padType) {
		return (padType == Key.PAD_TLS) || (padType == Key.PAD_SSL) || (padType == Key.PAD_PKCS5);
	}

	/**
	 * Method  pads and encrypts data
	 * @param key
	 * @param bytes
	 * @param offset
	 * @param length
	 * @param finished
	 * @return byte[] data encrypted with key and padding provided
	 * @throws IOException
	 */

	final byte[] encryptImpl(CL3BasedKey key, byte[] bytes, int offset, int length, boolean finished) throws IOException {

		// Get length characteristics of data
		int fullBlocks = length / getBlockLength();
		int remainderBytes = length % getBlockLength();
		int lastBlockIndex = fullBlocks * getBlockLength();

		// Confirm that data divisible into blocks if not finished
		if (!finished && remainderBytes != 0) {
			/* [MSG "K0170", "invalid data length"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K0170" ) ); //$NON-NLS-1$
		}

		// Compute destination size: if finished, add a block for padding
		// IMPORTANT:  Padding for DES and 3DES always adds an extra full block to the output
		// TODO: This code does not implement what the above comment says.  Confirm that implementation
		//       is ok and adjust comment.
		//       If data is composed of full blocks only then one block of padding is added to the end.  If, however,
		//       the data contains a partial block, that partial block is padded but no additional block is added.
		byte[] outputBuffer;
		if (finished) {
			outputBuffer = new byte[(fullBlocks + 1) * getBlockLength()];
		} else {
			outputBuffer = new byte[lastBlockIndex];
		}

		// Encrypt full blocks
		cl3Call(key.workingKey, CL3.ENCRYPT, this.iv, 0, bytes, offset, outputBuffer, 0, lastBlockIndex);

		// Pad and encrypt remainder if necessary
		if (finished) {
			// Pad the block to specified scheme
			byte[] tempBlock = null;
			// TODO: If this really works get rid of the padding idea.  Actually we want to support the adding of random pad blocks so take that into account
			switch( key.padtype ){
				case Key.PAD_SSL:
					tempBlock = Util.padSSL(bytes, offset + lastBlockIndex, remainderBytes, getBlockLength());
					break;
				case Key.PAD_TLS:
					tempBlock = Util.padTLS10(bytes, offset + lastBlockIndex, remainderBytes, getBlockLength());
					break;
				case Key.PAD_PKCS5:
					tempBlock = Util.padPKCS5(bytes, offset + lastBlockIndex, remainderBytes, getBlockLength());
					break;
			}

			// Encrypt the final block
			cl3Call(key.workingKey, CL3.ENCRYPT, this.iv, 0, tempBlock, 0, outputBuffer, lastBlockIndex, getBlockLength());
		}
		return outputBuffer;
	}

	abstract void cl3Call(CL3 workingKey, int encrypt, byte[] bs, int i, byte[] bytes, int offset, byte[] outputBuffer, int j, int lastBlockIndex);

	/**
	 * Method decrypts data and removes padding from the final block, if any.
	 * Data must be provided in full blocks.
	 * @param key
	 * @param bytes
	 * @param offset
	 * @param length
	 * @param finished
	 * @return byte[] decrypted and unpadded data
	 * @throws IOException If data format and length not compatible with encrypted data
	 */

	final byte[] decryptImpl(CL3BasedKey key, byte[] bytes, int offset, int length, boolean finished) throws IOException {
		// Verify that input is evenly divisible into blocks
		if (length % getBlockLength() != 0) {
			/* [MSG "K0170", "invalid data length"] */
			throw new IOException( com.ibm.oti.util.Msg.getString( "K0170" ) ); //$NON-NLS-1$
		}

		// Get length characteristics of data
		int blockCount = length / getBlockLength();
		int lastBlockIndex = (blockCount - 1) * getBlockLength();

		// If finished, we must decrypt all but last block which must be checked for padding
		if (finished) {
			blockCount--;
		}

		byte[] outputBuffer = new byte[blockCount * getBlockLength()];

		cl3Call(key.workingKey, CL3.DECRYPT, this.iv, 0, bytes, offset, outputBuffer, 0, blockCount * getBlockLength());

		// If finished, we must check the last block for padding
		if (finished) {
			byte[] tempBuf = new byte[getBlockLength()];
			cl3Call(key.workingKey, CL3.DECRYPT, this.iv, 0, bytes, offset + lastBlockIndex, tempBuf, 0, getBlockLength());

			// Unpad the final block according the padding scheme specified
			byte[] unpadded = null;
			switch(key.padtype){
				case Key.PAD_TLS:
					unpadded = Util.unpadTLS10(tempBuf);
					break;
				case Key.PAD_SSL:
					unpadded = Util.unpadSSL(tempBuf);
					break;
				case Key.PAD_PKCS5:
					unpadded = Util.unpadPKCS5(tempBuf);
					break;
			}
			return Util.concatenate(outputBuffer, unpadded);
		} else {
			return outputBuffer;
		}
	}

	public abstract Key createKey(byte[] keybytes) throws IOException;
}
