package com.ibm.oti.crypto;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.IOException;
import java.util.Enumeration;
import java.util.Vector;

public class NativeProvider extends Provider {

private long providerId;
int instanceCount;

private static Vector providerCache = new Vector();
private static Runnable shutdownHook = null;

static NativeProvider getNativeProvider(int algorithm, int keylength) {
	NativeProvider provider = null;

	// look in the cache first
	if (providerCache!=null) {
		for(int i=0; i<providerCache.size(); i++) {
			provider = (NativeProvider)providerCache.elementAt(i);
			if (provider.getAlgorithm()==algorithm && provider.getEffectiveKeyBitLength()==keylength) {
				provider.instanceCount++;
				return provider;
			}
		}
	}

	// try to create a new native provider
	try {
		long providerId = createProviderImpl(algorithm, keylength);
		provider = new NativeProvider(algorithm, keylength, providerId);
		providerCache.addElement(provider);
		return provider;
	} catch (IOException e) {
		return null;
	}
}

protected NativeProvider(int algorithm, int effectiveKeyBitLength, long providerId) {

	super(algorithm, effectiveKeyBitLength);

	// check to see if a shutdown hook is registered
	if (shutdownHook == null) {
		shutdownHook = new Runnable() {
			public void run() {
				Enumeration entries = providerCache.elements();
				while( entries.hasMoreElements()){
					NativeProvider provider = (NativeProvider)providerCache.elementAt(0);

					// Must set this to 1 because the count is decremented in destroy
					// before check for 0 for actual destruction
					provider.instanceCount=1;

					provider.destroy();
				}
			}
		};
		com.ibm.oti.vm.VM.addShutdownClass(shutdownHook);
	}

	this.providerId = providerId;
	this.instanceCount = 1;

}

public Key createKey(byte[] keybytes) throws IOException {

	// If no keybytes have been provided we must create key in native
	if( keybytes == null ){
		throw new NullPointerException();
	}

	long keyId = createKeyImpl(providerId, keybytes);
	if (keyId != 0) {
		return new NativeKey(this, keybytes, keyId);
	} else {
		// K01fa = The key could not be imported
		throw new IOException(com.ibm.oti.util.Msg.getString("K01fa"));
	}
}

void destroyKey(Key key) {
	destroyKeyImpl(((NativeKey)key).keyId);
}

void cryptInit(Key key, int operation, int padtype, byte[] iv) {

	// it is valid to pass a NULL IV, so no check performed
	cryptInitImpl(((NativeKey)key).keyId, operation, padtype, iv);

}

byte[] cryptUpdate(Key key, byte[] bytes, int offset, int length, boolean finished) {

	// It is an error not to provide any bytes to be encrypted, and we must not call native
	if( bytes == null ){
		throw new NullPointerException();
	}

	return cryptUpdateImpl(((NativeKey)key).keyId, bytes, offset, length, finished);
}

public void destroy() {
	instanceCount--;
	if (instanceCount==0) {
		destroyProviderImpl(providerId);
		providerCache.removeElement(this);
		super.destroy();
	}
}

protected synchronized static native long createProviderImpl(int algorithm, int keylength) throws IOException;
protected synchronized native void destroyProviderImpl(long providerId);

synchronized native long createKeyImpl(long providerId, byte[] keybytes);
synchronized native void destroyKeyImpl(long keyId);

synchronized native void cryptInitImpl(long keyId, int operation, int padtype, byte[] iv);
synchronized native byte[] cryptUpdateImpl(long keyId, byte[] bytes, int offset, int length, boolean finished);

}
