package com.ibm.oti.crypto;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.IOException;

public abstract class Provider {
	/**
	 * Algorithm code: DES
	 */
	public static final int ALG_DES = 1;
	/**
	 * Algorithm code: 3DES
	 */
	public static final int ALG_3DES = 2;
	/**
	 * Algorithm code: RC2
	 */
	public static final int ALG_RC2 = 3;
	/**
	 * Algorithm code: RC4
	 */
	public static final int ALG_RC4 = 4;
	/**
	 * Algorithm code: AES
	 */
	public static final int ALG_AES = 5;
	/**
	 * Algorithm code: NULL
	 */
	public static final int ALG_NULL = 6;

	protected int algorithm;
	protected int effectiveKeyBitLength;
	protected boolean isDestroyed = false;

/**
 * Returns a provider instance for the given algorithm, such as DES or 3DES.
 *
 * @param		algorithm				The algorithm to use
 * @param		effectiveKeyBitLength	The effective key bit length
 * @return		an instance of a provider for the given algorithm
 *
 * @throws 		IOException 	if the algorithm could not be found or the effective key bit length is not supported for the algorithm.
 */
public static Provider getProvider(int algorithm, int effectiveKeyBitLength) throws IOException {

	// try to find a native provider
	Provider provider = NativeProvider.getNativeProvider(algorithm, effectiveKeyBitLength);
	if (provider != null) return provider;

	// try to find a java provider
	switch(algorithm) {
		case ALG_DES:
			return new DESProvider(effectiveKeyBitLength);
		case ALG_3DES:
			return new TripleDESProvider(effectiveKeyBitLength);
		case ALG_NULL:
			return new NullProvider();
		default:
			throw new IOException();
	}
}

protected Provider(int algorithm, int effectiveKeyBitLength) {
	this.algorithm = algorithm;
	this.effectiveKeyBitLength = effectiveKeyBitLength;
}

/**
 * Returns the algorithm supported by this provider.
 *
 * @return		the integer constant representing the algorithm
 */
public int getAlgorithm() {
	return algorithm;
}

/**
 * Returns the byte length of a block for this provider.  If this provider implements a
 * stream cipher, 0 is returned.
 *
 * Sublcasses may not override since Native Providers don't have API to retun the appropriate values
 *
 * @return		the number of bytes in a block
 */
public final int getBlockLength() {
	switch (algorithm) {
		case ALG_DES:
		case ALG_3DES:
		case ALG_RC2:
			return 8;
		case ALG_RC4:
			return 0;
		case ALG_AES:
			return 16;
		case ALG_NULL:
		default:
			return 0;
	}
}

/**
 * Returns the number of bits of random data contained in a key for this provider.
 *
 * @return		the effective key bit length.
 */
public int getEffectiveKeyBitLength() {
	return effectiveKeyBitLength;
}

/**
 * Returns the byte length of a key for this provider.  This is the number of bytes in the
 * key and may be greater than the effective key length.
 *
 * Sublcasses may not override since Native Providers don't have API to retun the appropriate values
 *
 * @return		the number of bytes in a key
 */
public final int getKeyLength() {
	switch(algorithm) {
		case ALG_DES:
			return 8;
		case ALG_3DES:
			return 24;
		case ALG_RC2:
		case ALG_RC4:
		case ALG_AES:
			return 16;
		case ALG_NULL:
			return 0;
		default:
			return -1;
	}
}

/**
 * Returns the byte length of an initialization vector (IV) for this provider.
 *
 * Sublcasses may not override since Native Providers don't have API to retun the appropriate values
 * @return		the number of bytes in an initialization vector
 */
public final int getIVLength() {
	switch(algorithm) {
		case ALG_DES:
		case ALG_3DES:
		case ALG_AES:
		case ALG_RC2:
		case ALG_RC4:
		case ALG_NULL:
			return getBlockLength();
		default:
			return -1;
	}
}

/**
 * Returns whether this provider has been destroyed by a call to destroy().
 *
 * @return		<code>true</code> if this provider is destroyed, <code>false</code> otherwise.
 */
public boolean isDestroyed() {
	return isDestroyed;
}

/**
 * Creates a new key using this provider with the specified key.
 *
 * @param 		keybytes	The byte array representing the key.
 * @return 		Object for cryptographic operations using the key.
 *
 * @throws 		IOException 	if the key could not be created
 */
public Key createKey(byte[] keybytes) throws IOException {
	return new Key(this, keybytes);
}

/**
 * Destroys this provider.
 */
public void destroy() {
	if (!isDestroyed) {
		isDestroyed = true;
	}
}

abstract void destroyKey(Key key);
abstract void cryptInit(Key key, int operation, int padtype, byte[] iv) throws IOException;
abstract byte[] cryptUpdate(Key key, byte[] bytes, int offset, int length, boolean finished) throws IOException;
}
