package com.ibm.oti.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

import java.util.Hashtable;

/**
 * CharacterConverter is the superclass of all classes used for converting
 * between characters and the byte representations used by different
 * encoding methods. It provides a default conversion such that every byte
 * is converted to the character having the byte value. Characters are converted
 * to a single byte having the value of the low 8 bits of the character value.
 *
 * @author		OTI
 * @version		initial
 */
public class CharacterConverter {
	private String name;

	private static byte[] EMPTY = new byte[0];

	private static final boolean useNative = com.ibm.oti.vm.VM.useNatives();
	private static Hashtable mappings = new Hashtable(350);

	static void initializeMappings() {
		// All aliases should use uppercase letters.
		// Aliases with all '-' and no '_' are found when
		// either '-' or '_' are used in the name to lookup.
		// So a alias containing an '_' much match exactly.

		mappings.put("8859-1", "ISO8859_1");
		mappings.put("ISO8859-1", "ISO8859_1");
		mappings.put("ISO-8859-1", "ISO8859_1");
		mappings.put("ISO-8859-1:1987", "ISO8859_1");
		mappings.put("ISO-IR-100", "ISO8859_1");
		mappings.put("LATIN1", "ISO8859_1");
		mappings.put("CSISOLATIN1", "ISO8859_1");
		mappings.put("CP819", "ISO8859_1");
		mappings.put("IBM-819", "ISO8859_1");
		mappings.put("IBM819", "ISO8859_1");
		mappings.put("L1", "ISO8859_1");

		mappings.put("8859-2", "ISO8859_2");
		mappings.put("ISO8859-2", "ISO8859_2");
		mappings.put("ISO-8859-2", "ISO8859_2");
		mappings.put("ISO-8859-2:1987", "ISO8859_2");
		mappings.put("ISO-IR-101", "ISO8859_2");
		mappings.put("LATIN2", "ISO8859_2");
		mappings.put("CSISOLATIN2", "ISO8859_2");
		mappings.put("CP912", "ISO8859_2");
		mappings.put("IBM-912", "ISO8859_2");
		mappings.put("IBM912", "ISO8859_2");
		mappings.put("L2", "ISO8859_2");

		mappings.put("8859-3", "ISO8859_3");
		mappings.put("ISO8859-3", "ISO8859_3");
		mappings.put("ISO-8859-3", "ISO8859_3");
		mappings.put("ISO-8859-3:1988", "ISO8859_3");
		mappings.put("ISO-IR-109", "ISO8859_3");
		mappings.put("LATIN3", "ISO8859_3");
		mappings.put("CSISOLATIN3", "ISO8859_3");
		mappings.put("CP913", "ISO8859_3");
		mappings.put("IBM-913", "ISO8859_3");
		mappings.put("L3", "ISO8859_3");

		mappings.put("8859-4", "ISO8859_4");
		mappings.put("ISO8859-4", "ISO8859_4");
		mappings.put("ISO-8859-4", "ISO8859_4");
		mappings.put("ISO-8859-4:1988", "ISO8859_4");
		mappings.put("ISO-IR-110", "ISO8859_4");
		mappings.put("LATIN4", "ISO8859_4");
		mappings.put("CSISOLATIN4", "ISO8859_4");
		mappings.put("CP914", "ISO8859_4");
		mappings.put("IBM-914", "ISO8859_4");
		mappings.put("L4", "ISO8859_4");

		mappings.put("8859-5", "ISO8859_5");
		mappings.put("ISO8859-5", "ISO8859_5");
		mappings.put("ISO-8859-5", "ISO8859_5");
		mappings.put("ISO-8859-5:1988", "ISO8859_5");
		mappings.put("ISO-IR-144", "ISO8859_5");
		mappings.put("CYRILLIC", "ISO8859_5");
		mappings.put("CSISOLATINCYRILLIC", "ISO8859_5");
		mappings.put("CP915", "ISO8859_5");
		mappings.put("IBM-915", "ISO8859_5");
		mappings.put("IBM915", "ISO8859_5");

		mappings.put("8859-6", "ISO8859_6");
		mappings.put("ARABIC", "ISO8859_6");
		mappings.put("ASMO-708", "ISO8859_6");
		mappings.put("CP1089", "ISO8859_6");
		mappings.put("CSISOLATINARABIC", "ISO8859_6");
		mappings.put("ECMA-114", "ISO8859_6");
		mappings.put("IBM-1089", "ISO8859_6");
		mappings.put("IBM1089", "ISO8859_6");
		mappings.put("ISO-8859-6", "ISO8859_6");
		mappings.put("ISO-8859-6:1987", "ISO8859_6");
		mappings.put("ISO-IR-127", "ISO8859_6");
		mappings.put("ISO8859-6", "ISO8859_6");

		mappings.put("8859-7", "ISO8859_7");
		mappings.put("CP813", "ISO8859_7");
		mappings.put("ISO-8859-7", "ISO8859_7");
		mappings.put("CSISOLATINGREEK", "ISO8859_7");
		mappings.put("ECMA-118", "ISO8859_7");
		mappings.put("ELOT-928", "ISO8859_7");
		mappings.put("GREEK", "ISO8859_7");
		mappings.put("GREEK8", "ISO8859_7");
		mappings.put("IBM-813", "ISO8859_7");
		mappings.put("IBM813", "ISO8859_7");
		mappings.put("ISO-8859-7:1987", "ISO8859_7");
		mappings.put("ISO-IR-126", "ISO8859_7");
		mappings.put("ISO8859-7", "ISO8859_7");

		mappings.put("8859-8", "ISO8859_8");
		mappings.put("CP916", "ISO8859_8");
		mappings.put("CSISOLATINHEBREW", "ISO8859_8");
		mappings.put("HEBREW", "ISO8859_8");
		mappings.put("IBM-916", "ISO8859_8");
		mappings.put("IBM916", "ISO8859_8");
		mappings.put("ISO-8859-8", "ISO8859_8");
		mappings.put("ISO-8859-8:1988", "ISO8859_8");
		mappings.put("ISO-IR-138", "ISO8859_8");
		mappings.put("ISO8859-8", "ISO8859_8");

		mappings.put("8859-9", "ISO8859_9");
		mappings.put("ISO8859-9", "ISO8859_9");
		mappings.put("ISO-8859-9", "ISO8859_9");
		mappings.put("ISO-IR-148", "ISO8859_9");
		mappings.put("LATIN5", "ISO8859_9");
		mappings.put("CSISOLATIN5", "ISO8859_9");
		mappings.put("CP920", "ISO8859_9");
		mappings.put("IBM-920", "ISO8859_9");
		mappings.put("IBM920", "ISO8859_9");
		mappings.put("L5", "ISO8859_9");

		mappings.put("8859-10", "ISO8859_10");
		mappings.put("ISO-8859-10", "ISO8859_10");
		mappings.put("ISO_8859-10:1992", "ISO8859_10");
		mappings.put("CSISOLATIN6", "ISO8859_10");
		mappings.put("IBM-919", "ISO8859_10");
		mappings.put("ISO-IR-157", "ISO8859_10");
		mappings.put("ISO8859-10", "ISO8859_10");
		mappings.put("L6", "ISO8859_10");
		mappings.put("LATIN6", "ISO8859_10");

		mappings.put("8859-13", "ISO8859_13");
		mappings.put("ISO-8859-13", "ISO8859_13");
		mappings.put("ISO8859-13", "ISO8859_13");

		mappings.put("8859-14", "ISO8859_14");
		mappings.put("ISO-8859-14", "ISO8859_14");
		mappings.put("ISO_8859-14:1998", "ISO8859_14");
		mappings.put("ISO-IR-199", "ISO8859_14");
		mappings.put("ISO8859-14", "ISO8859_14");
		mappings.put("ISOCELTIC", "ISO8859_14");
		mappings.put("L8", "ISO8859_14");
		mappings.put("LATIN8", "ISO8859_14");

		mappings.put("8859-15", "ISO8859_15");
		mappings.put("ISO8859-15", "ISO8859_15");
		mappings.put("ISO-8859-15", "ISO8859_15");
		mappings.put("LATIN0", "ISO8859_15");
		mappings.put("LATIN9", "ISO8859_15");
		mappings.put("CSISOLATIN0", "ISO8859_15");
		mappings.put("CSISOLATIN9", "ISO8859_15");
		mappings.put("CP923", "ISO8859_15");
		mappings.put("IBM-923", "ISO8859_15");
		mappings.put("IBM923", "ISO8859_15");
		mappings.put("ISO8859_15_FDIS", "ISO8859_15");
		mappings.put("L9", "ISO8859_15");

		mappings.put("8859-16", "ISO8859_16");
		mappings.put("ISO-8859-16", "ISO8859_16");
		mappings.put("ISO8859-16", "ISO8859_16");

		mappings.put("SHIFT-JIS", "MS932");
		mappings.put("MS_KANJI", "MS932");
		mappings.put("CSSHIFTJIS", "MS932");
		mappings.put("CSWINDOWS31J", "MS932");
		mappings.put("WINDOWS-31J", "MS932");
		mappings.put("X-SJIS", "MS932");
		mappings.put("WINDOWS-932", "MS932");

		mappings.put("936", "MS936");
		mappings.put("WINDOWS-936", "MS936");

		mappings.put("CP1361", "MS949");
		mappings.put("MS949", "MS949");
		mappings.put("IBM-1361", "MS949");
		mappings.put("IBM1361", "MS949");
		mappings.put("KSC5601-1992", "MS949");
		mappings.put("MS1361", "MS949");
		mappings.put("MS_949", "MS949");
		mappings.put("WINDOWS-949", "MS949");
		mappings.put("X-WINDOWS-949", "MS949");

		mappings.put("950", "MS950");
		mappings.put("WINDOWS-950", "MS950");
		mappings.put("X-WINDOWS-950", "MS950");

		mappings.put("EUCJIS", "EUC_JP");
		mappings.put("CSEUCPKDFMTJAPANESE", "EUC_JP");
		mappings.put("EXTENDED_UNIX_CODE_PACKED_FORMAT_FOR_JAPANESE", "EUC_JP");
		mappings.put("EUC-JP", "EUC_JP");
		mappings.put("EUCJP", "EUC_JP");
		mappings.put("X-EUC-JP", "EUC_JP");
		mappings.put("X-EUCJP", "EUC_JP");
		mappings.put("EUC-JP-LINUX", "EUC_JP");

		mappings.put("KSC5601", "EUC_KR");
		mappings.put("KS_C_5601-1987", "EUC_KR");
		mappings.put("KSC-5601", "EUC_KR");
		mappings.put("EUC-KR", "EUC_KR");
		mappings.put("EUCKR", "EUC_KR");
		mappings.put("KSC5601-1987", "EUC_KR");
		mappings.put("5601", "EUC_KR");
		mappings.put("CP970", "EUC_KR");
		mappings.put("IBM-970", "EUC_KR");
		mappings.put("IBM-EUCKR", "EUC_KR");

		mappings.put("JIS", "ISO2022JP");
		mappings.put("ISO-2022-JP", "ISO2022JP");
		mappings.put("CSISO2022JP", "ISO2022JP");
		mappings.put("JIS-ENCODING", "ISO2022JP");
		mappings.put("CSJISENCODING", "ISO2022JP");
		mappings.put("ISO2022-JP", "ISO2022JP");

		mappings.put("BIG5-HKSCS", "BIG5");
		mappings.put("BIG5-0", "BIG5");

		mappings.put("GB18030-2000", "GB18030");
		mappings.put("IBM-1392", "GB18030");
		mappings.put("WINDOWS-54936", "GB18030");

		mappings.put("IBM-437", "CP437");
		mappings.put("IBM437", "CP437");
		mappings.put("437", "CP437");
		mappings.put("CSPC8CODEPAGE437", "CP437");

		mappings.put("IBM-850", "CP850");
		mappings.put("IBM850", "CP850");
		mappings.put("850", "CP850");
		mappings.put("CSPC850MULTILINGUAL", "CP850");

		mappings.put("IBM-852", "CP852");
		mappings.put("IBM852", "CP852");
		mappings.put("852", "CP852");
		mappings.put("CSPCP852", "CP852");

		mappings.put("IBM-858", "CP858");
		mappings.put("IBM858", "CP858");
		mappings.put("858", "CP858");
		mappings.put("CSPC858MULTILINGUAL", "CP858");

		mappings.put("IBM-860", "CP860");
		mappings.put("IBM860", "CP860");
		mappings.put("860", "CP860");
		mappings.put("CSIBM860", "CP860");

		mappings.put("IBM-863", "CP863");
		mappings.put("IBM863", "CP863");
		mappings.put("863", "CP863");
		mappings.put("CSIBM863", "CP863");

		mappings.put("IBM-865", "CP865");
		mappings.put("IBM865", "CP865");
		mappings.put("865", "CP865");
		mappings.put("CSIBM865", "CP865");

		mappings.put("IBM-866", "CP866");
		mappings.put("IBM866", "CP866");
		mappings.put("866", "CP866");

		mappings.put("IBM-1250", "CP1250");
		mappings.put("WINDOWS-1250", "CP1250");

		mappings.put("IBM-1251", "CP1251");
		mappings.put("WINDOWS-1251", "CP1251");

		mappings.put("IBM-1252", "CP1252");
		mappings.put("WINDOWS-1252", "CP1252");

		mappings.put("IBM-1253", "CP1253");
		mappings.put("WINDOWS-1253", "CP1253");

		mappings.put("IBM-1254", "CP1254");
		mappings.put("WINDOWS-1254", "CP1254");

		mappings.put("IBM-1255", "CP1255");
		mappings.put("WINDOWS-1255", "CP1255");

		mappings.put("IBM-1256", "CP1256");
		mappings.put("WINDOWS-1256", "CP1256");

		mappings.put("IBM-1257", "CP1257");
		mappings.put("WINDOWS-1257", "CP1257");

		mappings.put("IBM-1129", "CP1258");
		mappings.put("IBM-1258", "CP1258");
		mappings.put("WINDOWS-1258", "CP1258");

		mappings.put("UTF-8", "UTF8");

		mappings.put("646", "ASCII");
		mappings.put("ANSI_X3.4-1968", "ASCII");
		mappings.put("ANSI_X3.4-1986", "ASCII");
		mappings.put("CP367", "ASCII");
		mappings.put("ISO-646.IRV:1991", "ASCII");
		mappings.put("ISO646-US", "ASCII");
		mappings.put("US-ASCII", "ASCII");
		mappings.put("ASCII7", "ASCII");
		mappings.put("CSASCII", "ASCII");
		mappings.put("DEFAULT", "ASCII");
		mappings.put("DIRECT", "ASCII");
		mappings.put("IBM-367", "ASCII");
		mappings.put("ISO-646.IRV:1983", "ASCII");
		mappings.put("ISO-IR-6", "ASCII");
		mappings.put("US", "ASCII");

		mappings.put("UTF-16BE", "UNICODEBIGUNMARKED");
		mappings.put("UTF-16LE", "UNICODELITTLEUNMARKED");
		mappings.put("UTF-16", "UTF16");
	}

/**
 * Constructs a new CharacterConverter which provides the default conversion
 * between characters and bytes. See the class comment for details.
 *
 * @author		OTI
 * @version		initial
 */
public CharacterConverter() {
}

/**
 * Counts the numbers of characters represented by the
 * specified number of bytes starting at the specified
 * offset in the byte array.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the byte array
 * @param		offset	the starting offset in the byte array
 * @param		count	the number of bytes to convert
 * @return		the number of characters represented by the specified
 *			bytes, or -1 if there are no characters available and
 *			the bytes are invalid for this conversion
 */
public int countChars(byte[] value, int offset, int count) {
	return count;
}

/**
 * Converts the specified number of characters from the byte
 * array.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the byte array
 * @param		offset	the starting offset in the byte array
 * @param		count	the number of bytes to convert
 * @return		the character array with the converted characters
 *
 * @exception	IndexOutOfBoundsException when offset < 0, charOffset < 0,
 *				charOffset + total > chars.length
 */
public char[] convert(byte[] value, int offset, int length) {
	int numChars = countChars(value, offset, length);
	if (numChars == -1) {
		return new char[0];
	}
	char[] chars = new char[numChars];
	convert(value, offset, chars, 0, numChars);
	return chars;
}

/**
 * Converts the specified number of characters from the byte
 * array.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the byte array
 * @param		offset	the starting offset in the byte array
 * @param		length	the number of bytes to convert
 * @param		buffer  the CharBuffer object containing the buffer char array
 * @return		the offset of the first unconverted byte in the byte array
 *
 * @exception	IndexOutOfBoundsException when offset < 0, charOffset < 0,
 *				charOffset + total > chars.length
 */
public int convert(byte[] value, int offset, int length, CharBuffer buffer) {
	int result = countChars(value, offset, length);
	if (result == -1) return result;
	if (result > buffer.getSize()) result = buffer.getSize();
	int charOffset = buffer.getOffset();
	int bytePos = convert(value, offset, buffer.getChars(), charOffset, result);
	buffer.setPos(charOffset + result);
	buffer.setOffset(charOffset + result);
	return bytePos;
}

/**
 * Converts the specified number of characters from the byte
 * array.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the byte array
 * @param		offset	the starting offset in the byte array
 * @param		chars	the character array
 * @param		charOffset the starting offset in the char array
 * @param		total	the number of characters to convert
 * @return		the offset of the first unconverted byte in the byte array
 *
 * @exception	IndexOutOfBoundsException when offset < 0, charOffset < 0,
 *				charOffset + total > chars.length
 */
public int convert(byte[] value, int offset, char[] chars, int charOffset, int total) {
	if (useNative)
		return convertImpl(value, offset, chars, charOffset, total);
	for (int i=total; --i >= 0;) {
		chars[charOffset++] = (char)((int)value[offset++] & 0xff);
	}
	return offset;
}

private native int convertImpl(byte[] value, int offset, char[] chars, int charOffset, int total);

/**
 * Converts the specified number of characters starting at the specified offset
 * in the character array into a byte array. The number of bytes may be greater
 * than the number of characters, depending on the encoding used.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the character array
 * @param		offset	the starting offset in the character array
 * @param		count	the number of characters to convert
 * @return		a byte array containing the converted characters
 *
 * @exception	IndexOutOfBoundsException when offset < 0, count < 0, or
 *				offset + count > value.length
 */
public byte[] convert(char[] value, int offset, int count) {
	if (useNative)
		return convertImpl(value, offset, count);
	int end = offset + count, i = 0;
	byte[] result = new byte[count];
	while (offset < end) {
		int ch = value[offset++];
		if (ch > 0xff) {
			if (ch >= 0xd800 && ch < 0xdc00 &&
				offset < end &&	value[offset] >= 0xdc00 &&
				value[offset] < 0xe000)
					offset++;
			result[i++] = '?';
		} else result[i++] = (byte)ch;
	}
	if (i < result.length) {
		byte[] newResult = new byte[i];
		System.arraycopy(result, 0, newResult, 0, i);
		return newResult;
	}
	return result;
}

private native byte[] convertImpl(char[] value, int offset, int count);

/**
 * Creates a new CharacterConverter for the specified encoding
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		encoding	the encoding method
 * @return		a new CharacterConverter for the encoding, or null if there is no
 *				converter for the encoding
 */
public static CharacterConverter getConverter(String encoding) {
	Class converterClass = null;
	// bootstrapping problem - cannot call String.toUpperCase() as this causes
	// Locale to get initialized before the System properties are initialized
	char[] upper = new char[encoding.length()];
	for (int i=0; i<upper.length; i++) {
		upper[i] = Character.toUpperCase(encoding.charAt(i));
	}
	encoding = new String(upper);
	String className = (String)mappings.get(encoding);
	if (className == null && mappings.size() != 0) {
		for (int i=0; i<upper.length; i++) {
			if (upper[i] == '_') {
				upper[i] = '-';
			}
		}
		encoding = new String(upper);
		className = (String)mappings.get(encoding);
	}
	if (className != null) encoding = className;
	try {
		converterClass = Class.forName("com.ibm.oti.io.CharacterConverter_" + encoding);
	} catch (ClassNotFoundException e) {
		if (mappings.size() != 0) {
			return getNativeConverter(encoding);
		}
		initializeMappings();
		className = (String)mappings.get(encoding);
		if (className == null) {
			for (int i=0; i<upper.length; i++) {
				if (upper[i] == '_') {
					upper[i] = '-';
				}
			}
			encoding = new String(upper);
			className = (String)mappings.get(encoding);
		}
		if (className != null) {
			try {
				converterClass = Class.forName("com.ibm.oti.io.CharacterConverter_" + className);
			} catch (ClassNotFoundException e2) {
				return getNativeConverter(className);
			}
		} else {
			return getNativeConverter(encoding);
		}
	}
	try {
		CharacterConverter converter = (CharacterConverter)converterClass.newInstance();
		converter.setName(encoding);
		return converter;
	}
	catch (IllegalAccessException e) {return null;}
	catch (InstantiationException e) {return null;}
}

/**
 * Gets a NativeCharacterConverter object if native character conversion is supported.
 *
 * @param encoding the Java code page identifier
 * @return a NativeCharacterConverter object if native conversion is supported or NULL if not supported
 */
private static CharacterConverter getNativeConverter(String encoding) {
	boolean useNativeConverter = NativeCharacterConverter.supportsNativeConversion();
	NativeCharacterConverter nativeConverter = null;

	if (useNativeConverter) {
		// check if the given code page is supported by the OS
		nativeConverter = new NativeCharacterConverter();
		useNativeConverter = nativeConverter.supportsCodePage(encoding);
	}
	if (useNativeConverter) {
		// encoding conversion supported by device's OS
		nativeConverter.setJavaEncoding(encoding);
		nativeConverter.setName(encoding);
		return nativeConverter;
	} else {
		return null;
	}
}

/**
 * Creates a new CharacterConverter for the specified encoding
 * method. If the encoding does not exist, return the default
 * converter.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		encoding	the encoding method
 * @return		a new CharacterConverter for the encoding
 */
public static CharacterConverter getDefaultConverter(String encoding) {
	CharacterConverter converter = getConverter(encoding);
	if (converter == null) converter = getConverter("ISO8859_1");
	if (converter == null) return new CharacterConverter();
	return converter;
}

public boolean isCalled(String encoding) {
	return name.equals(encoding);
}

public void setName(String encoding) {
	name = encoding;
}

/**
 * Return a modeless CharacterConverter, to be used for a
 * single conversion, such as used by java.lang.String. The
 * instance must be thread safe.
 *
 * @return a thread safe CharacterConverter
 */
public CharacterConverter getModeless() {
	return this;
}

/**
 * Return any bytes required to close the stream after
 * a char[] to byte[] conversion. Not called by
 * modeless conversions, such as java.lang.String.
 *
 * @return the byte[] sequence to close the conversion
 */
public byte[] getClosingBytes() {
	return EMPTY;
}

}
