package com.ibm.oti.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

public class NativeCharacterConverter extends CharacterConverter {

	private String javaEncoding;
	private long osCodePage = -1;
	private static boolean supportsNativeConv = supportsNativeCharConv();

	public void setJavaEncoding(String encoding) {
		this.javaEncoding = encoding;
	}

	/**
	 * Checks if the platform has implemented support for
	 * native character converters.
	 *
	 * @return true is native character conversion is supported or false otherwise
	 */
	private static native boolean supportsNativeCharConv();

	/**
	 * Checks if the given code page is supported by the OS.
	 *
	 * @param encoding the Java code page identifier
	 * @return true if code page is supported by the OS or false otherwise
	 */
	native boolean supportsCodePage(String encoding);

	/**
	 * Checks if the Native Character Conversion is implemented and supported
	 * by the current OS.
	 *
	 * @return true if native character conversion is supported, false otherwise
	 */
	public static boolean supportsNativeConversion() {
		return supportsNativeConv;
	}

	/**
	 * Converts the specified number of characters starting at the specified offset
	 * in the character array into a byte array. The number of bytes may be greater
	 * than the number of characters, depending on the encoding used.
	 *
	 * @param		value	the character array
	 * @param		offset	the starting offset in the character array
	 * @param		count	the number of characters to convert
	 * @return		a byte array containing the converted bytes
	 *
	 * @exception	IndexOutOfBoundsException when offset < 0, count < 0, or
	 *				offset + count > value.length
	 */
	public byte[] convert(char[] value, int offset, int count) {
		if (0 <= offset && offset <= value.length && 0 <= count && count <= value.length - offset) {
			if (count == 0) {
				return new byte[0];
			}
			return convertCharsToBytesImpl(value, offset, count, this.javaEncoding, this.osCodePage);
		} else {
			throw new IndexOutOfBoundsException();
		}
	}

	private native byte[] convertCharsToBytesImpl(char[] srcText, int offset, int count, String encoding, long osCodePage);

	/**
	 * Converts the specified number of characters from the byte
	 * array.
	 *
	 * @param		value	the byte array
	 * @param		offset	the starting offset in the byte array
	 * @param		count	the number of bytes to convert
	 * @return		the character array with the converted characters
	 *
	 * @exception	IndexOutOfBoundsException when offset < 0, charOffset < 0,
	 *				charOffset + total > chars.length
	 */
	public char[] convert(byte[] value, int offset, int count) {
		if (0 <= offset && offset <= value.length && 0 <= count && count <= value.length - offset) {
			if (count == 0) {
				return new char[0];
			}
			return convertBytesToCharsImpl(value, offset, count, this.javaEncoding, this.osCodePage);
		} else {
			throw new IndexOutOfBoundsException();
		}
	}

	private native char[] convertBytesToCharsImpl(byte[] value, int offset, int count, String encoding, long osCodePage);

	/**
	 * Uses the OS natives to convert the specified number of characters from the byte
	 * array.
	 *
	 * @param		value	the byte array
	 * @param		offset	the starting offset in the byte array
	 * @param		length	the number of bytes to convert
	 * @param		buffer  the CharBuffer object containing the buffer char array
	 * @return		the offset of the first unconverted byte in the byte array
	 *
	 * @exception	IndexOutOfBoundsException when offset < 0, charOffset < 0,
	 *				offset + count > value.length
	 */
	public int convert(byte[] value, int offset, int count, CharBuffer buffer) {
		int bytePos = 0;
		int[] charPos = new int[1];
		charPos[0] = buffer.getPos();

		if (0 <= offset && offset <= value.length && 0 <= count && count <= value.length - offset) {
			bytePos = convertStreamBytesToCharsImpl(value, offset, count, buffer.getChars(), buffer.getOffset(), buffer.getSize(), charPos, this.javaEncoding, this.osCodePage);
			buffer.setPos(charPos[0]);
			return bytePos;
		} else {
			throw new IndexOutOfBoundsException();
		}
	}

	private native int convertStreamBytesToCharsImpl(byte[] srcText, int srcOffset, int srcCount, char[] dstText, int dstOffset, int dstCount, int[] dstPos, String encoding, long osCodePage);

}
