package com.ibm.oti.security.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;

import com.ibm.oti.security.provider.RSAPublicKey;
import com.ibm.oti.security.provider.X509Certificate;
import com.ibm.oti.security.provider.X500Principal;

import javax.microedition.pki.CertificateException;

public class KeyStore {

	public static final int USAGE_TRUSTED_SSL_SERVER = 1;
	public static final int USAGE_PROTECTION_DOMAIN_ROOT = 2;

	private static final int CURRENT_KEYSTORE_VERSION = 2;

	private static KeyStore systemKeyStore = null;

	private int version;
	private boolean readOnly = false;

	private static Entry mruEntry = null;

	/**
	 * Returns the global system keystore.
	 */
	public static KeyStore getSystemKeyStore() {
		if (systemKeyStore==null) {
			systemKeyStore = new KeyStore();

			systemKeyStore.setReadOnly();
		}
		return systemKeyStore;
	}

	private static InputStream getInputStreamOnKeyStore(){

		InputStream is = null;

		String fileProperty;
		if ((fileProperty=System.getProperty("com.ibm.j2mekeystore.path")) != null) {
			// load the keystore from an absolute path
			is = SecurityPolicy.getStreamForPath(fileProperty, false);
		}

		String resourceProperty = null;
		if (is==null && (resourceProperty=System.getProperty("com.ibm.j2mekeystore.resourcename")) != null) {
			// load the keystore from a resource
			is = SecurityPolicy.getStreamForResourceName(resourceProperty);
		}

		if (fileProperty==null && resourceProperty==null) {
			// load the keystore from the default location if no system property was set
			is = SecurityPolicy.getStreamForPath("lib/j2me.keystore", true);
		}

		return is;

	}

	public abstract static class Entry {

		public static final int FORMAT_X509 = 1;
		public static final int FORMAT_COMPRESSED = 2;

		protected int format;
		protected int usage;
		protected String protectionDomain;

		/**
		 * Returns the usage of this entry.  This is an OR of one or more of the USAGE_* constants of KeyStore.
		 *
		 * For example, 3 would be CERT_TRUSTED_SSL_SERVER | CERT_PROTECTION_DOMAIN_ROOT, which indicates that
		 * this certificate is valid as both a trusted ssl server and as a protection domain root.
		 */
		public int getUsage() {
			return usage;
		}

		/**
		 * Returns whether this certificate may be used to establish secure socket connections (SSL)
		 * with servers that have this certificate in thier certificate chain.
		 */
		public boolean isTrustedSSLServerCertificate() {
			return (usage&USAGE_TRUSTED_SSL_SERVER)!=0;
		}

		/**
		 * Returns the protection domain that this certificate is bound to or null if
		 * this certificate is not bound to a protection domain.
		 */
		public String getProtectionDomain() {
			return protectionDomain;
		}

		/**
		 * Returns the subject distinguished name for this entry.
		 */
		public abstract X500Principal getSubject();

		/**
		 * Returns the issuer distinguished name for this entry.
		 */
		public abstract X500Principal getIssuer();

		/**
		 * Return the date when the certificate starts to be valid.
		 */
		public abstract long getValidFrom();

		/**
		 * Return the date when the certificate stops being valid.
		 */
		public abstract long getValidTo();

		/**
		 * Returns the public key of this entry.
		 */
		public abstract RSAPublicKey getPublicKey();

		/**
		 * Returns the encoded public key of this entry.
		 */
		public abstract byte[] getPublicKeyBytes();

		protected abstract void readFromImpl(DataInputStream is) throws IOException;

		protected abstract void writeToImpl(DataOutputStream os) throws IOException;

		protected static Entry readFrom(DataInputStream is) throws IOException {
			int format = is.readInt();

			int usage = is.readInt();

			// if it had USAGE_PROTECTION_DOMAIN_ROOT read the protection domain
			String protectionDomain = null;
			if (0 != (usage&USAGE_PROTECTION_DOMAIN_ROOT))
				protectionDomain = is.readUTF();

			Entry entry = createEntry(format);
			entry.usage = usage;
			entry.protectionDomain = protectionDomain;
			entry.readFromImpl(is);
			return entry;
		}

		private static Entry createEntry(int format) {
			Entry entry;
			if (format==FORMAT_X509) {
				entry = new X509Entry();
			} else {
				// the format is FORMAT_COMPRESSED
				entry = new CompressedEntry();
			}
			entry.format = format;
			return entry;
		}

		public void writeTo(DataOutputStream os) throws IOException {
			os.writeInt(format);

			os.writeInt(usage);

			// if it had USAGE_PROTECTION_DOMAIN_ROOT write the protection domain
			if (0 != (usage&USAGE_PROTECTION_DOMAIN_ROOT))
				os.writeUTF(protectionDomain);

			writeToImpl(os);
		}
	}

	public static class CompressedEntry extends Entry {

		private String subjectDNString;
		private X500Principal subjectDN;
		private String issuerDNString;
		private X500Principal issuerDN;
		private byte[] publicKeyBytes;
		private RSAPublicKey publicKey;
		private byte[] serialNumber;
		private long validFrom;
		private long validTo;

		private CompressedEntry() {
		}

		public X500Principal getSubject() {
			if (subjectDN==null) {
				subjectDN = new X500Principal(subjectDNString);
			}
			return subjectDN;
		}

		public X500Principal getIssuer() {
			if (issuerDN==null) {
				issuerDN = new X500Principal(issuerDNString);
			}
			return issuerDN;
		}

		public long getValidFrom() {
			return validFrom;
		}

		public long getValidTo() {
			return validTo;
		}

		public RSAPublicKey getPublicKey() {
			if (publicKey==null) {
				publicKey = new RSAPublicKey(publicKeyBytes);
			}
			return publicKey;
		}

		public byte[] getPublicKeyBytes() {
			return publicKeyBytes;
		}

		protected void readFromImpl(DataInputStream is) throws IOException {
			subjectDNString = is.readUTF();
			subjectDN = null;
			if (is.readBoolean()) {
				// the issuer DN is the same as the subject DN
				issuerDNString = subjectDNString;
				issuerDN = null;
			} else {
				issuerDNString = is.readUTF();
				issuerDN = null;
			}

			validFrom = is.readLong();
			validTo = is.readLong();

			int len = is.readInt();
			publicKeyBytes = new byte[len];
			is.read(publicKeyBytes);
			publicKey = null;

			len = is.readInt();
			serialNumber = new byte[len];
			is.read(serialNumber);
		}

		protected void writeToImpl(DataOutputStream os) throws IOException {
			if (subjectDNString == null)
				subjectDNString = subjectDN.toString();
			os.writeUTF(subjectDNString);
			if (issuerDNString == null)
				issuerDNString = issuerDN.toString();
			if (subjectDNString.equals(issuerDNString)) {
				// the issuer DN is the same as the subject DN (don't write both)
				os.writeBoolean(true);
			} else {
				// else, they are different (write issuer DN)
				os.writeBoolean(false);
				os.writeUTF(issuerDNString);
			}

			os.writeLong(validFrom);
			os.writeLong(validTo);
			if (publicKeyBytes==null)
				getPublicKeyBytes();
			os.writeInt(publicKeyBytes.length);
			os.write(publicKeyBytes);
			os.writeInt(serialNumber.length);
			os.write(serialNumber);
		}

	}

	public static class X509Entry extends Entry {

		private X509Certificate cert;
		private byte[] encoded;
		private X500Principal subjectDN = null;
		private X500Principal issuerDN = null;
		private RSAPublicKey publicKey = null;
		private byte[] serialNumber;

		private X509Entry() {
		}

		public X500Principal getSubject() {
			if (subjectDN==null) {
				subjectDN = new X500Principal(cert.getSubject());
			}
			return subjectDN;
		}

		public X500Principal getIssuer() {
			if (subjectDN==null) {
				subjectDN = new X500Principal(cert.getSubject());
			}
			return subjectDN;
		}

		public long getValidFrom() {
			return cert.getNotBefore();
		}

		public long getValidTo() {
			return cert.getNotAfter();
		}

		public RSAPublicKey getPublicKey() {
			if (publicKey==null) {
				publicKey = new RSAPublicKey(cert.getPublicKey());
			}
			return publicKey;
		}

		public byte[] getPublicKeyBytes() {
			return cert.getPublicKey();
		}

		protected void readFromImpl(DataInputStream is) throws IOException {
			try {
				cert = X509Certificate.certificateFromASN1Object(is);
			} catch (CertificateException e) {
				throw new IOException("CertificateException: " + e.getMessage());
			}
		}

		protected void writeToImpl(DataOutputStream os) throws IOException {
				os.write(cert.getEncoded());
		}

	}

	public KeyStore() {
		version = CURRENT_KEYSTORE_VERSION;
	}

	/**
	 * Sets this keystore as read only.  Any subsequent calls to addEntry(), deleteEntry(), or readFrom() will throw
	 * an exception.
	 */
	private void setReadOnly() {
		readOnly = true;
	}

	/**
	 * Returns the trusted certificate entry with the given subject distinguished name or null if none is found.
	 *
	 * @param	name	X509Principal	The subject distinguished name.
	 */
	public Entry getEntry( X500Principal name ){

		if( (mruEntry != null) && (name.equals( mruEntry.getSubject())) ){
			return mruEntry;
		}

		InputStream is = getInputStreamOnKeyStore();
		if( is == null ){
			// K0345 = Cannot open system keystore
			System.err.println( com.ibm.oti.util.Msg.getString( "K0345" ) );
			return null;
		}

		Entry result = null;
		try{
			DataInputStream dis = new DataInputStream(is);

			version = dis.readInt();
			if (version != CURRENT_KEYSTORE_VERSION) {
				// K0008 = Cannot read version
				System.err.println( com.ibm.oti.util.Msg.getString( "K0008" ) );
				return result;
			}

			int entryCount = dis.readInt();
			while(entryCount-- > 0) {
				Entry entry = Entry.readFrom(dis);
				if( entry.getSubject().equals( name ) ){
					mruEntry = entry;
					result = entry;
					return result;
				}
			}
		} catch( IOException ex ){
			System.err.println( ex.getMessage() );
			ex.printStackTrace();
		} finally {
			try{ is.close(); } catch( IOException ex ){}
		}

		return result;

	}

}
