package com.ibm.oti.security.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003  All Rights Reserved
 */

import java.util.Enumeration;
import java.util.Hashtable;

public class ProtectionDomain {

	/**
	 * Permission levels defined for protection domains
	 */
	public static final int PERMISSION_LEVEL_DENY = 1;
	public static final int PERMISSION_LEVEL_USER_ONESHOT = 2;
	public static final int PERMISSION_LEVEL_USER_SESSION = 3;
	public static final int PERMISSION_LEVEL_USER_BLANKET = 4;
	public static final int PERMISSION_LEVEL_ALLOW = 5;

	/**
	 * bindings between protection domain permission levels and user permission modes.
	 * For example, for the protection domain permission level of oneshot, there are
	 * three acceptable user permission modes: deny always, deny once, and grant once
	 */
	private static final int[] USER_PERMISSION_MODES_FOR_ONESHOT = new int[] {
		UserPermissionAgent.MODE_DENY_ONCE,
		UserPermissionAgent.MODE_GRANT_ONCE };
	private static final int[] USER_PERMISSION_MODES_FOR_SESSION = new int[] {
		UserPermissionAgent.MODE_DENY_ONCE,
		UserPermissionAgent.MODE_GRANT_ONCE,
		UserPermissionAgent.MODE_GRANT_SESSION,
		UserPermissionAgent.MODE_DENY_SESSION };
	private static final int[] USER_PERMISSION_MODES_FOR_BLANKET = new int[] {
		UserPermissionAgent.MODE_DENY_ALWAYS,
		UserPermissionAgent.MODE_DENY_ONCE,
		UserPermissionAgent.MODE_GRANT_ONCE,
		UserPermissionAgent.MODE_GRANT_SESSION,
		UserPermissionAgent.MODE_DENY_SESSION,
		UserPermissionAgent.MODE_GRANT_ALWAYS };

	private String id;
	private Hashtable permissions;

ProtectionDomain(String id, Hashtable permissions) {
	this.id = id;
	this.permissions = permissions;
}

/**
 * Returns an enumeration of the Strings representing the permission names in this protection domain.
 */
public Enumeration getPermissionNames() {
	return permissions.keys();
}

/**
 * Returns the id of this domain as defined in the security policy file.
 */
public String getId() {
	return id;
}

/**
 * Returns the highest allowed permission level for the given permission name.
 */
public int getHighestPermissionLevel(String permissionName) {
	int[] p = (int[])permissions.get(permissionName);
	if (p==null) return PERMISSION_LEVEL_DENY;
	return p[0];
}

/**
 * Returns the default permission level for the given permission name.
 */
public int getDefaultPermissionLevel(String permissionName) {
	int[] p = (int[])permissions.get(permissionName);
	if (p==null) return PERMISSION_LEVEL_DENY;
	if (p[0]==PERMISSION_LEVEL_DENY || p[0]==PERMISSION_LEVEL_ALLOW) return p[0];
	return p[1];
}

/**
 * Returns the allowed user permission modes for the given permission.  The user permission modes are
 * the set of responses the user can select for a permission request.  These are based on the higest
 * permission level for a given permission.
 */
public int[] getAllowedUserPermissionModes(String permissionName) {
	int highestPermission = getHighestPermissionLevel(permissionName);

	if (highestPermission==ProtectionDomain.PERMISSION_LEVEL_USER_ONESHOT)
		return USER_PERMISSION_MODES_FOR_ONESHOT;

	if (highestPermission==ProtectionDomain.PERMISSION_LEVEL_USER_SESSION)
		return USER_PERMISSION_MODES_FOR_SESSION;

	if (highestPermission==ProtectionDomain.PERMISSION_LEVEL_USER_BLANKET)
		return USER_PERMISSION_MODES_FOR_BLANKET;

	return null;
}

}
