package com.ibm.oti.security.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

public class SecurityPolicy {

	private static SecurityPolicy instance = null;

	private ProtectionDomain[] protectionDomains;

/**
 * Returns the security policy for this VM.  By default, the security policy is read from the filesystem in lib\security.policy relative
 * to the value of the java.home system property.
 *
 * If the com.ibm.securitypolicy.path system property is set then the policy is read from the filesystem at the given path relative
 * to the value of the java.home system property.
 *
 * If the com.ibm.securitypolicy.resourcename property is set then the policy is read from the resource with the given name in the JXE
 * or classpath entry containing this class.
 */
public static SecurityPolicy getInstance() {
	if (instance == null) {
		InputStream is = null;

		String fileProperty;
		if ((fileProperty=System.getProperty("com.ibm.securitypolicy.path")) != null) {
			// load the security policy from an absolute path
			is = getStreamForPath(fileProperty, false);
		}

		String resourceProperty = null;
		if (is==null && (resourceProperty=System.getProperty("com.ibm.securitypolicy.resourcename")) != null) {
			// load the security policy from a resource
			is = getStreamForResourceName(resourceProperty);
		}

		if (fileProperty==null && resourceProperty==null) {
			// load the security policy from the default location if no system property was set
			is = getStreamForPath("lib/security.policy", true);
		}

		// parse the policy file
		if (is!=null) {
			try {
				byte[] policyBytes = new byte[is.available()];
				is.read(policyBytes);
				is.close();
				instance = new SecurityPolicy(policyBytes);
			} catch (IOException e) {
				// K01f5 = Error parsing security policy: {0}
				System.err.println(com.ibm.oti.util.Msg.getString("K01f5", e.getMessage()));
				e.printStackTrace();
			} catch (ParserException e) {
				// K01f5 = Error parsing security policy: {0}
				System.err.println(com.ibm.oti.util.Msg.getString("K01f5", e.getMessage()));
				e.printStackTrace();
			}
		}

		if (instance==null) {
			// K01f3 = Using default security policy instead
			System.err.println(com.ibm.oti.util.Msg.getString("K01f3"));
			instance = getDefaultPolicy();
		}
	}
	return instance;
}

static InputStream getStreamForPath(String path, boolean relative) {
	try {
        StringBuffer policyPath = new StringBuffer();
		if (relative) {
            String javaHome = System.getProperty("java.home");
            policyPath.append( javaHome );
            if( !endsWithSeperator(javaHome) ){
                policyPath.append("/");
            }
		}
        policyPath.append(path);

		return new com.ibm.oti.connection.file.FileInputStream(policyPath.toString());
	} catch (IOException e) {
		// K01f2 = Error loading security policy from file: {0}
		System.err.println(com.ibm.oti.util.Msg.getString("K01f2", path));
		e.printStackTrace();
   }
   return null;
}

private static boolean endsWithSeperator(String filePath) {
    return ((filePath.endsWith("/")) || (filePath.endsWith("\\")));
}

static InputStream getStreamForResourceName(String resName) {
	InputStream is = null;
	try {
		is = Class.forName("com.ibm.oti.security.midp.SecurityPolicy").getResourceAsStream("/"+resName);
	} catch (ClassNotFoundException e) {}
	if (is==null) {
		// K01f4 = Error loading security policy from resource: {0}
		System.err.println(com.ibm.oti.util.Msg.getString("K01f4", resName));
		return null;
	}
	return is;
}

/**
 * Constructs and returns the default security policy.  This policy defines one domain, "untrusted", that is
 * equivalent to the untrusted domain as defined in the "Security for MIDP Applications" section of the
 * MIDP 2.0 specification.
 */
static SecurityPolicy getDefaultPolicy() {
	Hashtable permissions = new Hashtable(7);
	int[] permissionAllow = new int[]{ProtectionDomain.PERMISSION_LEVEL_ALLOW, ProtectionDomain.PERMISSION_LEVEL_ALLOW};
	int[] permissionOneshot = new int[]{ProtectionDomain.PERMISSION_LEVEL_USER_ONESHOT,
			ProtectionDomain.PERMISSION_LEVEL_USER_ONESHOT};
	permissions.put("javax.microedition.rms", permissionAllow);
	permissions.put("javax.microedition.midlet", permissionAllow);
	permissions.put("javax.microedition.lcdui", permissionAllow);
	permissions.put("javax.microedition.lcdui.game", permissionAllow);
	permissions.put("javax.microedition.media", permissionAllow);
	permissions.put("javax.microedition.media.control", permissionAllow);
	permissions.put("javax.microedition.io.Connector.http", permissionOneshot);
	permissions.put("javax.microedition.io.Connector.https", permissionOneshot);

	ProtectionDomain[] domains = new ProtectionDomain[1];
	domains[0] = new ProtectionDomain("untrusted", permissions);

	return new SecurityPolicy(domains);

}

private SecurityPolicy(ProtectionDomain[] protectionDomains) {
	this.protectionDomains = protectionDomains;
}

/**
 * Creates a security policy from the byte encoding.  A ParserException is thrown if a syntax error occurs while
 * parsing.
 *
 * @throws	ParserException		if a syntax error occurs.
 */
public SecurityPolicy(byte[] policyBytes) throws ParserException {
	Parser parser = new Parser(policyBytes);
	parser.parse();
	this.protectionDomains = parser.getDomains();
}

/**
 * Returns the number of protection domains defined in the security policy.
 */
public int getProtectionDomainCount() {
	return protectionDomains.length;
}

/**
 * Returns the protection domain at index i.
 */
public ProtectionDomain getProtectionDomain(int index) {
	return protectionDomains[index];
}

/**
 * Returns the protection domain with the given id or null if none is found.
 */
public ProtectionDomain getProtectionDomain(String id) {
	for(int i = 0; i < protectionDomains.length; i++) {
		if (protectionDomains[i].getId().equals(id)) {
			return protectionDomains[i];
		}
	}
	return null;
}

private class Parser {

	private final byte[] DOMAIN_IDENTIFIER = new byte[]{'d','o','m','a','i','n',':'};
	private final byte[] ALIAS_IDENTIFIER = new byte[]{'a','l','i','a','s',':'};
	private final byte[] ALLOW_IDENTIFIER = new byte[]{'a','l','l','o','w'};
	private final byte[] BLANKET_IDENTIFIER = new byte[]{'b','l','a','n','k','e','t'};
	private final byte[] SESSION_IDENTIFIER = new byte[]{'s','e','s','s','i','o','n'};
	private final byte[] ONESHOT_IDENTIFIER = new byte[]{'o','n','e','s','h','o','t'};

	private byte[] data;
	private int pos;

	private int line = 1;
	private int posLineStart = 0;

	private String nameToken = null;
	private int highestPermission = -1;
	private int defaultPermission = -1;
	private Vector apiNames = new Vector();
	private Hashtable permissions = new Hashtable();
	private Hashtable domains = new Hashtable();
	private ProtectionDomain[] domainArray;
	private Hashtable aliases = new Hashtable();

	public Parser(byte[] data) throws ParserException {
		this.data = data;
		pos = 0;
		parse();

		domainArray = new ProtectionDomain[domains.size()];
		int i = 0;
		for(Enumeration e=domains.keys(); e.hasMoreElements(); ) {
			String domainId = (String)e.nextElement();
			Hashtable permissions = (Hashtable)domains.get(domainId);
			domainArray[i++] = new ProtectionDomain(domainId, permissions);
		}
	}

	public ProtectionDomain[] getDomains() {
		return domainArray;
	}

	private void parse() throws ParserException {
		while (pos<data.length) {
			parseDirective();
		}
	}

	private boolean parseDirective() throws ParserException {
		int posStart = pos;
		if (parseDomainDef() || parseAliasDef()) {
			while (parseNewline()) {}
			return true;
		} else {
			// K01df = unexpected token at line {0}, column {1}
			throw new ParserException(com.ibm.oti.util.Msg.getString("K01df", Integer.toString(line), Integer.toString(pos-posLineStart+1)));
		}
	}

	private boolean parseDomainDef() throws ParserException {
		int posStart = pos;

		if (!hasBytesAtPos(DOMAIN_IDENTIFIER)) return false;
		pos += DOMAIN_IDENTIFIER.length;

		if (!parseNameToken()) {
			pos = posStart;
			return false;
		}

		while (parseNewline()) {}

		permissions = new Hashtable();
		if (!parsePermission()) {
			pos = posStart;
			return false;
		}
		while(parsePermission()) {}

		domains.put(nameToken, permissions);

		return true;
	}

	private boolean parseAliasDef() throws ParserException {
		int posStart = pos;

		if (!hasBytesAtPos(ALIAS_IDENTIFIER)) return false;
		pos += ALIAS_IDENTIFIER.length;

		if (!parseNameToken()) {
			pos = posStart;
			return false;
		}

		if (!parseNewline()) {
			pos = posStart;
			return false;
		}
		while (parseNewline()) {}

		if (!parseApiNames(true)) {
			pos = posStart;
			return false;
		}

		for(int i=0; i<apiNames.size(); i++) {
			// see if this name is actually an alias
			String apiName = (String)apiNames.elementAt(i);
			Vector aliasDef = (Vector)aliases.get(apiName);
			if (aliasDef!=null) {
				apiNames.removeElementAt(i);
				for(Enumeration e=aliasDef.elements(); e.hasMoreElements(); )
					apiNames.addElement(e.nextElement());
			}
		}
		aliases.put(nameToken, apiNames);
		apiNames = new Vector();

		return true;
	}

	private boolean parseNameToken() throws ParserException {
		int posNewline = pos;
		while(!isNewline(posNewline)) {
			posNewline++;
			if (posNewline==data.length) return false;
		}
		int posLeft = pos;
		while(isWhitespace(posLeft)) {
			posLeft++;
			if (posLeft==posNewline) return false;
		}
		int posRight = posNewline;
		while(isWhitespace(posLeft)) {
			posRight--;
			if (posRight==pos) return false;
		}
		try {
			nameToken = new String(data, posLeft, posRight - posLeft, "ISO8859_1");
		} catch (UnsupportedEncodingException e) {
			// will not happen
			throw new RuntimeException(e.getMessage());
		}
		pos = posNewline;
		return true;
	}

	private boolean parsePermission() throws ParserException {
		int posStart = pos;

		if (!parsePermissionLevel()) return false;

		if (pos==data.length || data[pos]!=':') {
			pos = posStart;
			return false;
		}
		pos++;

		if (!parseApiNames(true)) {
			pos = posStart;
			return false;
		}

		while(parseNewline()) {}

		int[] permission = new int[]{highestPermission, defaultPermission};
		for(Enumeration e = apiNames.elements(); e.hasMoreElements(); ) {
			String apiName = (String)e.nextElement();
			// see if this name is actually an alias
			Vector aliasDef = (Vector)aliases.get(apiName);
			if (aliasDef!=null) {
				for(int i=0; i<aliasDef.size(); i++) {
					permissions.put(aliasDef.elementAt(i), permission);
				}
			} else {
				permissions.put(apiName, permission);
			}
		}

		return true;
	}

	private boolean parsePermissionLevel() throws ParserException {
		if (hasBytesAtPos(ALLOW_IDENTIFIER)) {
			pos += ALLOW_IDENTIFIER.length;
			highestPermission = ProtectionDomain.PERMISSION_LEVEL_ALLOW;
			defaultPermission = ProtectionDomain.PERMISSION_LEVEL_ALLOW;
			return true;
		}
		int posStart = pos;
		if (!parseUserPermissionLevel(false)) {
			return false;
		}

		if (pos<data.length && data[pos]=='(') {
			pos++;
			if (!parseUserPermissionLevel(true)) {
				pos = posStart;
				return false;
			}
			if (pos>=data.length || data[pos]!=')') {
				// K01de = unexpected end of file
				throw new ParserException(com.ibm.oti.util.Msg.getString("K01de", posStart));
			}
			pos++;
			return true;
		} else {
			defaultPermission = highestPermission;
		}
		return true;
	}

	private boolean parseApiNames(boolean clear) throws ParserException {
		int posStart = pos;
		if (clear) apiNames.removeAllElements();

		while(parseWhitespace()) {}

		int posEnd = pos;
		while(posEnd<data.length && !isWhitespace(posEnd) && !isNewline(posEnd) && data[posEnd]!=',') {
			posEnd++;
		}
		if (posEnd == pos) {
			pos = posStart;
			return false;
		}
		try {
			apiNames.addElement(new String(data, pos, posEnd-pos, "ISO8859_1"));
		} catch (UnsupportedEncodingException e) {
			// will not happen
			throw new RuntimeException(e.getMessage());
		}
		pos = posEnd;

		while(parseWhitespace()) {}

		if (pos!=data.length && data[pos]==',') {
			pos++;
			while(parseApiNames(false)) {}
		}
		return true;
	}

	private boolean parseUserPermissionLevel(boolean storeAsDefault) throws ParserException {
		int result = -1;
		if (hasBytesAtPos(BLANKET_IDENTIFIER)) {
			pos += BLANKET_IDENTIFIER.length;
			result = ProtectionDomain.PERMISSION_LEVEL_USER_BLANKET;
		} else
		if (hasBytesAtPos(SESSION_IDENTIFIER)) {
			pos += SESSION_IDENTIFIER.length;
			result = ProtectionDomain.PERMISSION_LEVEL_USER_SESSION;
		} else
		if (hasBytesAtPos(ONESHOT_IDENTIFIER)) {
			pos += ONESHOT_IDENTIFIER.length;
			result = ProtectionDomain.PERMISSION_LEVEL_USER_ONESHOT;
		}
		if (result==-1) return false;
		if (storeAsDefault) {
			// store in the defaultPermission variable
			defaultPermission = result;
		} else {
			// store in the highestPermission variable
			highestPermission = result;
		}
		return true;
	}

	private boolean isNewline(int pos) {
		return newlineLength(pos) > 0;
	}

	private boolean parseNewline() {
		int len = newlineLength(pos);
		if (len > 0) {
			pos += len;
			// increment line counters
			line++;
			posLineStart = pos;
			return true;
		}
		return false;
	}

	private int newlineLength(int pos) {
		if (pos>=data.length) return 0;
		if (data[pos]=='\r') {
			if (pos+1<data.length) {
				if (data[pos+1]=='\n') {
					return 2;
				}
			}
			return 1;
		}
		if (data[pos]=='\n') {
			return 1;
		}
		return 0;
	}

	private boolean isWhitespace(int pos) {
		return whitespaceLength(pos) > 0 || continuationLength(pos) > 0;
	}

	private boolean parseWhitespace() {
		int len = whitespaceLength(pos);
		if (len > 0) {
			pos += len;
			return true;
		} else {
			// try parsing a continuation (another kind of whitespace)
			len = continuationLength(pos);
			if (len > 0) {
				pos += len;
				// a continuation is newline SP, so increment line counters
				line++;
				posLineStart = pos-1;
				return true;
			}
		}

		return false;
	}

	private int whitespaceLength(int pos) {
		if (pos>=data.length) return 0;
		if (data[pos]==' ') {
			return 1;
		}
		return 0;
	}

	private int continuationLength(int pos) {
		// newline SP is a continuation
		int len = newlineLength(pos);
		if (len > 0) {
			if (pos+len<data.length) {
				if (data[pos+len]==' ') {
					return len+1;
				}
			}
		}
		return 0;
	}

	private boolean hasBytesAtPos(byte[] match) {
		if (data.length-pos < match.length) return false;
		for(int i=0; i<match.length; i++) {
			if (data[pos+i]!=match[i]) return false;
		}
		return true;
	}

}

/**
 * An exception that represents a syntax error in the security policy file.
 */
public class ParserException extends Exception {
	public ParserException(String message) {
		super(message);
	}
}

}
