package com.ibm.oti.util.math;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

import java.util.Random;

/**
 * BigInteger objects represent arbitrary precision decimal
 * integers. They contain values that cannot be changed.
 * Thus, most operations on the BigInteger objects yield new
 * instances of BigInteger.
 *
 * @author		OTI
 * @version		initial
 */
public class BigInteger

{
	private long[] data;

	/**
	 * Constant: -1 as represented by a BigInteger.
	 */
	private static final BigInteger NEGATIVE_ONE = new BigInteger(new long[] {-1});

	/**
	 * Constant: 0 as represented by a BigInteger.
	 */
	public final static BigInteger ZERO = BigInteger.valueOf(0L);

	/**
	 * Constant: 1 as represented by a BigInteger.
	 */
	public final static BigInteger ONE = BigInteger.valueOf(1L);

	// Natives used by the implementation.
	public static native long[] addImpl(long[] source1, long[] source2);
	public static native long[] subImpl(long[] source1, long[] source2);
	public static native long[] mulImpl(long[] source1, long[] source2);
	public static native long[] divImpl(long[] source1, long[] source2);
	public static native long[] remImpl(long[] source1, long[] source2);
	public static native long[] negImpl(long[] source);
	public static native long[] shlImpl(long[] source, int shiftVal);
	public static native int compImpl(long[] source1, long[] source2);

/**
 * Constructs a new instance of this class given an
 * array containing longs representing the bit pattern
 * for the answer.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data long[]
 *					the bits of the value of the new instance.
 */
private BigInteger(long[] data) {
	this.data = data;
}

/**
 * Constructs a new instance of this class of the
 * specified length, whose content is produced by
 * aquiring random bits from the specified random
 * number generator.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bitLength int
 *					the number of bits to have in the result.
 * @param		rnd Random
 *					the generator to produce the bits.
 */
public BigInteger (int bitLength, Random rnd) {
	if(bitLength < 0)
		throw new IllegalArgumentException();
	randomData(bitLength, rnd);
}

/**
 * Constructs a new instance of this class of the
 * specified length, whose content is produced by
 * aquiring random bits from the specified random
 * number generator. The result is guaranteed to be
 * prime up to the given degree of certainty.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bitLength int
 *					the number of bits to have in the result.
 * @param		certainty int
 *					the degree of certainty required that the
 *					result is prime.
 * @param		rnd Random
 *					the generator to produce the bits.
 */
public BigInteger (int bitLength, int certainty, Random rnd) {
	if (bitLength < 2)
		throw new ArithmeticException();
	while (true) {
		randomData(bitLength, rnd);
		data[0] |= 1L; // make the result odd, to cut half the failures.
		if(isProbablePrime(certainty, rnd))
			break;
	}
}

/**
 * Fills in the receiver by aquiring random the given
 * number of bits from the specified random number generator.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bitLength int
 *					the number of bits to have in the result.
 * @param		rnd Random
 *					the generator to produce the bits.
 */
private void randomData (int bitLength, Random rnd) {
	int longLength=(bitLength+64)/64;
	data=new long[longLength];
	int rem=bitLength%64;
	if (rem == 0) {
		if (bitLength == 0) return;
		longLength--;
		rem = 64;
	}
	for(int i=0;i<longLength;i++)
		while ((data[i] = rnd.nextLong()) == 0);
	int bits = getHighestSetBit() % 64 + 1;
	if (bits > rem)
		data[longLength-1] >>>= bits - rem;
	else if (bits < rem)
		data[longLength-1] <<= rem - bits;
	normalize();
}

/**
 * Ensures that the receiver is in canonical form.
 *
 * @author		OTI
 * @version		initial
 */
private BigInteger normalize() {
	int len=data.length;
	while(len >= 2 && ((data[len-1]==0L && data[len-2]>=0L)
					|| (data[len-1]==-1L && data[len-2]<0L)))
		len--;
	if(len<data.length) {
		long[] temp = data;
		data = new long[len];
		System.arraycopy(temp,0,data,0,len);
	}
	return this;
}

/**
 * Constructs a new instance of this class given an
 * array containing bytes representing the bit pattern
 * for the answer.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bytes byte[] the bits of the value of the new instance.
 */
public BigInteger (byte[] bytes) {
	int bytelen=bytes.length;
	if(bytelen==0)
		throw new NumberFormatException();
	long sign = bytes[0]<0 ? 0xFFL : 0;
	// bytes are in BigEndian order - parse in reverse
	int longLength = bytelen/8+1;
	data = new long[longLength];
	int longPtr=0;
	int bytePtr=bytelen-1;
	int modCtr = 0;
	for(;bytePtr >= 0;bytePtr--) {
		data[longPtr] |= ((long)bytes[bytePtr] & 0xFFL) << (modCtr*8);
		modCtr = (modCtr +1)%8;
		if(modCtr==0)
			longPtr++;
	}
	while(longPtr<longLength) {
		data[longPtr] |= sign << (modCtr*8);
		modCtr = (modCtr +1)%8;
		if(modCtr==0)
			longPtr++;
	}
	normalize();
}

/**
 * Constructs a new instance of this class given an
 * array containing bytes representing the bit pattern
 * for the answer, and a sign flag.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		sign int the sign of the result.
 * @param		bytes byte[] the bits of the value of the new instance.
 */
public BigInteger (int sign, byte[] bytes)
{
	fromBytes(sign, bytes);
}

public BigInteger (String val, int radix) {
	int max=val.length() - 1;
	if(radix < Character.MIN_RADIX || radix > Character.MAX_RADIX || max<0)
		throw new NumberFormatException();
	data=ZERO.data;
	long[] bigRadix=new long[1]; bigRadix[0]=radix;
	long[] digitAr=new long[1];
	boolean negative=val.charAt(0)=='-';
	for(int at=negative?1:0; at<=max; at++) {
		int digit=Character.digit(val.charAt(at),radix);
		if(digit==-1)
			throw new NumberFormatException(val);
		data = com.ibm.oti.util.math.BigInteger.mulImpl(data,bigRadix);
		if(digit!=0) {
			digitAr[0]=digit;
			data = com.ibm.oti.util.math.BigInteger.addImpl(data,digitAr);
		}
	}
	normalize();
	if(negative)
		data=com.ibm.oti.util.math.BigInteger.negImpl(data);
}

/**
 * Fills in the value of the receiver given an array
 * containing bytes representing the bit pattern
 * and a sign flag.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		sign int the sign of the result.
 * @param		bytes byte[] the bits of the value of the new instance.
 */
private void fromBytes (int sign, byte[] bytes) {
	if(sign==0) {
		for(int i=bytes.length; --i>=0;)
			if(bytes[i]!=0)
				throw new NumberFormatException();
		data=ZERO.data;
	}
	else {
		if(sign<-1 || sign>1)
			throw new NumberFormatException();
		// bytes are in BigEndian order - parse in reverse
		int longLength;
		if (bytes.length == 0) longLength = 1;
		else longLength = (bytes.length+(bytes[0]<0?1:0))/8+1;
		data = new long[longLength];
		int longPtr=0;
		int modCtr = 0;
		for(int bytePtr=bytes.length; --bytePtr >= 0;) {
			data[longPtr] |= ((long)bytes[bytePtr] & 0xFFL) << (modCtr*8);
			modCtr = (modCtr +1)%8;
			if(modCtr==0)
				longPtr++;
		}
		normalize();
		if(sign<0)
			data = negImpl(data);
	}
}

/**
 * Answers an array of bytes containing the value of the
 * receiver in the same format used by the matching
 * constructor.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		byte[]
 *					the bits of the value of the receiver.
 */
public byte[] toByteArray () {
	int len = data.length;
	int bytePtr = len*8;
	byte[] bytes = new byte[bytePtr--];
	for(int i=0;i<len;i++) {
		long work=data[i];
		for(int j=0;j<8;j++) {
			bytes[bytePtr--] = (byte)work;
			work >>>= 8;
		}
	}
	// normalize bytes (in BigEndian order)
	int start=0;
	len *= 8;
	while(start <= len-2 && ((bytes[start]==0 && bytes[start+1]>=0)
					|| (bytes[start]==-1 && bytes[start+1]<0)))
		start++;
	if(start>0) {
		byte[] temp = bytes;
		bytes = new byte[len-start];
		System.arraycopy(temp,start,bytes,0,len-start);
	}
	return bytes;
}

/**
 * Answers true if the receiver is probably prime to the
 * given degree of certainty.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		certainty int
 *					the degree of certainty required.
 * @return		boolean
 *					true if the receiver is prime and
 *					false otherwise.
 */
public boolean isProbablePrime(int certainty) {
	return isProbablePrime (certainty, new Random());
}

/**
 * Answers true if the receiver is probably prime to the
 * given degree of certainty, using the random number
 * generator to provide seed value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		certainty int
 *					the degree of certainty required.
 * @param		rnd Random
 *					the random number to use for checking.
 * @return		boolean
 *					true if the receiver is prime and
 *					false otherwise.
 */
private boolean isProbablePrime (int certainty, Random rnd) {

	// primality test algorithm from Appendix 2 of
	//     http://www.nist.gov/itl/div897/pubs/fip186.htm
	//
	// given n, this algorithm produces a false prime with
	// probability < 1/4^n

	if (certainty < 1)
		return true;

	BigInteger w = this.abs();

	if (w.data.length == 1) {
		// the number one is not considered prime
		if (w.data[0] == 1) return false;
		// the number two is considered prime
		if (w.data[0] == 2) return true;
	}
	// do fast check for divisibility by 2
	if ((w.data[0] & 1L)==0)
		return false;

	int bitSize = w.getHighestSetBit() + 1;
	if(bitSize<=3) // constructor call below fails if bitSize<=2
		// 3-bit prime numbers (0-7) are 2,3,5,7 - already verified w is odd
		return true;
	BigInteger wMinus1 = w.subtract(ONE);

	int a = wMinus1.getLowestSetBit();
	if(a==-1)
		a=0;
	BigInteger m = wMinus1.shiftRight(a);
// Step 3
	int n=certainty / 2 + 1;
	for(int i=0; i<n; i++) {
		BigInteger b;
		do {
			// generate random integer b in range 1 < b < w
			b = new BigInteger(bitSize-1, rnd);
		} while( /*this.compareTo(b) < 0 ||  GUARANTEED BY BITSIZE */ b.equals(ZERO) );
// Step 4
		int j=0;
		BigInteger z = b.modPow(m, w);
// Step 5
		while(true) {
			boolean zEqONE=z.equals(ONE);
			if((j==0 && zEqONE) || z.equals(wMinus1)) {
				//goto step9;
				break;
			}
// Step 6
			if(j>0 && zEqONE)
				//goto step8;
				return false;
// Step 7
			j++;
			if( j>=a )
				//goto step8;
				return false;
			//z = z.modPow(TWO, w);
			z = z.multiply(z).mod(w);
		}
// Step 8
		// inlined above
// Step 9
	} // end while

	// w is probably prime
	return true;
}

/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison. In this case the argument must also
 * be a BigInteger which represents the same number
 *
 * @author		OTI
 * @version		initial
 *
 * @param		o Object
 *					the object to compare with this object.
 * @return		boolean
 *					<code>true</code>
 *						if the object is the same as this object
 *					<code>false</code>
 *						if it is different from this object.
 * @see			#hashCode
 */
public boolean equals (Object o) {
	if(!(o instanceof BigInteger))
		return false;
	BigInteger bi = (BigInteger)o;
	if(data.length != bi.data.length)
		return false;
	for(int a=0;a<data.length;a++)
		if(data[a]!=bi.data[a])
			return false;
	return true;
}

/**
 * Answers an integer indicating the relative positions
 * of the receiver and the argument in the natural order
 * of elements of the receiver's class.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int
 *					which should be <0 if the receiver
 *					should sort before the argument, 0
 *					if the receiver should sort in the
 *					same position as the argument, and
 *					>0 if the receiver should sort after
 *					the argument.
 * @param		val BigInteger
 *					an object to compare the receiver to
 * @exception	ClassCastException
 *					if the argument can not be converted
 *					into something comparable with the
 *					receiver.
 */
public int compareTo(BigInteger val){
	int len1 = data.length;
	int len2 = val.data.length;
	boolean neg1 = data[len1 - 1] < 0L;
	boolean neg2 = val.data[len2 - 1] < 0L;
	if (neg1 != neg2)
		return neg2 ? 1 : -1;
	else if (len1 != len2) {
		if (!neg1) // positive/zero case
			return len1 > len2 ? 1 : -1;
		else // negative case
			return len1 > len2 ? -1 : 1;
	}
	if (data[len1 - 1] != val.data[len1 - 1])
		return data[len1 - 1] > val.data[len1 - 1] ? 1 : -1;
	for (int i = len1 - 2; i >= 0; i--)
		if (data[i] != val.data[i]) {
			// unsigned comparison
			boolean s1 = data[i] < 0L;
			boolean s2 = val.data[i] < 0L;
			if (s1 == s2)
				return data[i] > val.data[i] ? 1 : -1;
			else return s1 ? 1 : -1;
		}
	return 0;
}

/**
 * Answers the int value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int		the value of the receiver.
 */
public int intValue() {
	return (int)data[0];
}

/**
 * Answers the long value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		long		the value of the receiver.
 */
public long longValue() {
	return data[0];
}

/**
 * Answers a BigInteger with the same value as val
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger		(BigInteger) val
 */
public static BigInteger valueOf(long val) {
	long[] temp = new long[1];
	temp[0] = val;
	return new BigInteger(temp);
}

/**
 * Answers the sum of the receiver and a BigInteger
 *
 * @author		OTI
 * @version		initial
 * @param		val		a BigInteger to add
 *
 * @return		BigInteger		this + val
 */
public BigInteger add(BigInteger val) {
	return new BigInteger(addImpl(this.data,val.data));
}

/**
 * Answers the negative of the receiver
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger		(-1) * this
 */
public BigInteger negate() {
	return new BigInteger(negImpl(this.data));
}

/**
 * Answers the sign of the receiver
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger		-1, 0, or 1 if the receiver is negative, zero, or positive
 */
public int signum() {
	if(data[data.length-1]<0)
		return -1;
	// normalized, so only need to check upper word - zero in hiword means the
	// number is positive, except in the case where the number is zero
	if(data[data.length-1]==0 && data.length==1)
		return 0;
	return 1;
}

/**
 * Answers the absolute value of the receiver
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger		absolute value of the receiver
 */
public BigInteger abs() {
	if(data[data.length-1]<0)
		return this.negate();
	return this;
}

/**
 * Answers the receiver to the power of exponent.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	ArithmeticException
 *					if the exponent is negative.
 *
 * @return		BigInteger		this ^ exponent
 */
public BigInteger pow(int exponent) {
	if(exponent < 0)
		throw new ArithmeticException();
	long[] powersOf2 = data;
	long[] answer = ONE.data;
	while(exponent != 0) {
		if((exponent & 1) == 1)
			answer = mulImpl(answer,powersOf2);
		powersOf2 = mulImpl(powersOf2,powersOf2);
		exponent >>>= 1;
	}
	return new BigInteger(answer);
}

/**
 * Answers the receiver to the power of exponent modulo a BigInteger
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	ArithmeticException
 *					modulo is <= 0
 *
 * @return		BigInteger		this ^ exponent (mod modulo)
 */
public BigInteger modPow(BigInteger exponent,BigInteger modulo) {
	if(modulo.signum() <= 0)
		throw new ArithmeticException();
	long[] powersOf2 = data;
	if(exponent.signum() < 0) {
		exponent = exponent.negate();
		powersOf2 = (this.modInverse(modulo)).data;
	}
	long[] exp = exponent.data;
	long[] mod = modulo.data;
	long[] answer = ONE.data;
	int explen = exp.length;
	if (explen == 1 && exp[0] == 0)
		answer = remImpl(answer, mod);
	else {
		for(int i=0;i<explen;i++) {
			long work = exp[i];
			for(int j=0;j<64;j++) {
				if((work & 1L) == 1L)
					answer = remImpl(mulImpl(answer,powersOf2),mod);
				powersOf2 = remImpl(mulImpl(powersOf2,powersOf2),mod);
				work >>>= 1;
			}
		}
	}
	if (answer[answer.length-1] < 0)
		return new BigInteger(addImpl(answer, mod));
	return new BigInteger(answer);
}

/**
 * Answers the greatest common divisor of abs(this) and abs(val), zero if this==val==0
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger		gcd(abs(this), abs(val))
 */
public BigInteger gcd(BigInteger val) {
	// write better implementation using binary algorithm
	long[] v,u,r;
	switch(this.signum()) {
		case 0: return val.abs();
		case -1: v = negImpl(this.data); break;
		default: v = this.data; break;
	}
	switch(val.signum()) {
		case 0: return this.abs();
		case -1: u = negImpl(val.data); break;
		default: u = val.data; break;
	}
	while(true) {
		// check if v==0
		if(v.length==1 && v[0]==0)
			return new BigInteger(u);
		// since u is postive, r is positive
		r = remImpl(u,v);
		u = v;
		v = r;
	}
}

/**
 * Answers the inverse of the receiver modulo a BigInteger,
 * if it exists.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		modulo BigInteger
 *					a BigInteger to divide
 * @return		BigInteger		this^(-1) (mod modulo)
 *
 * @exception	ArithmeticException
 *					if modulo is <= 0, or gcd(this,modulo) != 1
 */
public BigInteger modInverse(BigInteger modulo) {
	// modular inverse algorithm based on the Extended Euclidean Algorithm
	if (modulo.compareTo(ZERO) > 0) {
		BigInteger s1 = ONE;
		BigInteger s = ZERO;
		BigInteger r1 = this;
		BigInteger r = modulo;
		BigInteger q;

		do {
			q = r1.divide(r);
			s = s1.subtract(q.multiply(s1 = s));
			r = r1.subtract(q.multiply(r1 = r));
		} while (!r.equals(ZERO));

		if (r1.equals(NEGATIVE_ONE)) {
			s1 = s1.negate();
			r1 = ONE;
		}

		if (r1.equals(ONE)) {
			if (s1.compareTo(ZERO) < 0)
				return s1.add(modulo);
			return s1;
		}
	}
	throw new ArithmeticException();
}

/**
 * Answers the index of the lowest set bit in the receiver,
 * or -1 if no bits are set.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger
 *					the bit index of the least significant set
 *					bit in the receiver.
 */
public int getLowestSetBit() {
	for(int longNum=0;longNum<data.length;longNum++)
		if(data[longNum]!=0) {
			long word=data[longNum];
			for(int byteNum=0;/*byteNum<8*/;byteNum++, word>>>=8)
				if((word&255) != 0) {
					int val=(int)(word&255);
					for(int answer = longNum*64 + byteNum*8;;answer++, val>>>=1)
						if((val&1)!=0)
							return answer;
				}
		}
	return -1;
}

/**
 * Answers the index of the highest set bit in the receiver,
 * or -1 if no bits are set.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger
 *					the bit index of the most significant set
 *					bit in the receiver.
 */
private int getHighestSetBit() {
	for(int longNum=data.length-1;longNum>=0;longNum--)
		if(data[longNum]!=0) {
			long word=data[longNum];
			for(int byteNum=7;;byteNum--, word<<=8)
				if((word & 0xFF00000000000000L) != 0) {
					int val=(int)(word>>>56);
					for(int answer = longNum*64 + byteNum*8 + 7;;answer--, val<<=1)
						if((val&128)!=0)
							return answer;
				}
		}
	return -1;
}

/**
 * Answers the index of the highest unset bit in the receiver,
 * or -1 if no bits are unset.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger
 *					the bit index of the most significant unset
 *					bit in the receiver.
 */
private int getHighestUnSetBit() {
	for(int longNum=data.length-1;longNum>=0;longNum--)
		if(data[longNum]!=-1) {
			long word=data[longNum];
			for(int byteNum=7;;byteNum--, word<<=8)
				if((word & 0xFF00000000000000L) != 0xFF00000000000000L) {
					int val=(int)(word>>>56);
					for(int answer = longNum*64 + byteNum*8 + 7;;answer--, val<<=1)
						if((val&128)!=128)
							return answer;
				}
		}
	return -1;
}

/**
 * Answers a BigInteger with the value of the reciever
 * divided by 2^shiftval.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		shiftval int
 *					the amount to shift the receiver.
 * @return		BigInteger
 *					this >> val
 */
public BigInteger shiftRight(int shiftval) {
	return new BigInteger(shlImpl(this.data,-shiftval));
}

/**
 * Answers a BigInteger with the value of the reciever
 * multiplied by 2^shiftval.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		shiftval int
 *					the amount to shift the receiver.
 * @return		BigInteger
 *					this << val
 */
public BigInteger shiftLeft(int shiftval) {
	return new BigInteger(shlImpl(this.data,shiftval));
}

/**
 * Answers the difference of the receiver and a BigInteger.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to subtract
 * @return		BigInteger
 *					this - val
 */
public BigInteger subtract(BigInteger val) {
	return new BigInteger(subImpl(this.data,val.data));
}

/**
 * Answers the product of the receiver and a BigInteger.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to multiply
 * @return		BigInteger
 *					this * val
 */
public BigInteger multiply(BigInteger val) {
	return new BigInteger(mulImpl(this.data,val.data));
}

/**
 * Multiply the receiver by decimal value of 10.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		BigInteger
 *					BigInteger representation of this*10.
 */
final BigInteger multiplyByTen () {
	return shiftLeft(1).add(shiftLeft(3));
}

/**
 * Answers the quotient of the receiver and a BigInteger.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to divide
 * @return		BigInteger
 *					this / val
 *
 * @exception	ArithmeticException
 *					if val is zero.
 */
public BigInteger divide(BigInteger val) {
	if(val.data.length==1 && val.data[0]==0)
		throw new ArithmeticException();
	return new BigInteger(divImpl(this.data,val.data));
}

/**
 * Answers the remainder of the receiver divided by a BigInteger
 *
 * @author		OTI
 * @version		initial
 * @param		val		a BigInteger to divide
 *
 * @exception	ArithmeticException
 *					if val is zero
 *
 * @return		BigInteger		this % val
 */
public BigInteger remainder(BigInteger val) {
	if(val.data.length==1 && val.data[0]==0)
		throw new ArithmeticException();
	return new BigInteger(remImpl(this.data,val.data));
}

/**
 * Answers the remainder of the receiver modulo a
 * BigInteger (a positive value).
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val
 *					the value to divide
 * @return		BigInteger
 *					this (mod val)
 *
 * @exception	ArithmeticException
 *					if val is zero
 */
public BigInteger mod(BigInteger val) {
	if(val.signum() <= 0)
		throw new ArithmeticException();
	BigInteger temp = new BigInteger(remImpl(this.data,val.data));
	if(temp.signum()==-1)
		temp.data = addImpl(temp.data,val.data);
	return temp;
}

/**
 * Answers the quotient and remainder of the receiver
 * divided by a BigInteger.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to divide.
 * @return		BigInteger[2]
 *					{this / val, this % val )}
 *
 * @exception	ArithmeticException
 *					if val is zero.
 */
public BigInteger[] divideAndRemainder (BigInteger val) {
	if(val.data.length==1 && val.data[0]==0)
		throw new ArithmeticException();
	BigInteger[] ar = new BigInteger[2];
	ar[0] = new BigInteger(divImpl(this.data,val.data));
	ar[1] = new BigInteger(remImpl(this.data,val.data));
	return ar;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver. In this case, a string of
 * decimal digits.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					a printable representation for the receiver.
 */
public String toString () {
	return this.toString(10);
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver as a sequence of digits in
 * the specified radix.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					a printable representation for the receiver.
 */
public String toString (int radix) {
	if(radix < Character.MIN_RADIX || radix > Character.MAX_RADIX)
		radix = 10;

	int i, count;
	char [] buffer;
	boolean negative;
	BigInteger temp,temp2;
	BigInteger bigRadix = BigInteger.valueOf(radix);

	if(this.equals(ZERO)) return "0";
	if(negative = (this.signum() < 0)) {
		temp = this.negate();
		count = 2;
	} else {
		temp = this;
		count = 1;
	}
	temp2 = temp;
	while (!(temp2 = temp2.divide(bigRadix)).equals(ZERO)) count++;

	buffer = new char [i = count];
	int digit;
	do {
		BigInteger[] result = temp.divideAndRemainder(bigRadix);
		temp = result[0];
		digit = result[1].intValue();
		buffer [--i] = (char)(digit < 10 ? digit + '0' : digit + 'a' - 10);
	} while (!temp.equals(ZERO));
	if (negative) buffer [0] = '-';
	return new String (buffer, 0, count);
}

/**
 * Answers the most positive of either the receiver
 * or the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to compare.
 * @return		BigInteger
 *					the larger value.
 */
public BigInteger max (BigInteger val) {
	return compImpl(this.data,val.data) >= 0 ? this : val;
}

/**
 * Answers the most negative of either the receiver
 * or the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val BigInteger
 *					the value to compare.
 * @return		BigInteger
 *					the smaller value.
 */
public BigInteger min (BigInteger val) {
	return compImpl(this.data,val.data) <= 0 ? this : val;
}

/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>.equals</code> must answer the same value for this
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int the receiver's hash.
 *
 * @see			#equals
 */
public int hashCode () {
	long hash=0;
	for(int i=data.length-1;i>=0;i--)
		hash = hash*31 + data[i];
	return (int)(hash ^ hash>>>32);
}

/**
 * Answers the length in bits of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int
 *					the receiver's bit length.
 */
public int bitLength() {
	int len;
	if(this.signum()>=0)
		len=this.getHighestSetBit();
	else
		len=this.getHighestUnSetBit();
	// if no bits are set, above returns -1, so this method returns bitlength of zero
	return len+1;
}

/**
 * Sets the specified bit in the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		n int
 *					the bit to set.
 */
public BigInteger setBit (int n) {
	if(n<0)
		throw new ArithmeticException();
	int wordnum=n/64, bit = n%64;
	int newlen = data.length;
	int sign=this.signum();
	if(wordnum >= data.length) {
		// setting a bit in a negative number beyond the end has no effect
		if(sign==-1)
			return this;
		else
			newlen = wordnum + (bit==63 ? 2 : 1);
	} else {
		// if the bit is already set, do nothing
		if(((data[wordnum] >> bit) & 1L) == 1)
			return this;
		// see if new sign word will be necessary
		if(bit==63 && wordnum == data.length-1 && sign>=0)
			newlen++;
	}
	long[] newdata=new long[newlen];
	System.arraycopy(data,0,newdata,0,data.length);
	// don't need to sign extend because if it was negative, we wouldn't have had to grow it
	newdata[wordnum] |= (1L << bit);
	// normalize in case setting the bit made upper word redundant
	return new BigInteger(newdata).normalize();
}

}
