
package com.ibm.oti.vm;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.util.Hashtable;
import java.io.IOException;
import java.io.InputStream;

/**
 * This class contains helper methods for loading resource
 * bundles and formatting external message strings.
 *
 * @author		OTI
 * @version		initial
 */

public final class MsgHelp {

/**
 * Generates a formatted text string given a source string
 * containing "argument markers" of the form "{argNum}"
 * where each argNum must be in the range 0..9. The result
 * is generated by inserting the toString of each argument
 * into the position indicated in the string.
 * <p>
 * To insert the "{" character into the output, use a single
 * backslash character to escape it (i.e. "\{"). The "}"
 * character does not need to be escaped.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		format String
 *					the format to use when printing.
 * @param		args Object[]
 *					the arguments to use.
 * @return		String
 *					the formatted message.
 */
public static String format (String format, Object[] args) {
	StringBuffer answer = new StringBuffer(format.length() + (args.length * 20));
	String[] argStrings = new String[args.length];
	for (int i = 0; i < args.length; ++i) {
		if (args[i] == null)
			argStrings[i] = "<null>";
		else
			argStrings[i] = args[i].toString();
	}
	int lastI = 0;
	for (int i = format.indexOf('{', 0); i >= 0; i = format.indexOf('{', lastI)) {
		if (i != 0 && format.charAt(i-1) == '\\') {
			// It's escaped, just print and loop.
			if (i != 1)
				answer.append(format.substring(lastI,i-1));
			answer.append('{');
			lastI = i+1;
		} else {
			// It's a format character.
			if (i > format.length()-3) {
				// Bad format, just print and loop.
				answer.append(format.substring(lastI, format.length()));
				lastI = format.length();
			} else {
				int argnum = (byte) Character.digit(format.charAt(i+1), 10);
				if (argnum < 0 || format.charAt(i+2) != '}') {
					// Bad format, just print and loop.
					answer.append(format.substring(lastI, i+1));
					lastI = i+1;
				} else {
					// Got a good one!
					answer.append(format.substring(lastI, i));
					if (argnum >= argStrings.length)
						answer.append("<missing argument>");
					else
						answer.append(argStrings[argnum]);
					lastI = i + 3;
				}
			}
		}
	}
	if (lastI < format.length())
		answer.append(format.substring(lastI, format.length()));
	return answer.toString();
}

// Loads properties from the specified resource. The properties are of
// the form <code>key=value</code>, one property per line.
public static Hashtable loadMessages(String resourceName) throws IOException {
	InputStream resourceStream;
	String resName;

	String language = System.getProperty("user.language");
	if (language == null) language = "en";
	String region = System.getProperty("user.region");
	if (region == null) region = "US";
	String variant = System.getProperty("user.variant");

	if (variant != null) {
		resName = resourceName + "_" + language + "_" + region + "_" + variant + ".properties";
		resourceStream = VM.getResourceAsStream(resName, true);
		if (resourceStream != null) return load(resourceStream);
	}

	resName = resourceName + "_" + language + "_" + region + ".properties";
	resourceStream = VM.getResourceAsStream(resName, true);
	if (resourceStream != null) return load(resourceStream);

	resName = resourceName + "_" + language + ".properties";
	resourceStream = VM.getResourceAsStream(resName, true);
	if (resourceStream != null) return load(resourceStream);

	resourceStream = VM.getResourceAsStream(resourceName + ".properties", true);
	return resourceStream != null ? load(resourceStream) : null;
}

public static Hashtable load(InputStream resourceStream) throws IOException {
	final int
		NONE = 0, SLASH = 1, UNICODE = 2,
		CONTINUE = 3, DONE = 4, IGNORE = 5;

	Hashtable messages = new Hashtable();
	int mode = NONE, unicode = 0, count = 0;
	char nextChar, buf[] = new char[40];
	int offset = 0, keyLength = -1;
	boolean firstChar = true;
	byte[] inbuf = new byte[256];
	int inbufCount = 0, inbufPos = 0;

	while (true) {
		if (inbufPos == inbufCount) {
			if ((inbufCount = resourceStream.read(inbuf)) == -1) break;
			inbufPos = 0;
		}
		nextChar = (char)(inbuf[inbufPos++] & 0xff);

		if (offset == buf.length) {
			char[] newBuf = new char[buf.length * 2];
			System.arraycopy(buf, 0, newBuf, 0, offset);
			buf = newBuf;
		}
		if (mode == UNICODE) {
			int digit = Character.digit(nextChar, 16);
			if (digit >= 0) {
				unicode = (unicode << 4) + digit;
				if (++count < 4) continue;
			}
			mode = NONE;
			buf[offset++] = (char)unicode;
			if (nextChar != '\n') continue;
		}
		if (mode == SLASH) {
			mode = NONE;
			switch (nextChar) {
				case '\r':
					mode = CONTINUE; // Look for a following \n
					continue;
				case '\n':
					mode = IGNORE; // Ignore whitespace on the next line
					continue;
				case 'b':
					nextChar = '\b';
					break;
				case 'f':
					nextChar = '\f';
					break;
				case 'n':
					nextChar = '\n';
					break;
				case 'r':
					nextChar = '\r';
					break;
				case 't':
					nextChar = '\t';
					break;
				case 'u':
					mode = UNICODE;
					unicode = count = 0;
					continue;
			}
		} else {
			switch (nextChar) {
				case '#':
				case '!':
					if (firstChar) {
						while (true) {
							if (inbufPos == inbufCount) {
								if ((inbufCount = resourceStream.read(inbuf)) == -1) {
									inbufPos = -1;
									break;
								}
								inbufPos = 0;
							}
							nextChar = (char)inbuf[inbufPos++]; // & 0xff not required
							if (nextChar == '\r' || nextChar == '\n') break;
						}
						continue;
					}
					break;
				case '\n':
					if (mode == CONTINUE) { // Part of a \r\n sequence
						mode = IGNORE; // Ignore whitespace on the next line
						continue;
					}
					// fall into the next case
				case '\r':
					mode = NONE;
					firstChar = true;
					if (keyLength >= 0) {
						String temp = new String(buf, 0, offset);
						messages.put(temp.substring(0,keyLength), temp.substring(keyLength));
					}
					keyLength = -1;
					offset = 0;
					continue;
				case '\\':
					mode = SLASH;
					continue;
				case ':':
				case '=':
					if (keyLength == -1) { // if parsing the key
						keyLength = offset;
						continue;
					}
					break;
			}
			if (isWhitespace(nextChar)) {
				if (mode == CONTINUE) mode = IGNORE;
				// if key length == 0 or value length == 0
				if (offset == 0 || offset == keyLength || mode == IGNORE) continue;
				if (keyLength == -1) { // if parsing the key
					mode = DONE;
					continue;
				}
			}
			if (mode == IGNORE || mode == CONTINUE) mode = NONE;
		}
		firstChar = false;
		if (mode == DONE) {
			keyLength = offset;
			mode = NONE;
		}
		buf[offset++] = nextChar;
	}
	if (keyLength >= 0) {
		String temp = new String(buf, 0, offset);
		messages.put(temp.substring(0,keyLength), temp.substring(keyLength));
	}
	resourceStream.close();
	return messages;
}

private static boolean isWhitespace(char c) {
	// Optimized case for ASCII
	if ((c >= 0x1c && c <= 0x20) || (c >= 0x9 && c <= 0xd)) return true;
	if (c < 0x2000) return false;
	return c <= 0x200b || c == 0x2028 || c == 0x2029 || c == 0x3000;
}

}
