package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * ByteArrayOutputStream is a class whose underlying stream is represented by
 * a byte array.  As bytes are written to this stream, the local byte array may
 * be expanded to hold more bytes.
 *
 * @author		OTI
 * @version		initial
 *
 * @see 		ByteArrayInputStream
 */
public class ByteArrayOutputStream extends OutputStream {
	/**
	 * The byte array containing the bytes written.
	 */
	protected byte[] buf;
	/**
	 * The number of bytes written.
	 */
	protected int count;

/**
 * Constructs a new ByteArrayOutputStream with a default size of 32 bytes.  If
 * more than 32 bytes are written to this instance, the underlying byte array
 * will expand to accomodate.
 *
 */
public ByteArrayOutputStream() {
	super();
	buf = new byte[32];
}

/**
 * Constructs a new ByteArrayOutputStream with a default size of
 * <code>size</code> bytes.  If more than <code>size</code> bytes are
 * written to this instance, the underlying byte array will expand to
 * accomodate.
 *
 * @param		size		an non-negative integer representing the initial size for the
 *							underlying byte array.
 */
public ByteArrayOutputStream(int size) {
	super();
	if (size >= 0)
		buf = new byte[size];
	// K005e = size must be >= 0
	else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K005e"));
}

/**
 * Close this ByteArrayOutputStream.  This implementation releases System resources used for this stream.
 *
 * @throws 		IOException		If an error occurs attempting to close this OutputStream.
 */
public void close() throws IOException {
	/**
	 * Although the spec claims "A closed stream cannot perform output operations and cannot be reopened.",
	 * this implementation must do nothing.
	 */
	super.close();
}

private void expand(int i) {
	byte[] newbuf = new byte[(count + i) * 2];
	System.arraycopy(buf, 0, newbuf, 0, count);
	buf = newbuf;
}

/**
 * Reset this ByteArrayOutputStream to the beginning of the underlying byte array.
 * All subsequent writes will overwrite any bytes previously stored in this stream.
 *
 */
public synchronized void reset() {
	count = 0;
}

/**
 * Answers the total number of bytes written to this stream thus far.
 *
 * @return		the number of bytes written to this Stream.
 */
public int size() {
	return count;
}

/**
 * Answer the contents of this ByteArrayOutputStream as a byte array.  Any changes made
 * to the receiver after returning will not be reflected in the byte array returned to the caller.
 *
 * @return 		this streams current contents as a byte array.
 */
public synchronized byte[] toByteArray() {
	byte[] newArray = new byte[count];
	System.arraycopy(buf, 0, newArray, 0, count);
	return newArray;
}

/**
 * Answer the contents of this ByteArrayOutputStream as a String.  Any changes made
 * to the receiver after returning will not be reflected in the String returned to the caller.
 *
 * @return 		this streams current contents as a String.
 */

public String toString() {
	return new String(buf, 0, count);
}

/**
 * Writes <code>count</code> <code>bytes</code> from the byte array
 * <code>buffer</code> starting at offset <code>index</code> to the
 * ByteArrayOutputStream.
 *
 * @param		buffer		the buffer to be written
 * @param		offset		offset in buffer to get bytes
 * @param		count		number of bytes in buffer to write
 *
 * @throws		NullPointerException 		If buffer is null.
 * @throws		IndexOutOfBoundsException 	If offset or count are outside of bounds.
 */
public synchronized void write(byte[] buffer, int offset, int count) {
	/* Unsure what to do here, spec is unclear */
	if (buf == null) return;
	if (buffer != null) {
		// avoid int overflow
		if (0 <= offset && offset <= buffer.length && 0 <= count && count <= buffer.length - offset) {
			/* Expand if necessary */
			if (this.count + count > buf.length) {
				expand(count);
			}
			System.arraycopy(buffer, offset, buf, this.count, count);
			this.count += count;
		// K002f = Arguments out of bounds
		} else throw new ArrayIndexOutOfBoundsException(com.ibm.oti.util.Msg.getString("K002f"));
	} else throw new NullPointerException();
}

/**
 * Writes the specified byte <code>oneByte</code> to the OutputStream.  Only
 * the low order byte of <code>oneByte</code> is written.
 *
 * @param		oneByte		the byte to be written
 */
public synchronized void write(int oneByte) {
	if (count >= buf.length) {
		expand(1);
	}
	buf[count++] = (byte)oneByte;
}

}
