package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2004  All Rights Reserved
 */

/**
 * DataOutputStream is a filter class which can write typed data
 * to a Stream.  Typically, this stream can be read in by
 * a DataInputStream.  Types that can be written include
 * byte, 16-bit short, 32-bit int, 32-bit float, 64-bit long,
 * 64-bit double, byte strings, and UTF Strings.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			DataInputStream
 */
public class DataOutputStream extends OutputStream implements DataOutput
{
	/**
	 * The target OutputStream for this filter.
	 */
	protected OutputStream out;

/**
 * Constructs a new DataOutputStream on the OutputStream <code>out</code>.  All
 * writes can now be filtered through this stream.  Note that data written by this
 * Stream is not in a human readable format but can be reconstructed by using a
 * DataInputStream on the resulting output.
 *
 * @param		out			the target OutputStream to filter writes on.
 */
public DataOutputStream(OutputStream out) {
	this.out = out;
}

/**
 * Flush this DataOutputStream to ensure all pending data is sent out to the target
 * OutputStream.  This implementation flushes the target OutputStream.
 *
 * @throws 		IOException	If an error occurs attempting to flush this DataOutputStream.
 */
public void flush() throws IOException {
	out.flush();
}

/**
 * Writes <code>count</code> <code>bytes</code> from the byte array
 * <code>buffer</code> starting at offset <code>index</code> to the
 * OutputStream.
 *
 * @param		buffer		the buffer to be written
 * @param		offset		offset in buffer to get bytes
 * @param		count		number of bytes in buffer to write
 *
 * @throws 		IOException 	If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readFully(byte[])
 * @see 		DataInput#readFully(byte[], int, int)
 */
public void write(byte buffer[], int offset, int count) throws IOException {
	if (buffer != null) {
		out.write(buffer, offset, count);
	// K0047 = buffer is null
	} else throw new NullPointerException(com.ibm.oti.util.Msg.getString("K0047"));
}

/**
 * Writes the specified <code>byte</code> to the OutputStream.
 *
 * @param		oneByte		the byte to be written
 *
 * @throws 		IOException 	If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readByte()
 */
public void write(int oneByte) throws IOException {
	out.write(oneByte);
}

/**
 * Writes a boolean to this output stream.
 *
 * @param 		val			the boolean value to write to the OutputStream
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readBoolean()
 */
public final void writeBoolean(boolean val) throws IOException {
	out.write(val ? 1:0);
}

/**
 * Writes a 8-bit byte to this output stream.
 *
 * @param 		val			the byte value to write to the OutputStream
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readByte()
 * @see 		DataInput#readUnsignedByte()
 */
public final void writeByte(int val) throws IOException {
	out.write(val);
}

/**
 * Writes the specified 16-bit character to the OutputStream. Only the
 * lower 2 bytes are written with the higher of the 2 bytes written first.
 * This represents the Unicode value of val.
 *
 * @param		val			the character to be written
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readChar()
 */
public final void writeChar(int val) throws IOException {
	out.write(val >> 8);
	out.write(val);
}

/**
 * Writes the specified 16-bit characters contained in str to the OutputStream.
 * Only the lower 2 bytes of each character are written with the higher
 * of the 2 bytes written first. This represents the Unicode value of each
 * character in str.
 *
 * @param		str			the String whose characters are to be written.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readChar()
 */
public final void writeChars(String str) throws IOException {
	byte newBytes[] = new byte[str.length() * 2];
	for (int index = 0; index < str.length(); index++) {
		int newIndex = index == 0 ? index : index * 2;
		newBytes[newIndex] = (byte)(str.charAt(index) >> 8);
		newBytes[newIndex + 1] = (byte)str.charAt(index);
	}
	out.write(newBytes);
}

/**
 * Writes a 64-bit double to this output stream.  The resulting output
 * is the 8 bytes resulting from calling Double.doubleToLongBits().
 *
 * @param		val			the double to be written.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readDouble()
 */
public final void writeDouble(double val) throws IOException {
	writeLong(Double.doubleToLongBits(val));
}

/**
 * Writes a 32-bit float to this output stream.  The resulting output
 * is the 4 bytes resulting from calling Float.floatToIntBits().
 *
 * @param		val			the float to be written.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readFloat()
 */
public final void writeFloat(float val) throws IOException {
	writeInt(Float.floatToIntBits(val));
}

/**
 * Writes a 32-bit int to this output stream.  The resulting output
 * is the 4 bytes, highest order first, of val.
 *
 * @param		val			the int to be written.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readInt()
 */
public final void writeInt(int val) throws IOException {
	out.write(val >> 24);
	out.write(val >> 16);
	out.write(val >> 8);
	out.write(val);
}

/**
 * Writes a 64-bit long to this output stream.  The resulting output
 * is the 8 bytes, highest order first, of val.
 *
 * @param		val			the long to be written.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readLong()
 */
public final void writeLong(long val) throws IOException {
	writeInt((int)(val >> 32));
	writeInt((int)val);
}

/**
 * Writes the specified 16-bit short to the OutputStream. Only the
 * lower 2 bytes are written with the higher of the 2 bytes written first.
 *
 * @param		val			the short to be written
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readShort()
 * @see 		DataInput#readUnsignedShort()
 */
public final void writeShort(int val) throws IOException {
	writeChar(val);
}

/**
 * Writes the specified String out in UTF format.
 *
 * @param		str			the String to be written in UTF format.
 *
 * @throws 		IOException		If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readUTF()
 */
public final void writeUTF(String str) throws IOException {
	int length = str.length();
	if (length <= DataInputStream.MAX_BUF_SIZE / 3) {
		int size = length * 3;
		byte[] utfBytes = new byte[size];
		int utfIndex = 0;
		for (int i=0; i < length; i++) {
			int charValue = (int) str.charAt(i);
			if (charValue > 0 && charValue <= 127) {
				utfBytes[utfIndex++] = (byte) charValue;
			} else if (charValue <= 2047) {
				utfBytes[utfIndex++] = (byte) (0xc0 | (0x1f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			} else {
				utfBytes[utfIndex++] = (byte) (0xe0 | (0x0f & (charValue >> 12)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			}
		}
		writeShort(utfIndex);
		write(utfBytes, 0, utfIndex);
	} else {
		long utfCount;
		if (length <= 65535 && (utfCount = countUTFBytes(str)) <= 65535) {
			writeShort((int)utfCount);
			writeUTFBytes(str, utfCount);
		// K0068 = String is too long
		} else throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0068"));
	}
}

long countUTFBytes(String str) {
	int utfCount = 0, length = str.length();
	for (int i = 0; i < length; i++) {
		int charValue = (int) str.charAt(i);
		if (charValue > 0 && charValue <= 127)
			utfCount++;
		else if (charValue <= 2047)
			utfCount += 2;
		else
			utfCount += 3;
	}
	return utfCount;
}

void writeUTFBytes(String str, long count) throws IOException {
	boolean single = true;
	int size = (int)count;
	if (count > DataInputStream.MAX_BUF_SIZE) {
		single = false;
		size = DataInputStream.MAX_BUF_SIZE;
	}
	byte[] utfBytes = new byte[size];

	int utfIndex = 0, i = 0, length = str.length();
	int end = length;
	while (i < length) {
		if (!single) {
			end = i + ((utfBytes.length - utfIndex) / 3);
			if (end > length) end = length;
		}
		for (int j=i; j < end; j++) {
			int charValue = (int) str.charAt(j);
			if (charValue > 0 && charValue <= 127) {
				utfBytes[utfIndex++] = (byte) charValue;
			} else if (charValue <= 2047) {
				utfBytes[utfIndex++] = (byte) (0xc0 | (0x1f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			} else {
				utfBytes[utfIndex++] = (byte) (0xe0 | (0x0f & (charValue >> 12)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			}
		}
		if (single || utfIndex > utfBytes.length - 300) {
			write(utfBytes, 0, utfIndex);
			if (single) return;
			utfIndex = 0;
		}
		i = end;
	}
	if (utfIndex > 0) write(utfBytes, 0, utfIndex);
}

/**
 * Close this FilterOutputStream. This implementation closes the target stream.
 *
 * @throws 		IOException		If an error occurs attempting to close this stream.
 */
public void close() throws IOException {
	out.close();
}
}
