package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

import com.ibm.oti.io.CharBuffer;

/**
 * InputStreamReader is class for turning a byte Stream into a character Stream.  Data
 * read from the source input stream is converted into characters by either a default
 * or provided character converter.  By default, the encoding is assumed to ISO8859_1.
 * The InputStreamReader contains a buffer of bytes read from the source input stream
 * and converts these into characters as needed.  The buffer size is 8K.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			OutputStreamWriter
 */
public class InputStreamReader extends Reader {
	private InputStream in;
	com.ibm.oti.io.CharacterConverter converter;
	private byte bytes[] = new byte[4096];
	private int pos, count;
	private int charsAvailable;

/**
 * Constructs a new InputStreamReader on the InputStream <code>in</code>.  Now
 * character reading can be filtered through this InputStreamReader.  This constructor
 * assumes the default conversion of ISO8859_1 (ISO-Latin-1).
 *
 * @param		in		the InputStream to convert to characters.
 */
public InputStreamReader(InputStream in) {
	super(in);
	this.in = in;
	String encoding;
	if ((encoding = System.getProperty("microedition.encoding")) == null)
		encoding = "ISO8859_1";
	converter = com.ibm.oti.io.CharacterConverter.getDefaultConverter(encoding);
}

/**
 * Constructs a new InputStreamReader on the InputStream <code>in</code>.  Now
 * character reading can be filtered through this InputStreamReader.  This constructor
 * takes a String parameter <code>enc</code> which is the name of an encoding.
 * If the encoding cannot be found, an UnsupportedEncodingException error is thrown.
 *
 * @param		in		the InputStream to convert to characters.
 * @param		enc		a String describing the character converter to use.
 *
 * @throws 		UnsupportedEncodingException	if the encoding cannot be found.
 */
public InputStreamReader(InputStream in, final String enc) throws UnsupportedEncodingException {
	super(in);
	this.in = in;
	if ((converter = com.ibm.oti.io.CharacterConverter.getConverter(enc)) == null)
		throw new UnsupportedEncodingException(enc);
}

/**
 * Close this InputStreamReader.  This implementation closes the source InputStream and
 * releases all local storage.
 *
 *
 * @throws 		IOException		If an error occurs attempting to close this InputStreamReader.
 */
public void close() throws IOException {
	synchronized (lock) {
		if (bytes != null) {
			in.close();
			converter = null;
			in = null;
			bytes = null;
		}
	}
}

/**
 * The buffer containing converted characters has been exhausted.
 * Move any unconverted bytes to the beginning of the byte array
 * buffer then try to fill it from the read.  Next, convert
 * whatever can be converted and return that count.
 *
 * @return		the number of bytes read
 *
 * @throws 		IOException
 */
private int fillbuf(int length) throws IOException {
	/* If unconverted bytes still available, move them down */
	if (count > 0 && pos > 0) {
		count -= pos;
		if (count > 0) System.arraycopy(bytes, pos, bytes, 0, count);
		pos = 0;
	}
	int room = bytes.length - count;
	int result = in.read(bytes, count, length < room ? length : room);
	if (result >= 0) {
		count += result;
		charsAvailable = 0;
	} else {
		// EOF but need more bytes
		if (count > 0)
			throw new IOException();
	}
	return result;
}

private int convert(char[] buffer, int offset, int length) throws IOException {
	// Ask the converter how many characters we will get back */
	CharBuffer charBuf = new CharBuffer(buffer, offset, length);
	int result = converter.convert(bytes, pos, count - pos, charBuf);
	pos = result;
	// byte format error
	if (result != -1) {
		return charBuf.getPos() - offset;
	} else throw new IOException();
}

/**
 * Answers a boolean indicating whether or not this Reader supports mark() and
 * reset().
 *
 * @return 		<code>true</code> if mark() and reset() are supported, <code>false</code>
 *				otherwise.
 */
public boolean markSupported() {
	if (in != null)
		return in.markSupported();
	return false;
}

/**
 * Set a Mark position in this Reader.  The parameter <code>readLimit</code> indicates how many
 * characters can be read before a mark is invalidated.  Sending reset() will reposition the
 * reader back to the marked position provided <code>readLimit</code> has not been surpassed.
 *
 * @param		readLimit	an int representing how many characters must be read before
 *							invalidating the mark.
 *
 * @throws 		IOException	If an error occurs attempting mark this Reader.
 */
public void mark(int readLimit) throws IOException {
	if (in != null && in.markSupported()) in.mark(readLimit);
	else throw new IOException();
}

/**
 * Reset this Readers position to the last <code>mark()</code> location.
 * Invocations of <code>read()/skip()</code> will occur from this new location.
 * If this Reader was not marked, the implementation of <code>reset()</code>
 * is implementation specific.  See the comment for the specific Reader subclass
 * for implementation details.  The default action is to throw <code>IOException</code>.
 *
 * @throws 		IOException If a problem occured or the receiver does not support <code>mark()/reset()</code>.
 */
public void reset() throws IOException {
	if (in != null) in.reset();
	else throw new IOException();
}

/**
 * Skips <code>count</code> number of characters in this Reader.  Subsequent
 * <code>read()</code>'s will not return these characters unless <code>reset()</code>
 * is used.  This method may perform multiple reads to read <code>count</code>
 * characters.
 *
 * @param		count		the maximum number of characters to skip.
 * @return		the number of characters actually skipped.
 *
 * @throws 		IOException If the Reader is already closed or some other IO error occurs.
 */
public long skip(long count) throws IOException {
	return super.skip(count);
}

/**
 * Reads a single character from this InputStreamReader and returns the result as
 * an int.  The 2 higher-order characters are set to 0. If the end of reader was encountered
 * then return -1.  The byte value is either obtained from converting bytes in this readers
 * buffer or by first filling the buffer from the source InputStream and then reading from the
 * buffer.
 *
 * @return 		the character read or -1 if end of reader.
 *
 * @throws 		IOException If the InputStreamReader is already closed or some other IO error occurs.
 */
public int read() throws IOException {
	synchronized (lock) {
		if (bytes != null) {
			char[] buf = new char[1];
			int converted = 0;
			if (pos < count)
				converted = convert(buf, 0, 1);
			while (converted == 0) {
				int read = fillbuf(1);
				if (read == -1) return -1;
				converted = convert(buf, 0, 1);
			}
			return buf[0];
		// K0070 = InputStreamReader is closed.
		} else throw new IOException(com.ibm.oti.util.Msg.getString("K0070"));
	}
}

/**
 * Reads at most <code>count</code> characters from this Reader and stores them
 * at <code>offset</code> in the character array <code>buf</code>.  Returns the
 * number of characters actually read or -1 if the end of reader was encountered.
 * The bytes are either obtained from converting bytes in this readers buffer or
 * by first filling the buffer from the source InputStream and then reading from the
 * buffer.
 *
 * @param 		buf			character array to store the read characters
 * @param 		offset 		offset in buf to store the read characters
 * @param 		length 		maximum number of characters to read
 * @return		the number of characters read or -1 if end of reader.
 *
 * @throws 		IOException If the InputStreamReader is already closed or some other IO error occurs.
 */
public int read(char[] buf, int offset, int length) throws IOException {
	// avoid int overflow
	if (0 <= offset && offset <= buf.length && 0 <= length && length <= buf.length - offset) {
		int copylength = 0, newoffset = offset;
		synchronized (lock) {
			if (bytes != null) {
				int converted = 0;
				if (pos < count) {
					converted = convert(buf, offset, length);
					copylength += converted;
				}
				// always do the first read
				int available = 1;
				while (copylength < length) {
					newoffset += converted;
					if (copylength > 0 && available <= 0 && (available = in.available()) <= 0)
						break;
					int read = fillbuf(length - copylength);
					if (read == -1)
						return copylength == 0 ? -1 : copylength;
					available -= read;
					converted = convert(buf, newoffset, length - copylength);
					copylength += converted;
				}
				return copylength;
			// K0070 = InputStreamReader is closed.
			} else throw new IOException(com.ibm.oti.util.Msg.getString("K0070"));
		}
	} else throw new IndexOutOfBoundsException();
}

/**
 * Answers a <code>boolean</code> indicating whether or not this
 * InputStreamReader is ready to be read without blocking.  If the result
 * is <code>true</code>, the next <code>read()</code> will not
 * block.  If the result is <code>false</code> this Reader may
 * or may not block when <code>read()</code> is sent.  This implementation
 * answers <code>true</code> if there are bytes available in the buffer or
 * the source InputStream has bytes available.
 *
 * @return 		<code>true</code> if the receiver will not block when <code>read()</code> is
 *				called, <code>false</code> if unknown or blocking will occur.
 *
 * @throws 		IOException If the InputStreamReader is already closed or some other IO error occurs.
 */
public boolean ready() throws IOException {
	synchronized (lock) {
		if (bytes != null)
			return pos < count || in.available() > 0;
		// K0070 = InputStreamReader is closed.
		throw new IOException(com.ibm.oti.util.Msg.getString("K0070"));
	}
}

}
