package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * OutputStreamWriter is a class for turning a character output stream into a byte
 * output stream. The conversion of Unicode characters to their byte equivalents
 * is determinded by the converter used.  By default, the encoding is ISO8859_1 (ISO-Latin-1) but
 * can be changed by calling the constructor which takes an encoding.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			InputStreamReader
 */

public class OutputStreamWriter extends Writer {
	private OutputStream out;
	com.ibm.oti.io.CharacterConverter converter;
	private byte buf[] = new byte[8192];
	private int pos;
/**
 * Constructs a new OutputStreamWriter using <code>out</code> as the OutputStream to
 * write converted characters to.  The default character encoding is used (see class
 * description).
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		out			the non-null OutputStream to write converted bytes to.
 */

public OutputStreamWriter(OutputStream out) {
	super(out);
	this.out = out;
	String encoding;
	if ((encoding = System.getProperty("microedition.encoding")) == null)
		encoding = "ISO8859_1";
	converter = com.ibm.oti.io.CharacterConverter.getDefaultConverter(encoding);
}

/**
 * Constructs a new OutputStreamWriter using <code>out</code> as the OutputStream to
 * write converted characters to and <code>end</code> as the character encoding.
 * If the encoding cannot be found, an UnsupportedEncodingException error is thrown.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		out			the non-null OutputStream to write converted bytes to.
 * @param		enc			the non-null String describing the desired character encoding.
 *
 * @throws 		UnsupportedEncodingException	if the encoding cannot be found.
 */

public OutputStreamWriter(OutputStream out, final String enc) throws UnsupportedEncodingException {
	super(out);
	this.out = out;
	if ((converter = com.ibm.oti.io.CharacterConverter.getConverter(enc)) == null)
		throw new UnsupportedEncodingException(enc);
}

/**
 * Close this OutputStreamWriter. This implementation first flushes the
 * buffer and the target OutputStream.  The OutputStream is then closed
 * and the resources for the buffer and converter are freed.
 * <p>
 * Only the first invocation of this method has any effect.  Subsequent calls do no
 * work.
 *
 * @author		OTI
 * @version		initial
 *
 * @throws 		IOException	If an error occurs attempting to close this OutputStreamWriter.
 */
public void close() throws IOException {
	synchronized (lock) {
		if (isOpen()) {
			byte[] bytes = converter.getClosingBytes();
			if (bytes.length > 0) {
				writeBytes(bytes);
			}
			flush();
			out.flush();
			out.close();
			buf = null;
			converter = null;
		}
	}
}
/**
 * Flush this OutputStreamWriter.  This implementation ensures all
 * buffered bytes are written to the target OutputStream.  After writing the
 * bytes, the target OutputStream is then flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @throws 		IOException	If an error occurs attempting to flush this OutputStreamWriter.
 */

public void flush() throws IOException {
	synchronized (lock) {
		if (isOpen()) {
			if (pos > 0) out.write(buf, 0, pos);
			out.flush();
			pos = 0;
		// K0073 = OutputStreamWriter is closed.
		} else throw new IOException(com.ibm.oti.util.Msg.getString("K0073"));
	}
}

private boolean isOpen() {
	return converter != null;
}

/**
 * Writes <code>count</code> characters starting at <code>offset</code> in <code>buf</code> to this
 * Writer.  The characters are immediately converted to bytes by the character converter and stored
 * in a local buffer.  If the buffer becomes full as a result of this write, this Writer is
 * flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		buf			the non-null array containing characters to write.
 * @param 		offset 		offset in buf to retrieve characters
 * @param 		count 		maximum number of characters to write
 *
 * @throws 		IOException 					If this OuputStreamWriter has already been closed or some other
 *													IOException occurs.
 * @throws		ArrayIndexOutOfBoundsException 	If offset or count are outside of bounds.
 */
public void write(char[] buf, int offset, int count) throws IOException {
	// avoid int overflow
	if (0 <= offset && offset <= buf.length && 0 <= count && count <= buf.length - offset) {
		synchronized (lock) {
			if (isOpen()) {
				byte converted[] = converter.convert(buf, offset, count);
				writeBytes(converted);
			// K0073 = OutputStreamWriter is closed.
			} else throw new IOException(com.ibm.oti.util.Msg.getString("K0073"));
		}
	} else throw new ArrayIndexOutOfBoundsException();
}

private void writeBytes(byte[] converted) throws IOException {
	if (pos + converted.length > this.buf.length) {
		out.write(this.buf, 0, pos);
		out.write(converted, 0, converted.length);
		out.flush();
		pos = 0;
		return;
	}
	System.arraycopy(converted, 0, this.buf, pos, converted.length);
	pos += converted.length;
}

/**
 * Writes out the character <code>oneChar</code> to this Writer.  The low-order 2 bytes are
 * immediately converted to bytes by the character converter and stored in a local buffer.
 * If the buffer becomes full as a result of this write, this Writer is flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		oneChar		the character to write
 *
 * @throws 		IOException If this OuputStreamWriter has already been closed or some other IOException occurs.
 */
public void write(int oneChar) throws IOException {
	synchronized (lock) {
		if (isOpen()) {
			char convert[] = new char[] {(char) oneChar};
			byte converted[] = converter.convert(convert, 0, 1);
			if (pos + converted.length > this.buf.length) {
				out.write(this.buf, 0, pos);
				out.write(converted, 0, converted.length);
				out.flush();
				pos = 0;
				return;
			}
			System.arraycopy(converted, 0, this.buf, pos, converted.length);
			pos += converted.length;
		// K0073 = OutputStreamWriter is closed.
		} else throw new IOException(com.ibm.oti.util.Msg.getString("K0073"));
	}
}
/**
 * Writes <code>count</code> characters starting at <code>offset</code> in <code>str</code> to this
 * Writer.  The characters are immediately converted to bytes by the character converter and stored
 * in a local buffer.  If the buffer becomes full as a result of this write, this Writer is
 * flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		str			the non-null String containing characters to write.
 * @param 		offset 		offset in str to retrieve characters
 * @param 		count 		maximum number of characters to write
 *
 * @throws 		IOException 					If this OuputStreamWriter has already been closed or some other IOException occurs.
 * @throws		ArrayIndexOutOfBoundsException 	If offset or count are outside of bounds.
 */
public void write(String str, int offset, int count) throws IOException {
	// avoid int overflow
	if (0 <= offset && offset <= str.length() && 0 <= count && count <= str.length() - offset) {
		synchronized (lock) {
			char convert[] = new char[count];
			str.getChars(offset, offset + count, convert, 0);
			write(convert, 0, convert.length);
		}
	} else throw new StringIndexOutOfBoundsException();
}
}
