package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Floats are objects (non-base types) which represent
 * float values.
 *
 * @author		OTI
 * @version		initial
 */
public final class Float extends Object {

	/**
	 * The value which the receiver represents.
	 */
	final float value;

	/**
	 * Largest and smallest possible float values.
	 */
	public static final float MAX_VALUE = 3.40282346638528860e+38f;
	public static final float MIN_VALUE = 1.40129846432481707e-45f;

	/**
	 * A value which represents all invalid float results
	 * (NaN ==> Not a Number)
	 */
	public static final float NaN = 0.0f / 0.0f;

	/**
	 * Values to represent infinite results
	 */
	public static final float POSITIVE_INFINITY = 1.0f / 0.0f;
	public static final float NEGATIVE_INFINITY = -1.0f / 0.0f;

/**
 * Constructs a new instance of the receiver which represents
 * the float valued argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		the float to store in the new instance.
 */
public Float(float value) {
	this.value = value;
}
/**
 * Constructs a new instance of the receiver which represents
 * the double valued argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		the double to store in the new instance.
 */
public Float(double value) {
	this.value = (float)value;
}

/**
 * Answers the byte value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		byte		the value of the receiver.
 */
public byte byteValue() {
	return (byte) value;
}

/**
 * Answers the double value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		double		the value of the receiver.
 */
public double doubleValue() {
	return (double) value;
}
/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison. For Floats, the check verifies that
 * the receiver's value's bit pattern matches the bit pattern
 * of the argument, which must also be a Float.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object		the object to compare with this object
 * @return		<code>true</code>
 *					if the object is the same as this object
 *				<code>false</code>
 *					if it is different from this object
 * @see			#hashCode
 */
public boolean equals(Object object) {
	return (object == this) ||
		(object instanceof Float) &&
		(floatToIntBits(this.value) == floatToIntBits(((Float) object).value));
}
/**
 * Answers the binary representation of the argument, as
 * an int.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		The float value to convert
 * @return		the bits of the float.
 */
public static native int floatToIntBits(float value);

/**
 * Answers the receiver's value as a float.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's value
 */
public float floatValue() {
	return value;
}
/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>equals</code> must answer the same value for this
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	return floatToIntBits(value);
}
/**
 * Answers a float built from the binary representation
 * given in the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bits		the bits of the float
 * @return		the float which matches the bits
 */
public static native float intBitsToFloat(int bits);
/**
 * Answers the int value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int		the value of the receiver.
 */
public int intValue() {
	return (int) value;
}

/**
 * Answers true if the receiver represents an infinite
 * quantity, and false otherwise.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					if the argument is positive or negative infinity
 *              <code>false</code>
 *                  if it is not an infinite value
 */
public boolean isInfinite() {
	return isInfinite(value);
}

/**
 * Answers true if the argument represents an infinite
 * quantity, and false otherwise.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f			value to check for infinitness.
 * @return		<code>true</code>
 *					if the argument is positive or negative infinity
 *              <code>false</code>
 *                  if it is not an infinite value
 */
public static boolean isInfinite(float f) {
	return (f == POSITIVE_INFINITY) || (f == NEGATIVE_INFINITY);
}

/**
 * Answers true if the receiver does not represent
 * a valid float quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code>
 *					if the argument is Not A Number
 *              <code>false</code>
 *                  if it is a (potentially infinite) float number
 */
public boolean isNaN() {
	return isNaN(value);
}

/**
 * Answers true if the argument does not represent
 * a valid float quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f			value to check for numberness.
 * @return		<code>true</code>
 *					if the argument is Not A Number
 *              <code>false</code>
 *                  if it is a (potentially infinite) float number
 */
public static boolean isNaN(float f) {
	return f != f;
}
/**
 * Answers the long value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		long		the value of the receiver.
 */
public long longValue() {
	return (long) value;
}
/**
 * Answers the float which matches the passed in string.
 * NumberFormatException is thrown if the string does not
 * represent a valid float.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		the value to convert
 * @return		a float which would print as the argument
 */
public static float parseFloat(String string) throws NumberFormatException {
	return com.ibm.oti.util.FloatingPointParser.parseFloat(string);
}

/**
 * Answers the short value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		short		the value of the receiver.
 */
public short shortValue() {
	return (short) value;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return Float.toString(value);
}
/**
 * Answers a string containing a printable representation
 * of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f		the float to print
 * @return		a printable representation of the argument.
 */
public static String toString(float f) {
	return com.ibm.oti.util.NumberConverter.convert(f);
}

/**
 * Answers the float which matches the passed in string.
 * NumberFormatException is thrown if the string does not
 * represent a valid float.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		the value to convert
 * @return		a float which would print as the argument
 */
public static Float valueOf(String string) throws NumberFormatException {
	return new Float(parseFloat(string));
}

}
