package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

/**
 * StackTraceElement represents a stack frame.
 *
 * @see Throwable#getStackTrace()
 */
final class StackTraceElement {
	String declaringClass, methodName, fileName;
	int lineNumber;

private StackTraceElement() {} // prevent instantiation from java code - only the VM creates these

/**
 * Returns true if the specified object is another StackTraceElement instance
 * representing the same execution point as this instance.
 *
 */
public boolean equals(Object obj) {
	if (!(obj instanceof StackTraceElement)) return false;
	StackTraceElement castObj = (StackTraceElement) obj;

	// Unknown methods are never equal to anything (not strictly to spec, but spec does not allow null method/class names)
	if ((methodName == null) || (castObj.methodName == null)) return false;

	if (!getMethodName().equals(castObj.getMethodName())) return false;
	if (!getClassName().equals(castObj.getClassName())) return false;
	String localFileName = getFileName();
	if (localFileName == null) {
		if (castObj.getFileName() != null) return false;
	} else {
		if (!localFileName.equals(castObj.getFileName())) return false;
	}
	if (getLineNumber() != castObj.getLineNumber()) return false;

	return true;
}

/**
 * Returns the full name (i.e. including package) of the class where this
 * stack trace element is executing.
 *
 * @return the name of the class where this stack trace element is
 *         executing.
 */
public String getClassName() {
	return (declaringClass == null) ? "<unknown class>" : declaringClass;
}

/**
 * If available, returns the name of the file containing the Java code
 * source which was compiled into the class where this stack trace element
 * is executing.
 *
 * @return the name of the Java code source file which was compiled into the
 *         class where this stack trace element is executing. If not
 *         available, a <code>null</code> value is returned.
 */
public String getFileName() {
	return fileName;
}

/**
 * Returns the source file line number for the class where this stack trace
 * element is executing.
 *
 * @return the line number in the source file corresponding to where this
 *         stack trace element is executing.
 */
public int getLineNumber() {
	return lineNumber;
}

/**
 * Returns the name of the method where this stack trace element is
 * executing.
 *
 * @return the method in which this stack trace element is executing.
 *         Returns &lt;<code>unknown method</code>&gt; if the name of the
 *         method cannot be determined.
 */
public String getMethodName() {
	return (methodName == null) ? "<unknown method>" : methodName;
}

/**
 * Returns a hash code value for this stack trace element.
 */
public int hashCode() {
	// either both methodName and declaringClass are null, or neither are null
	if (methodName == null) return 0;	// all unknown methods hash the same
	return methodName.hashCode() ^ declaringClass.hashCode();	// declaringClass never null if methodName is non-null
}

/**
 * Returns <code>true</code> if the method name returned by
 * {@link #getMethodName()} is implemented as a native method.
 */
public boolean isNativeMethod() {
	return lineNumber == -2;
}

/**
 * Returns a string representation of this stack trace element.
 */
public String toString() {
	StringBuffer buf = new StringBuffer(80);

	buf.append(getClassName());
	buf.append('.');
	buf.append(getMethodName());

	if (isNativeMethod()) {
		buf.append("(Native Method)");
	} else {
		String fileName = getFileName();

		if (fileName == null) {
			buf.append("(Unknown Source)");
		} else {
			int lineNumber = getLineNumber();

			buf.append('(');
			buf.append(fileName);
			if (lineNumber >= 0) {
				buf.append(':');
				buf.append(lineNumber);
			}
			buf.append(')');
		}
	}
	return buf.toString();
}
}
