package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

import java.io.*;
import java.util.Hashtable;

/**
 * Class System provides a standard place for programs
 * to find system related information. All System API
 * is static.
 *
 * @author		OTI
 * @version		initial
 */
public final class System {

	// The standard input, output, and error streams.
	// Typically, these are connected to the shell which
	// ran the Java program.
	/**
	 * Default output stream
	 */
	public static final PrintStream out;
	/**
	 * Default error output stream
	 */
	public static final PrintStream err;

	//Get a ref to the Runtime instance for faster lookup
	private static final Runtime RUNTIME = Runtime.getRuntime();
	private static Hashtable systemProperties;

	// Initialize all the slots in System on first use.
	static {

		// Fill in the properties from the VM information.
		ensureProperties();
		out = new String.ConsolePrintStream(com.ibm.oti.connection.file.FileOutputStream.out);
		err = new String.ConsolePrintStream(com.ibm.oti.connection.file.FileOutputStream.err);
	}

static void completeInitialization() {
	String extraLibs = (String)systemProperties.get("com.ibm.util.extralibs.properties");
	if (extraLibs != null) {
		int lastIndex = 0;
		while (lastIndex < extraLibs.length()) {
			int index = extraLibs.indexOf(':', lastIndex);
			if (index == -1) index = extraLibs.length();
			String libName = extraLibs.substring(lastIndex, index).trim();
			lastIndex = index + 1;
			if (libName.length() == 0) continue;
			try {
				String extensionName = "com.ibm.oti." + libName + ".SystemPropertyExtension";
				Class extensionCls = Class.forName(extensionName);
				Object extension = extensionCls.newInstance();
				try {
					if (extension instanceof com.ibm.oti.util.SystemPropertiesHook) {
						systemProperties = ((com.ibm.oti.util.SystemPropertiesHook)extension).extendSystemProperties(systemProperties);
					} else {
					((com.ibm.oti.util.SystemProperties)extension).extendSystemProperties(systemProperties);
					}
				} catch (Throwable t) {
					t.printStackTrace();
				}
			} catch (Throwable t) {
			}
		}
	}
}

/**
 * Prevents this class from being instantiated.
 */
private System() {
}

/**
 * Copies the contents of <code>array1</code> starting at offset <code>start1</code>
 * into <code>array2</code> starting at offset <code>start2</code> for
 * <code>length</code> elements.
 *
 * @param		array1 		the array to copy out of
 * @param		start1 		the starting index in array1
 * @param		array2 		the array to copy into
 * @param		start2 		the starting index in array2
 * @param		length 		the number of elements in the array to copy
 */
public static native void arraycopy(Object array1, int start1, Object array2, int start2, int length);

/**
 * Private version of the arraycopy method used by the jit for reference arraycopies
 */
private static void arraycopy(Object[] A1, int offset1, Object[] A2, int offset2, int length) {
	if (A1 == null || A2 == null) throw new NullPointerException();
	if (offset1 >= 0 && offset2 >= 0 && length >= 0 &&
		length <= A1.length - offset1 && length <= A2.length - offset2)
	{
		// Check if this is a forward or backwards arraycopy
		if (A1 != A2 || offset1 > offset2 || offset1+length <= offset2) {
			for (int i = 0; i < length; ++i) {
				A2[offset2+i] = A1[offset1+i];
			}
		} else {
			for (int i = length-1; i >= 0; --i) {
				A2[offset2+i] = A1[offset1+i];
			}
		}
	} else throw new ArrayIndexOutOfBoundsException();
}

/**
 * Answers the current time expressed as milliseconds since
 * the time 00:00:00 UTC on January 1, 1970.
 *
 * @return		the time in milliseconds.
 */
public static native long currentTimeMillis();

/**
 * Get the host name for the microedition.hostname property.
 * This native may return null if the host name is not available.
 */
private static native String getHostnameImpl() throws IOException;

private static final int InitLocale = 0;
private static final int PlatformEncoding = 1;
private static final int FileEncoding = 2;
private static final int OSEncoding = 3;

/**
 * If systemProperties is unset, then create a new one based on the values
 * provided by the virtual machine.
 */
private static void ensureProperties() {
	systemProperties = new Hashtable();

	String platformEncoding = null;
	String fileEncoding, osEncoding = null;
	String definedFileEncoding = getEncoding(FileEncoding);
	String definedOSEncoding = getEncoding(OSEncoding);
	if (definedFileEncoding != null) {
		fileEncoding = definedFileEncoding;
		// if file.encoding is defined, and os.encoding is not, use the detected
		// platform encoding for os.encoding
		if (definedOSEncoding == null) {
			platformEncoding = getEncoding(PlatformEncoding);
			osEncoding = platformEncoding;
		} else {
			getEncoding(InitLocale);
		}
	} else {
		platformEncoding = getEncoding(PlatformEncoding);
		fileEncoding = platformEncoding;
	}
	// if os.encoding is not defined, file.encoding will be used
	if (osEncoding == null)
		osEncoding = definedOSEncoding;
	if (osEncoding != null)
		systemProperties.put("os.encoding", osEncoding);

	systemProperties.put("microedition.encoding", fileEncoding);

	systemProperties.put("microedition.configuration", "CLDC-1.1");
	systemProperties.put("microedition.profiles", "");
	systemProperties.put("microedition.profiles", "MIDP-2.0");
	systemProperties.put("microedition.commports",getCommPortList());

	systemProperties.put("com.ibm.oti.configuration", "midp20");
	systemProperties.put("com.ibm.oti.configuration.dir", "jclMidp20");

	String[] list = getPropertyList();
	for (int i=0; i<list.length; i+=2) {
		String key = list[i];
		if (key == null) break;
		systemProperties.put(key, list[i+1]);
	}

	String consoleEncoding = (String)systemProperties.get("console.encoding");
	if (consoleEncoding == null) {
		if (platformEncoding == null)
			platformEncoding = getEncoding(PlatformEncoding);
		consoleEncoding = platformEncoding;
		systemProperties.put("console.encoding", consoleEncoding);
	}

	if (systemProperties.get("microedition.platform") == null)
		systemProperties.put("microedition.platform", systemProperties.get("os.name"));
	String lang = (String)systemProperties.get("user.language");
	if (lang == null) lang = "en";
	String reg = (String)systemProperties.get("user.region");
	if (reg == null) reg = "US";
	String locale = lang + "-" + reg;
	String var = (String)systemProperties.get("user.variant");
	if (var != null && var.length() != 0) locale = locale + "-" + var;
	systemProperties.put("microedition.locale", locale);

	systemProperties.put("com.ibm.oti.jcl.build", "20061020_1321");
}
/**
 * Causes the virtual machine to stop running, and the
 * program to exit. If runFinalizersOnExit(true) has been
 * invoked, then all finalizers will be run first.
 *
 * @param		code		the return code.
 *
 * @throws		SecurityException 	if the running thread is not allowed to cause the vm to exit.
 *
 * @see			SecurityManager#checkExit
 */
public static void exit(int code) {
	RUNTIME.exit(code);
}

/**
 * Indicate to the virtual machine that it would be a
 * good time to collect available memory. Note that, this
 * is a hint only.
 */
public static void gc() {
	RUNTIME.gc();
}

/**
 * Answers the value of a particular system property.
 * Answers null if no such property exists,
 * <p>
 * The properties currently provided by the virtual
 * machine are:
 * <pre>
 *     java.vendor.url
 *     java.class.path
 *     user.home
 *     java.class.version
 *     os.version
 *     java.vendor
 *     user.dir
 *     user.timezone
 *     path.separator
 *     os.name
 *     os.arch
 *     line.separator
 *     file.separator
 *     user.name
 *     java.version
 *     java.home
 * </pre>
 *
 * @param		prop		the system property to look up
 * @return		the value of the specified system property, or null if the property doesn't exist
 */
public static String getProperty(String prop) {
	if (!prop.equals("")) {
		String result = (String)systemProperties.get(prop);
		if (result == null && prop.equals("microedition.hostname")) {
			try {
				result = getHostnameImpl();
				systemProperties.put("microedition.hostname", result);
			} catch (IOException e) {
				systemProperties.put("microedition.hostname", e.toString());
			}
		}
		return result;
	}
	// K00de = key is empty
	throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00de"));
}

/**
 * Sets the value of a particular system property.
 *
 * @param		prop		the system property to change
 * @param		value		the value to associate with prop
 * @return		the old value of the property, or null
 */
static String setProperty(String prop, String value) {
	return (String)systemProperties.put(prop, value);
}

/**
 * Answers an array of Strings containing key..value pairs
 * (in consecutive array elements) which represent the
 * starting values for the system properties as provided
 * by the virtual machine.
 *
 * @return		the default values for the system properties.
 */
private static native String [] getPropertyList();

/**
 * Return a string listing the available comm ports.
 */
private static native String getCommPortList();

/**
 * Return the requested encoding.
 * 		0 - initialize locale
 * 		1 - detected platform encoding
 * 		2 - command line defined file.encoding
 * 		3 - command line defined os.encoding
 */
private static native String getEncoding(int type);

/**
 * Answers an integer hash code for the parameter.
 * The hash code returned is the same one that would
 * be returned by java.lang.Object.hashCode(), whether
 * or not the object's class has overridden hashCode().
 * The hash code for null is 0.
 *
 * @param		anObject	the object
 * @return		the hash code for the object
 *
 * @see			java.lang.Object#hashCode
 */
public static native int identityHashCode(Object anObject);

}
