
package java.lang;

import java.io.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * This class is the superclass of all classes which
 * can be thrown by the virtual machine. The two direct
 * subclasses represent recoverable exceptions (Exception)
 * and unrecoverable errors (Error). This class provides
 * common methods for accessing a string message which
 * provides extra information about the circumstances in
 * which the throwable was created, and for filling in a
 * walkback (i.e. a record of the call stack at a
 * particular point in time) which can be printed later.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Error
 * @see			Exception
 * @see			RuntimeException
 */
public class Throwable {

	/**
	 * The message provided when the exception was created.
	 */
	private String detailMessage;

	/**
	 * An object which describes the walkback. This field is stored
	 * by the fillInStackTrace() native, and used by the
	 * getStackTraceImpl() native.
	 */
	private transient Object walkback;

/**
 * Constructs a new instance of this class with its
 * walkback filled in.
 */
public Throwable () {
	super ();
	fillInStackTrace();
}

/**
 * Constructs a new instance of this class with its
 * walkback and message filled in.
 *
 * @param		detailMessage String
 *				The detail message for the exception.
 */
public Throwable (String detailMessage) {
	this ();
	this.detailMessage = detailMessage;
}

/**
 * Record in the receiver a walkback from the point
 * where this message was sent. The message is
 * public so that code which catches a throwable and
 * then <em>re-throws</em> it can adjust the walkback
 * to represent the location where the exception was
 * re-thrown.
 *
 * @return the receiver
 */
native Throwable fillInStackTrace();

/**
 * Answers the extra information message which was provided
 * when the throwable was created. If no message was provided
 * at creation time, then answer null.
 *
 * @return		String
 *				The receiver's message.
 */
public String getMessage() {
	return detailMessage;
}

/**
 * Answers an array of StackTraceElement. Each StackTraceElement represents
 * a entry on the stack.
 *
 * @return an array of StackTraceElement representing the stack
 */
private native StackTraceElement[] getStackTraceImpl();

/**
 * Outputs a printable representation of the receiver's
 * walkback on the System.err stream.
 */
public void printStackTrace () {
	printStackTrace(System.err);
}

/**
 * Outputs a printable representation of the receiver's
 * walkback on the stream specified by the argument.
 *
 * @param		err PrintStream
 *				The stream to write the walkback on.
 */
void printStackTrace (PrintStream err) {
	err.println(toString());
	StackTraceElement[] stack = getStackTraceImpl();
	for (int i=0; i<stack.length; i++)
		err.println("\tat " + stack[i]);

}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @return		String
 *				a printable representation for the receiver.
 */
public String toString () {
	String msg = getMessage();
	String name = getClass().getName();
	if (msg == null) {
		return name;
	} else {
		return new StringBuffer(name.length() + 2 + msg.length())
			.append(name).append(": ").append(msg).toString();
	}
}

}
