package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Calendar is an abstract class which provides the conversion between Dates and
 * integer calendar fields, such as the month, year or minute. Subclasses of this
 * class implement a specific calendar type, such as the gregorian calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Date
 * @see			GregorianCalendar
 * @see			TimeZone
 */
public abstract class Calendar {
	/**
	 * Set to true when the calendar fields have been set from the time, set
	 * to false when a field is changed and the fields must be recomputed.
	 */
	boolean areFieldsSet;
	/**
	 * An integer array of calendar fields.
	 */
	protected int[] fields;
	/*
	 * A boolean array. Each element indicates if the corresponding field has been set.
	 */
	protected boolean[] isSet;
	/**
	 * Set to true when the time has been set, set to false when a field is
	 * changed and the time must be recomputed.
	 */
	boolean isTimeSet;
	/**
	 * The time in milliseconds since January 1, 1970.
	 */
	protected long time;

	transient int lastTimeFieldSet;
	private TimeZone zone;

	public static final int
		JANUARY = 0,
		FEBRUARY = 1,
		MARCH = 2,
		APRIL = 3,
		MAY = 4,
		JUNE = 5,
		JULY = 6,
		AUGUST = 7,
		SEPTEMBER = 8,
		OCTOBER = 9,
		NOVEMBER = 10,
		DECEMBER = 11,

		SUNDAY = 1,
		MONDAY = 2,
		TUESDAY = 3,
		WEDNESDAY = 4,
		THURSDAY = 5,
		FRIDAY = 6,
		SATURDAY = 7;

	public static final int
		YEAR = 1,
		MONTH = 2,
		DATE = 5,
		DAY_OF_MONTH = 5,
		DAY_OF_WEEK = 7,

		AM_PM = 9,
		HOUR = 10,
		HOUR_OF_DAY = 11,
		MINUTE = 12,
		SECOND = 13,
		MILLISECOND = 14,

		AM = 0,
		PM = 1;

	static final int FIELD_COUNT = 15;

/**
 * Initializes this Calendar instance using the default TimeZone and
 * Locale.
 *
 * @author		OTI
 * @version		initial
 */
protected Calendar() {
	isSet = new boolean[FIELD_COUNT];
	fields = new int[FIELD_COUNT];
	setTimeZone(TimeZone.getDefault());
}

/**
 * Answers if the Date specified by this Calendar instance is
 * after the Date specified by the parameter. The comparison
 * is not dependent on the timezones of the Calendars.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		calendar	the Calendar instance to compare
 * @return		true when this Calendar is after calendar, false otherwise
 *
 * @exception	IllegalArgumentException when the time is not set and the
 *				time cannot be computed from the current field values
 */
public boolean after(Object calendar) {
	if (!(calendar instanceof Calendar)) return false;
	return getTimeInMillis() > ((Calendar)calendar).getTimeInMillis();
}

/**
 * Answers if the Date specified by this Calendar instance is
 * before the Date specified by the parameter. The comparison
 * is not dependent on the timezones of the Calendars.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		calendar	the Calendar instance to compare
 * @return		true when this Calendar is before calendar, false otherwise
 *
 * @exception	IllegalArgumentException when the time is not set and the
 *				time cannot be computed from the current field values
 */
public boolean before(Object calendar) {
	if (!(calendar instanceof Calendar)) return false;
	return getTimeInMillis() < ((Calendar)calendar).getTimeInMillis();
}

/**
 * Computes the time from the fields if the time has not
 * already been set. Computes the fields from the time if the
 * fields are not already set.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	IllegalArgumentException when the time is not set and the
 *				time cannot be computed from the current field values
 */
void complete () {
	if (!isTimeSet) {
		computeTime();
		isTimeSet = true;
	}
	if (areFieldsSet) {
		for (int i=0; i<isSet.length; i++)
			isSet[i] = true;
	} else {
		computeFields();
		areFieldsSet = true;
	}
}

/**
 * Computes the Calendar fields from the time.
 *
 * @author		OTI
 * @version		initial
 */
protected abstract void computeFields();

/**
 * Computes the time from the Calendar fields.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	IllegalArgumentException when the time cannot be computed
 *				from the current field values
 */
protected abstract void computeTime();

/**
 * Compares the specified object to this Calendar and answer if they
 * are equal. The object must be an instance of Calendar and have the
 * same properties.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object	the object to compare with this object
 * @return		true if the specified object is equal to this Calendar, false otherwise
 */
public boolean equals(Object object) {
	if (this == object) return true;
	if (!(object instanceof Calendar)) return false;
	Calendar cal = (Calendar)object;
	return getTimeInMillis() == cal.getTimeInMillis() &&
		getTimeZone().equals(cal.getTimeZone());
}

/**
 * Gets the value of the specified field after computing the field
 * values from the time if required.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		field	the field
 * @return		the value of the specified field
 *
 * @exception	IllegalArgumentException when the fields are not set, the time
 *				is not set, and the time cannot be computed from the current
 *				field values
 */
public final int get(int field) {
	if (field >= DAY_OF_WEEK)
	complete();
	return fields[field];
}

/**
 * Constructs a new instance of the Calendar subclass appropriate for
 * the default Locale.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a Calendar subclass instance set to the current date and
 *				time in the default timezone
 */
public static synchronized Calendar getInstance() {
	return new GregorianCalendar();
}

/**
 * Constructs a new instance of the Calendar subclass appropriate for
 * the default Locale, using the specified TimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		timezone	the timezone to use
 * @return		a Calendar subclass instance set to the current date and
 *				time in the specified timezone
 */
public static synchronized Calendar getInstance(TimeZone timezone) {
	return new GregorianCalendar(timezone);
}

/**
 * Gets the time of this Calendar as a Date object.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a new Date initialized to the time of this Calendar
 *
 * @exception	IllegalArgumentException when the time is not set and the
 *				time cannot be computed from the current field values
 */
public final Date getTime () {
	return new Date(getTimeInMillis());
}

/**
 * Computes the time from the fields if required and answers the
 * time.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the time of this Calendar
 */
protected long getTimeInMillis () {
	if (!isTimeSet) {
		computeTime();
		isTimeSet = true;
	}
	return time;
}

/**
 * Gets the timezone of this Calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the timezone used by this Calendar
 */
public TimeZone getTimeZone() {
	return zone;
}

/**
 * Answers an integer hash code for the receiver. Objects which are
 * equal answer the same value for this method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	long time = getTimeInMillis();
	return getTimeZone().hashCode() + (int)(time >> 32) + (int)time;
}

/**
 * Sets a field to the specified value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		field	the Calendar field to modify
 * @param		value	the value
 */
public final void set(int field, int value) {
	if (field == HOUR || field == HOUR_OF_DAY || fields[field] != value) {
		fields[field] = value;
		areFieldsSet = false;
		if (field == HOUR || field == HOUR_OF_DAY) lastTimeFieldSet = field;
		isSet[field] = true;
		isTimeSet = false;
	}
}

/**
 * Sets the time of this Calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		date	a Date object
 */
public final void setTime (Date date) {
	setTimeInMillis(date.getTime());
}

/**
 * Sets the time of this Calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		milliseconds	the time as the number of milliseconds since Jan. 1, 1970
 */
protected void setTimeInMillis(long milliseconds) {
	time = milliseconds;
	isTimeSet = true;
	areFieldsSet = false;
	complete();
}

/**
 * Sets the timezone used by this Calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		timezone	a TimeZone
 */
public void setTimeZone(TimeZone timezone) {
	zone = timezone;
}

}
