package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * GregorianCalendar provides the conversion between Dates and integer calendar
 * fields, such as the month, year or minute, for the gregorian calendar. See
 * Calendar for the defined fields.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Calendar
 * @see			TimeZone
 * @see			SimpleTimeZone
 */
class GregorianCalendar extends Calendar {
	final static int
		BC = 0,
		AD = 1;

	private static final long gregorianCutover = -12219292800000l;
	private static final int changeYear = 1582;
	private static final int julianSkew = ((changeYear - 2000) / 400) + julianError() -
		((changeYear - 2000) / 100);
	private int dst_offset;

	static byte[] DaysInMonth = new byte[] {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
	private static int[] DaysInYear = new int[] {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};

		private static final int CACHED_YEAR = 0;
		private static final int CACHED_MONTH = 1;
		private static final int CACHED_DATE = 2;
		private static final int CACHED_DAY_OF_WEEK = 3;
		private static final int CACHED_TZ_OFFSET = 4;
        transient private boolean isCached = false;
        transient private int cachedFields[];
        transient private long nextMidnightMillis = 0L;
        transient private long lastMidnightMillis = 0L;

/**
 * Constructs a new GregorianCalendar initialized to the current
 * date and time.
 *
 * @author		OTI
 * @version		initial
 */
public GregorianCalendar() {
	super();
	setTimeInMillis(System.currentTimeMillis());
}

GregorianCalendar(long milliseconds) {
	setTimeInMillis(milliseconds);
}

/**
 * Constructs a new GregorianCalendar initialized to the current
 * date and time and using the specified TimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		timezone	the TimeZone
 */
public GregorianCalendar(TimeZone timezone) {
	super();
	setTimeZone(timezone);
	setTimeInMillis(System.currentTimeMillis());
}

GregorianCalendar(boolean ignored) {
}

private final int fullFieldsCalc(long time, int orgMillis, int offset, boolean getOffset) {
	long days = time / 86400000;

	// Cannot add ZONE_OFFSET to time as it might overflow
	long zoneMillis = (long)orgMillis + offset;
	while (zoneMillis < 0) {
		zoneMillis += 86400000;
		days--;
	}
	while (zoneMillis >= 86400000) {
		zoneMillis -= 86400000;
		days++;
	}
	int millis = (int)zoneMillis;

	int dayOfYear = computeYearAndDay(days, time + offset);
	int month = dayOfYear / 32;
	boolean leapYear = isLeapYear(fields[YEAR]);
	int date = dayOfYear - daysInYear(leapYear, month);
	if (date > daysInMonth(leapYear, month)) {date -= daysInMonth(leapYear, month); month++;}
	fields[DAY_OF_WEEK] = mod7(days - 3) + 1;
	int tzOffset = offset;
	if (getOffset) {
		// when getOffset is true, offset is the zoneOffset
		tzOffset = fields[YEAR] <= 0 ?
		0 : getTimeZone().getOffset(AD, fields[YEAR], month, date, fields[DAY_OF_WEEK], millis);
		int dstOffset = tzOffset;
		if (fields[YEAR] > 0) dstOffset -= offset;
		dst_offset = dstOffset;
		if (dstOffset != 0) {
			long oldDays = days;
			millis += dstOffset;
			if (millis < 0) {
				millis += 86400000;
				days--;
			} else if (millis >= 86400000) {
				millis -= 86400000;
				days++;
			}
			if (oldDays != days) {
				dayOfYear = computeYearAndDay(days, time - offset + dstOffset);
				month = dayOfYear / 32;
				leapYear = isLeapYear(fields[YEAR]);
				date = dayOfYear - daysInYear(leapYear, month);
				if (date > daysInMonth(leapYear, month)) {date -= daysInMonth(leapYear, month); month++;}
				fields[DAY_OF_WEEK] = mod7(days - 3) + 1;
			}
		}
	} else {
		TimeZoneTable zone = (TimeZoneTable)getTimeZone();
		boolean inDaylight = zone.inDaylightTime(new Date(time));
		int dstOffset = inDaylight ? zone.getDSTSavings() : 0;
		dst_offset = dstOffset;
	}

	fields[MILLISECOND] = (int)(millis % 1000);
	millis /= 1000;
	fields[SECOND] = (int)(millis % 60);
	millis /= 60;
	fields[MINUTE] = (int)(millis % 60);
	millis /= 60;
	fields[HOUR_OF_DAY] = (int)(millis % 24);
	millis /= 24;
	fields[AM_PM] = fields[HOUR_OF_DAY] > 11 ? 1 : 0;
	fields[HOUR] = fields[HOUR_OF_DAY] % 12;

	if (fields[YEAR] <= 0)
		fields[YEAR] = -fields[YEAR] + 1;
	fields[MONTH] = month;
	fields[DATE] = date;
	return tzOffset;
}

private final void updateCachedFields() {
	fields[YEAR] 			= cachedFields[CACHED_YEAR];
	fields[MONTH] 			= cachedFields[CACHED_MONTH];
	fields[DATE] 			= cachedFields[CACHED_DATE];
	fields[DAY_OF_WEEK] 	= cachedFields[CACHED_DAY_OF_WEEK];
}

/**
 * Computes the Calendar fields from the time.
 */
protected void computeFields() {
	actualComputeFields();
	for (int i=0; i < FIELD_COUNT; i++) isSet[i] = true;
}

private void actualComputeFields() {
	if (cachedFields == null)
		cachedFields = new int[]{0, 0, 0, 0, 0, 0};

	int millis = (int)(time % 86400000);
	int savedMillis = millis;
	 int dstOffset = dst_offset;
	int offset;
	boolean checkDaylightSavings;
	if (getTimeZone() instanceof TimeZoneTable) {
		checkDaylightSavings = false;
		offset = ((TimeZoneTable)getTimeZone()).getOffset(time);
	} else {
		checkDaylightSavings = true;
		offset = getTimeZone().getRawOffset() + dstOffset;
	}
	// compute without a change in daylight saving time
	long newTime = time + (long)offset;

	if(time > 0L && newTime < 0L && offset > 0)
		newTime = 0x7fffffffffffffffL;
	else if(time < 0L && newTime > 0L && offset < 0)
		newTime = 0x8000000000000000L;

	while (isCached) {
		if (offset != cachedFields[CACHED_TZ_OFFSET] ||
			offset <= -86400000 || offset >= 86400000)
		{
			isCached = false;
			break;
		}

		if (millis < 0) {
			millis += 86400000;
		}

		// Cannot add ZONE_OFFSET to time as it might overflow
		millis += offset;

		if (millis < 0) {
			millis += 86400000;
		} else if (millis >= 86400000) {
			millis -= 86400000;
		}

		if (newTime >= nextMidnightMillis || newTime <= lastMidnightMillis) {
			isCached = false;
			break;
		}
		if (checkDaylightSavings) {
			if (getTimeZone().useDaylightTime()) {
				// getOffset() must be called with standard time
				int checkMillis = millis - dstOffset;
				if (checkMillis < 0) checkMillis += 86400000;
				int checkOffset = cachedFields[CACHED_YEAR] <= 0 ? 0 :
					getTimeZone().getOffset(AD, cachedFields[CACHED_YEAR], cachedFields[CACHED_MONTH],
							cachedFields[CACHED_DATE], cachedFields[CACHED_DAY_OF_WEEK], checkMillis);
				if (checkOffset != offset) {
					isCached = false;
					break;
				}
			} else {
				if (dstOffset != 0) {
					isCached = false;
					break;
				}
			}
		}
		updateCachedFields();
		fields[MILLISECOND] = (int)(millis % 1000);
		millis /= 1000;
		fields[SECOND] = (int)(millis % 60);
		millis /= 60;
		fields[MINUTE] = (int)(millis % 60);
		millis /= 60;
		fields[HOUR_OF_DAY] = (int)(millis % 24);
		millis /= 24;
		fields[AM_PM] = fields[HOUR_OF_DAY] > 11 ? 1 : 0;
		fields[HOUR] = fields[HOUR_OF_DAY] % 12;
		break;
	}
	if (!isCached) {
		int newOffset = fullFieldsCalc(time, savedMillis, checkDaylightSavings ? getTimeZone().getRawOffset() : offset, checkDaylightSavings);
		if (checkDaylightSavings && newOffset != offset) {
			newTime = time + (long)offset;
			if(time > 0L && newTime < 0L && offset > 0)
				newTime = 0x7fffffffffffffffL;
			else if(time < 0L && newTime > 0L && offset < 0)
				newTime = 0x8000000000000000L;
		}
		offset = newOffset;
	}

	//Caching
	if(!isCached
		&& newTime != 0x7fffffffffffffffL
		&& newTime != 0x8000000000000000L
		&& (!getTimeZone().useDaylightTime() || getTimeZone() instanceof SimpleTimeZone ||
				getTimeZone() instanceof TimeZoneTable))
	{
		int cacheMillis = 0;

		cachedFields[CACHED_YEAR] = fields[YEAR];
		cachedFields[CACHED_MONTH] = fields[MONTH];
		cachedFields[CACHED_DATE] = fields[DATE];
		cachedFields[CACHED_DAY_OF_WEEK] = fields[DAY_OF_WEEK];
		cachedFields[CACHED_TZ_OFFSET] = offset;

		cacheMillis += (23 - fields[HOUR_OF_DAY]) * 60 * 60 * 1000;
		cacheMillis += (59 - fields[MINUTE]) * 60 * 1000;
		cacheMillis += (59 - fields[SECOND]) * 1000;
		nextMidnightMillis = newTime + (long)cacheMillis;

		cacheMillis =  fields[HOUR_OF_DAY] * 60 * 60 * 1000;
		cacheMillis += fields[MINUTE] * 60 * 1000;
		cacheMillis += fields[SECOND] * 1000;
		lastMidnightMillis = newTime - (long)cacheMillis;

		isCached = true;
	}
}

/**
 * Computes the time from the Calendar fields.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception	IllegalArgumentException when the time cannot be computed
 *				from the current field values
 */
protected void computeTime() {
		if (isSet[MONTH] && (fields[MONTH] < 0 || fields[MONTH] > 11))
			throw new IllegalArgumentException();

	long time;
	int hour = 0;
	if (isSet[HOUR_OF_DAY] && lastTimeFieldSet != HOUR)
		hour = fields[HOUR_OF_DAY];
	else if (isSet[HOUR])
		hour = (fields[AM_PM] * 12) + fields[HOUR];
	time = hour * 3600000;

	if (isSet[MINUTE]) time += fields[MINUTE] * 60000;
	if (isSet[SECOND]) time += fields[SECOND] * 1000;
	if (isSet[MILLISECOND]) time += fields[MILLISECOND];

	long days;
	int year = isSet[YEAR] ? fields[YEAR] : 1970;
	lastTimeFieldSet = 0;
	time %= 86400000;
		int month = isSet[MONTH] ? fields[MONTH] : 0;
		boolean leapYear = isLeapYear(year);
		days = daysFromBaseYear(year) + daysInYear(leapYear, month);
		if (isSet[DATE]) {
			if (fields[DATE] < 1 || fields[DATE] > daysInMonth(leapYear, month))
				throw new IllegalArgumentException();
			days += fields[DATE] - 1;
		}

	time += days * 86400000;
	// Use local time to compare with the gregorian change
	if (year == changeYear && time >= gregorianCutover + julianError() * 86400000)
		time -= julianError() * 86400000;
	int offset1 = getOffset(time);
	time -= offset1;
	this.time = time;
	if (!areFieldsSet) {
		actualComputeFields();
		areFieldsSet = true;
	}
}

private int computeYearAndDay(long dayCount, long localTime) {
	int year = 1970;
	long days = dayCount;
	if (localTime < gregorianCutover) days -= julianSkew;
	int approxYears;

	while ((approxYears = (int)(days / 365)) != 0) {
		year = year + approxYears;
		days = dayCount - daysFromBaseYear(year);
	}
	if (days < 0) {
		year = year - 1;
		days = days + 365 + (isLeapYear(year) ? 1 : 0);
		if (year == changeYear && localTime < gregorianCutover)
			days -= julianError();
	}
	fields[YEAR] = year;
	return (int) days + 1;
}

private long daysFromBaseYear(int year) {
	if (year >= 1970) {
		long days = (year - 1970) * (long)365 + ((year - 1969) / 4);
		if (year > changeYear)
			days -= ((year - 1901) / 100) - ((year - 1601) / 400);
		else days += julianSkew;
		return days;
	} else if (year <= changeYear) {
		return (year - 1970) * (long)365 + ((year - 1972) / 4) + julianSkew;
	}
	return (year - 1970) * (long)365 + ((year - 1972) / 4) -
		((year - 2000) / 100) + ((year - 2000) / 400);

}

private int daysInMonth(boolean leapYear, int month) {
	if (leapYear && month == FEBRUARY) return DaysInMonth[month] + 1;
	else return DaysInMonth[month];
}

private int daysInYear(boolean leapYear, int month) {
	if (leapYear && month > FEBRUARY) return DaysInYear[month] + 1;
	else return DaysInYear[month];
}

int getOffset(long localTime) {
	TimeZone timeZone = getTimeZone();
	if (!(timeZone instanceof TimeZoneTable) && !timeZone.useDaylightTime())
		return timeZone.getRawOffset();

	long dayCount = localTime / 86400000;
	int millis = (int)(localTime % 86400000);
	if (millis < 0) {
		millis += 86400000;
		dayCount--;
	}

	int year = 1970;
	long days = dayCount;
	if (localTime < gregorianCutover) days -= julianSkew;
	int approxYears;

	while ((approxYears = (int)(days / 365)) != 0) {
		year = year + approxYears;
		days = dayCount - daysFromBaseYear(year);
	}
	if (days < 0) {
		year = year - 1;
		days = days + 365 + (isLeapYear(year) ? 1 : 0);
		if (year == changeYear && localTime < gregorianCutover)
			days -= julianError();
	}
	if (year <= 0) return timeZone.getRawOffset();
	int dayOfYear = (int) days + 1;

	int month = dayOfYear / 32;
	boolean leapYear = isLeapYear(year);
	int date = dayOfYear - daysInYear(leapYear, month);
	if (date > daysInMonth(leapYear, month)) {date -= daysInMonth(leapYear, month); month++;}
	int dayOfWeek = mod7(dayCount - 3) + 1;
	int offset = timeZone.getOffset(AD, year, month, date, dayOfWeek, millis);
	return offset;
}

/**
 * Answers if the specified year is a leap year.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		year	the year
 * @return		true if the specified year is a leap year, false otherwise
 */
boolean isLeapYear(int year) {
	if (year > changeYear)
		return year % 4 == 0 && (year % 100 != 0 || year % 400 == 0);
	else
		return year % 4 == 0;
}

private static int julianError() {
	return changeYear / 100 - changeYear / 400 - 2;
}

private int mod7(long num1) {
	int rem = (int)(num1 % 7);
	if (num1 < 0 && rem < 0) return rem + 7;
	return rem;
}

/**
 * Computes the time from the fields if required and answers the
 * time.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the time of this Calendar
 */

/**
 * Sets the time of this Calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		milliseconds	the time as the number of milliseconds since Jan. 1, 1970
 */

}
