package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Hashtable associates keys with values. Keys and values cannot be null.
 * The size of the Hashtable is the number of key/value pairs it contains.
 * The capacity is the number of key/value pairs the Hashtable can hold.
 * The load factor is a float value which determines how full the Hashtable
 * gets before expanding the capacity. If the load factor of the Hashtable
 * is exceeded, the capacity is doubled.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Enumeration
 * @see			java.io.Serializable
 * @see			java.lang.Object#equals
 * @see			java.lang.Object#hashCode
 */

public class Hashtable {
	transient int elementCount;
	transient Entry[] elementData;
	private int threshold;
	transient int firstSlot = 0;
	transient int lastSlot = -1;
	private static final Enumeration emptyEnumerator = new Hashtable(0).getEmptyEnumerator();

private static Entry newEntry(Object key, Object value, int hash) {
	return new Entry(key, value);
}

private static final class Entry {
	private Object key, value;
	Entry next;

	Entry(Object theKey, Object theValue) {
		key = theKey;
		value = theValue;
	}

	public int getKeyHash() {
		return key.hashCode();
	}

	public boolean equalsKey(Object aKey, int hash) {
		return key.equals(aKey);
	}

	public String toString() {
		return key + "=" + value;
	}
}

private final class HashEnumerator implements Enumeration {
	boolean key;
	int start;
	Entry entry;
HashEnumerator(boolean isKey) {
	key = isKey;
	start = lastSlot + 1;
}
public boolean hasMoreElements() {
	if (entry != null) return true;
	while (--start >= firstSlot)
		if (elementData[start] != null) {
			entry = elementData[start];
			return true;
		}
	return false;
}
public Object nextElement() {
	if (hasMoreElements()) {
		Object result = key ? entry.key : entry.value;
		entry = entry.next;
		return result;
	} else throw new NoSuchElementException();
}
}

/**
 * Constructs a new Hashtable using the default capacity
 * and load factor.
 *
 * @author		OTI
 * @version		initial
 */
public Hashtable() {
	this(11);
}
/**
 * Constructs a new Hashtable using the specified capacity
 * and the default load factor.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		capacity	the initial capacity
 */
public Hashtable(int capacity) {
	if (capacity >= 0) {
		elementCount = 0;
		elementData = new Entry[capacity == 0 ? 1 : capacity];
		firstSlot = elementData.length;
		computeMaxSize();
	} else throw new IllegalArgumentException();
}

private HashEnumerator getEmptyEnumerator() {
	return new HashEnumerator(false);
}

/**
 * Removes all key/value pairs from this Hashtable, leaving the size zero
 * and the capacity unchanged.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			#isEmpty
 * @see			#size
 */
public synchronized void clear() {
	elementCount = 0;
	for (int i = elementData.length; --i >= 0;)
		elementData[i] = null;
}

private void computeMaxSize() {
	threshold = elementData.length / 4 * 3 + ((elementData.length % 4) * 3 / 4);
}
/**
 * Answers if this Hashtable contains the specified object as the value
 * of at least one of the key/value pairs.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value	the object to look for as a value in this Hashtable
 * @return		true if object is a value in this Hashtable, false otherwise
 *
 * @see			#containsKey
 * @see			java.lang.Object#equals
 */
public synchronized boolean contains(Object value) {
	if (value != null) {
		for (int i=elementData.length; --i >= 0;) {
			Entry entry = elementData[i];
			while (entry != null) {
				if (value.equals(entry.value)) return true;
				entry = entry.next;
			}
		}
		return false;
	} else throw new NullPointerException();
}
/**
 * Answers if this Hashtable contains the specified object as a key
 * of one of the key/value pairs.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		key	the object to look for as a key in this Hashtable
 * @return		true if object is a key in this Hashtable, false otherwise
 *
 * @see			#contains
 * @see			java.lang.Object#equals
 */
public synchronized boolean containsKey(Object key) {
	return getEntry(key) != null;
}

/**
 * Answers an Enumeration on the values of this Hashtable. The
 * results of the Enumeration may be affected if the contents
 * of this Hashtable are modified.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		an Enumeration of the values of this Hashtable
 *
 * @see			#keys
 * @see			#size
 * @see			Enumeration
 */
public synchronized Enumeration elements() {
	if (elementCount == 0) return emptyEnumerator;
	return new HashEnumerator(false);
}

/**
 * Answers the value associated with the specified key in
 * this Hashtable.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		key	the key of the value returned
 * @return		the value associated with the specified key, null if the specified key
 *				does not exist
 *
 * @see			#put
 */
public synchronized Object get(Object key) {
	int hash = key.hashCode();
	int index = (hash & 0x7FFFFFFF) % elementData.length;
	Entry entry = elementData[index];
	while (entry != null) {
		if (entry.equalsKey(key, hash)) return entry.value;
		entry = entry.next;
	}
	return null;
}

Entry getEntry(Object key) {
	int hash = key.hashCode();
	int index = (hash & 0x7FFFFFFF) % elementData.length;
	Entry entry = elementData[index];
	while (entry != null) {
		if (entry.equalsKey(key, hash)) return entry;
		entry = entry.next;
	}
	return null;
}

/**
 * Answers if this Hashtable has no key/value pairs, a size of zero.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		true if this Hashtable has no key/value pairs, false otherwise
 *
 * @see			#size
 */
public synchronized boolean isEmpty() {
	return elementCount == 0;
}

/**
 * Answers an Enumeration on the keys of this Hashtable. The
 * results of the Enumeration may be affected if the contents
 * of this Hashtable are modified.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		an Enumeration of the keys of this Hashtable
 *
 * @see			#elements
 * @see			#size
 * @see			Enumeration
 */
public synchronized Enumeration keys() {
	if (elementCount == 0) return emptyEnumerator;
	return new HashEnumerator(true);
}

/**
 * Associate the specified value with the specified key in this Hashtable.
 * If the key already exists, the old value is replaced. The key and value
 * cannot be null.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		key	the key to add
 * @param		value	the value to add
 * @return		the old value associated with the specified key, null if the key did
 *				not exist
 *
 * @see			#elements
 * @see			#get
 * @see			#keys
 * @see			java.lang.Object#equals
 */
public synchronized Object put(Object key, Object value) {
	if (key != null && value != null) {
		int hash = key.hashCode();
		int index = (hash & 0x7FFFFFFF) % elementData.length;
		Entry entry = elementData[index];
		while (entry != null && !entry.equalsKey(key, hash))
			entry = entry.next;
		if (entry == null) {
			if (++elementCount > threshold) {
				rehash();
				index = (hash & 0x7FFFFFFF) % elementData.length;
			}
			if (index < firstSlot) firstSlot = index;
			if (index > lastSlot) lastSlot = index;
			entry = newEntry(key, value, hash);
			entry.next = elementData[index];
			elementData[index] = entry;
			return null;
		}
		Object result = entry.value;
		entry.value = value;
		return result;
	} else throw new NullPointerException();
}

/**
 * Increases the capacity of this Hashtable. This method is sent when
 * the size of this Hashtable exceeds the load factor.
 *
 * @author		OTI
 * @version		initial
 */
protected void rehash() {
	int length = (elementData.length<<1)+1;
	if (length == 0) length = 1;
	int newFirst = length;
	int newLast = -1;
	Entry[] newData = new Entry[length];
	for (int i=lastSlot+1; --i >= firstSlot;) {
		Entry entry = elementData[i];
		while (entry != null) {
			int index = (entry.getKeyHash() & 0x7FFFFFFF) % length;
			if(index < newFirst) newFirst = index;
			if(index > newLast) newLast = index;
			Entry next = entry.next;
			entry.next = newData[index];
			newData[index] = entry;
			entry = next;
		}
	}
	firstSlot = newFirst;
	lastSlot = newLast;
	elementData = newData;
	computeMaxSize();
}
/**
 * Remove the key/value pair with the specified key from this Hashtable.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		key	the key to remove
 * @return		the value associated with the specified key, null if the specified key
 *				did not exist
 *
 * @see			#get
 * @see			#put
 */
public synchronized Object remove(Object key) {
	int hash = key.hashCode();
	int index = (hash & 0x7FFFFFFF) % elementData.length;
	Entry last = null;
	Entry entry = elementData[index];
	while (entry != null && !entry.equalsKey(key, hash)) {
		last = entry;
		entry = entry.next;
	}
	if (entry != null) {
		if (last == null) elementData[index] = entry.next;
		else last.next = entry.next;
		elementCount--;
		Object result = entry.value;
		entry.value = null;
		return result;
	}
	return null;
}
/**
 * Answers the number of key/value pairs in this Hashtable.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the number of key/value pairs in this Hashtable
 *
 * @see			#elements
 * @see			#keys
 */
public synchronized int size() {
	return elementCount;
}

/**
 * Answers the string representation of this Hashtable.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the string representation of this Hashtable
 */
public synchronized String toString() {
	if (isEmpty()) return "{}";

	StringBuffer buffer = new StringBuffer(size() * 28);
	buffer.append('{');
	for (int i=lastSlot; i >= firstSlot; i--) {
		Entry entry = elementData[i];
		while (entry != null) {
			if (entry.key != this) {
				buffer.append(entry.key);
			} else {
				buffer.append("(this Map)");
			}
			buffer.append('=');
			if (entry.value != this) {
				buffer.append(entry.value);
			} else {
				buffer.append("(this Map)");
			}
			buffer.append(", ");
			entry = entry.next;
		}
	}
	// Remove the last ", "
	if (elementCount > 0) buffer.setLength(buffer.length() - 2);
	buffer.append('}');
	return buffer.toString();
}

}
