package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * SimpleTimeZone represents a local time zone and its daylight savings time
 * rules for the gregorian calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Calendar
 * @see			TimeZone
 */

class SimpleTimeZone extends TimeZone {
	private String ID;
	private int rawOffset;
	private int startYear, startMonth, startDay, startDayOfWeek, startTime;
	private int endMonth, endDay, endDayOfWeek, endTime;
	private int startMode, endMode;
	private int startTimeMode, endTimeMode;
	private static final int
		DOM_MODE = 1,
		DOW_IN_MONTH_MODE = 2,
		DOW_GE_DOM_MODE = 3,
		DOW_LE_DOM_MODE = 4;

	/* Constant for representing start or end time in GMT time mode. */
	static final int UTC_TIME = 2;

	/* Constant for representing start or end time
	 * in standard local time mode,
	 * based on timezone's raw offset from GMT, does not include Daylight savings. */
	static final int STANDARD_TIME = 1;

	/* Constant for representing start or end time in
	 * local wall clock time mode,
	 * based on timezone's adjusted offset from GMT, it does include Daylight savings. */
	static final int WALL_TIME = 0;

	private boolean useDaylight;
	private GregorianCalendar daylightSavings;
	private int dstSavings = 3600000;

/**
 * Constructs a new SimpleTimeZone using the specified offset
 * for standard time from GMT and the specified time zone ID.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		offset	the offset from GMT of standard time in milliseconds
 * @param		name	the time zone ID
 */
public SimpleTimeZone(int offset, String name) {
	ID = name;
	rawOffset = offset;
}
/**
 * Constructs a new SimpleTimeZone using the specified offset
 * for standard time from GMT, the specified time zone ID and the rules
 * for daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		offset	the offset from GMT of standard time in milliseconds
 * @param		name	the time zone ID
 * @param		startMonth	the Calendar month in which daylight savings time starts
 * @param		startDay	the occurrence of the day of the week on which daylight savings time starts
 * @param		startDayOfWeek	the Calendar day of the week on which daylight savings time starts
 * @param		startTime	the time of day in milliseconds on which daylight savings time starts
 * @param		endMonth	the Calendar month in which daylight savings time ends
 * @param		endDay	the occurrence of the day of the week on which daylight savings time ends
 * @param		endDayOfWeek	the Calendar day of the week on which daylight savings time ends
 * @param		endTime	the time of day in milliseconds standard time on which daylight savings time ends
 */
public SimpleTimeZone(int offset, String name, int startMonth, int startDay,
	int startDayOfWeek, int startTime, int endMonth, int endDay, int endDayOfWeek,
	int endTime)
{
	this(offset, name, startMonth, startDay, startDayOfWeek, startTime,
		endMonth, endDay, endDayOfWeek, endTime, 3600000);
}
/**
 * Constructs a new SimpleTimeZone using the specified offset
 * for standard time from GMT, the specified time zone ID and the rules
 * for daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		offset	the offset from GMT of standard time in milliseconds
 * @param		name	the time zone ID
 * @param		startMonth	the Calendar month in which daylight savings time starts
 * @param		startDay	the occurrence of the day of the week on which daylight savings time starts
 * @param		startDayOfWeek	the Calendar day of the week on which daylight savings time starts
 * @param		startTime	the time of day in milliseconds on which daylight savings time starts
 * @param		endMonth	the Calendar month in which daylight savings time ends
 * @param		endDay	the occurrence of the day of the week on which daylight savings time ends
 * @param		endDayOfWeek	the Calendar day of the week on which daylight savings time ends
 * @param		endTime	the time of day in milliseconds standard time on which daylight savings time ends
 * @param		daylightSavings	the daylight savings time difference in milliseconds
 */
public SimpleTimeZone(int offset, String name, int startMonth, int startDay,
	int startDayOfWeek, int startTime, int endMonth, int endDay, int endDayOfWeek,
	int endTime, int daylightSavings)
{
	this(offset, name);
	if (daylightSavings <= 0)
		// K00e9 = DST offset: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e9", daylightSavings));
	dstSavings = daylightSavings;

	setStartRule(startMonth, startDay, startDayOfWeek, startTime);
	setEndRule(endMonth, endDay, endDayOfWeek, endTime);
}

/**
 * Constructs a new SimpleTimeZone using the specified offset
 * for standard time from GMT, the specified time zone ID, the rules
 * for daylight savings time, and the modes indicating UTC, standard, or wall time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		offset	the offset from GMT of standard time in milliseconds
 * @param		name	the time zone ID
 * @param		startMonth	the Calendar month in which daylight savings time starts
 * @param		startDay	the occurrence of the day of the week on which daylight savings time starts
 * @param		startDayOfWeek	the Calendar day of the week on which daylight savings time starts
 * @param		startTime	the time of day in milliseconds on which daylight savings time starts
 * @param		startTimeMode the mode (UTC, standard, or wall time) of the start time value
 * @param		endMonth	the Calendar month in which daylight savings time ends
 * @param		endDay	the occurrence of the day of the week on which daylight savings time ends
 * @param		endDayOfWeek	the Calendar day of the week on which daylight savings time ends
 * @param		endTime	the time of day in milliseconds standard time on which daylight savings time ends
 * @param		endTimeMode the mode (UTC, standard, or wall time) of the end time value
 * @param		daylightSavings	the daylight savings time difference in milliseconds
 */
SimpleTimeZone(int offset, String name, int startMonth,
		int startDay, int startDayOfWeek, int startTime, int startTimeMode,
		int endMonth, int endDay, int endDayOfWeek, int endTime,
		int endTimeMode, int daylightSavings)
{
	this(offset, name);
	if (daylightSavings <= 0)
		// K00e9 = DST offset: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e9", daylightSavings));
	dstSavings = daylightSavings;

	setStartRule(startMonth, startDay, startDayOfWeek, startTime);
	setEndRule(endMonth, endDay, endDayOfWeek, endTime);

	if (startTimeMode > UTC_TIME || startTimeMode < WALL_TIME) {
		// K03c5 = time mode: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K03c5", startTimeMode));
	}
	this.startTimeMode = startTimeMode;

	if (endTimeMode > UTC_TIME || endTimeMode < WALL_TIME) {
		// K03c5 = time mode: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K03c5", endTimeMode));
	}
	this.endTimeMode = endTimeMode;
}

/**
 * Gets the daylight savings offset in milliseconds for this SimpleTimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the daylight savings offset in milliseconds
 */
public int getDSTSavings() {
	return dstSavings;
}

/**
 * Gets the offset from GMT of this SimpleTimeZone for the specified date and time. The
 * offset includes daylight savings time if the specified date and time are within
 * the daylight savings time period.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		era	the GregorianCalendar era, either GregorianCalendar.BC or
 *					GregorianCalendar.AD
 * @param		year	the year
 * @param		month	the Calendar month
 * @param		day	the day of the month
 * @param		dayOfWeek	the Calendar day of the week
 * @param		time	the time of day in milliseconds
 * @return		the offset from GMT in milliseconds
 */
public int getOffset(int era, int year, int month, int day, int dayOfWeek, int time) {
	if (era != GregorianCalendar.BC && era != GregorianCalendar.AD)
		// K00ea = era: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00ea", era));
	checkRange(month, dayOfWeek, time);
	if (month != Calendar.FEBRUARY || day != 29 || !isLeapYear(year))
		checkDay(month, day);

	int previousMonth = month - 1;
	if (previousMonth < Calendar.JANUARY)
		previousMonth = Calendar.DECEMBER;

	if (!useDaylightTime() || era != GregorianCalendar.AD || year < startYear)
		return rawOffset;

	boolean afterStart = false, beforeEnd = false;

	int ruleDay = 0;
	int ruleTime = startTime;
	if (startTimeMode == UTC_TIME) ruleTime += rawOffset;
	int startRuleMonth = startMonth;
	int prevStartMonth = startMonth - 1;
	if (prevStartMonth < Calendar.JANUARY) prevStartMonth = Calendar.DECEMBER;
	if (month == startMonth || month == prevStartMonth || previousMonth == startMonth) {
		int daysInPrevStartMonth = GregorianCalendar.DaysInMonth[prevStartMonth];
		if (prevStartMonth == Calendar.FEBRUARY && isLeapYear(year)) daysInPrevStartMonth++;
		int daysInStartMonth = GregorianCalendar.DaysInMonth[startMonth];
		if (startMonth == Calendar.FEBRUARY && isLeapYear(year)) daysInStartMonth++;
		int firstDayOfMonth = mod7(dayOfWeek - day);
		if (month != startMonth) {
			if (month == prevStartMonth) {
				firstDayOfMonth = mod7(firstDayOfMonth + daysInPrevStartMonth);
			} else {
				firstDayOfMonth = mod7(firstDayOfMonth - daysInStartMonth);
			}
		}
		switch (startMode) {
			case DOM_MODE:
				ruleDay = startDay;
				break;
			case DOW_IN_MONTH_MODE:
				if (startDay >= 0) {
					ruleDay = mod7(startDayOfWeek - firstDayOfMonth) + 1 +
						(startDay - 1) * 7;
				} else {
					ruleDay = daysInStartMonth + 1 +
						mod7(startDayOfWeek - (firstDayOfMonth + daysInStartMonth)) +
						startDay * 7;
				}
				break;
			case DOW_GE_DOM_MODE:
				ruleDay = startDay +
					mod7(startDayOfWeek - (firstDayOfMonth + startDay - 1));
				break;
			case DOW_LE_DOM_MODE:
				ruleDay = startDay +
					mod7(startDayOfWeek - (firstDayOfMonth + startDay - 1));
				if (ruleDay != startDay) ruleDay -= 7;
				break;
		}
		if (ruleTime < 0) {
			ruleDay--;
			ruleTime += 86400000;
		}
		if (ruleTime >= 86400000) {
			ruleDay++;
			ruleTime -= 86400000;
		}
		// check > daysInStartMonth before < 1, since setting the ruleDay
		// to the daysInPrevStartMonth could be > daysInStartMonth
		if (ruleDay > daysInStartMonth) {
			if (++startRuleMonth > Calendar.DECEMBER)
				startRuleMonth = Calendar.JANUARY;
			ruleDay -= daysInStartMonth;
		}
		if (ruleDay < 1) {
			if (--startRuleMonth < Calendar.JANUARY)
				startRuleMonth = Calendar.DECEMBER;
			ruleDay += daysInPrevStartMonth;
		}
		if (month == startRuleMonth) {
			if (day > ruleDay || day == ruleDay && time >= ruleTime) {
				afterStart = true;
			}
		}
	}

	ruleTime = endTime;
	if (endTimeMode == WALL_TIME) ruleTime -= dstSavings;
	else if (endTimeMode == UTC_TIME) ruleTime += rawOffset;
	int endRuleMonth = endMonth;
	int prevEndMonth = endMonth - 1;
	if (prevEndMonth < Calendar.JANUARY) prevEndMonth = Calendar.DECEMBER;
	if (month == endMonth || month == prevEndMonth || previousMonth == endMonth) {
		int daysInPrevEndMonth = GregorianCalendar.DaysInMonth[prevEndMonth];
		if (prevEndMonth == Calendar.FEBRUARY && isLeapYear(year)) daysInPrevEndMonth++;
		int daysInEndMonth = GregorianCalendar.DaysInMonth[endMonth];
		if (endMonth == Calendar.FEBRUARY && isLeapYear(year)) daysInEndMonth++;
		int firstDayOfMonth = mod7(dayOfWeek - day);
		if (month != endMonth) {
			if (month == prevEndMonth) {
				firstDayOfMonth = mod7(firstDayOfMonth + daysInPrevEndMonth);
			} else {
				firstDayOfMonth = mod7(firstDayOfMonth - daysInEndMonth);
			}
		}
		switch (endMode) {
			case DOM_MODE:
				ruleDay = endDay;
				break;
			case DOW_IN_MONTH_MODE:
				if (endDay >= 0) {
					ruleDay = mod7(endDayOfWeek - firstDayOfMonth) + 1 +
						(endDay - 1) * 7;
				} else {
					ruleDay = daysInEndMonth + 1 +
						mod7(endDayOfWeek - (firstDayOfMonth + daysInEndMonth)) +
						endDay * 7;
				}
				break;
			case DOW_GE_DOM_MODE:
				ruleDay = endDay +
					mod7(endDayOfWeek - (firstDayOfMonth + endDay - 1));
				break;
			case DOW_LE_DOM_MODE:
				ruleDay = endDay +
					mod7(endDayOfWeek - (firstDayOfMonth + endDay - 1));
				if (ruleDay != endDay) ruleDay -= 7;
				break;
		}

		if (ruleTime < 0) {
			ruleDay--;
			ruleTime += 86400000;
		}
		if (ruleTime >= 86400000) {
			ruleDay++;
			ruleTime -= 86400000;
		}
		// check > daysInEndMonth before < 1, since setting the ruleDay
		// to the daysInPrevEndMonth could be > daysInEndMonth
		if (ruleDay > daysInEndMonth) {
			if (++endRuleMonth > Calendar.DECEMBER)
				endRuleMonth = Calendar.JANUARY;
			ruleDay -= daysInEndMonth;
		}
		if (ruleDay < 1) {
			if (--endRuleMonth < Calendar.JANUARY)
				endRuleMonth = Calendar.DECEMBER;
			ruleDay += daysInPrevEndMonth;
		}

		if (month == endRuleMonth) {
			if (day < ruleDay || day == ruleDay && time < ruleTime) {
				beforeEnd = true;
			}
		}
	}

	if (endRuleMonth >= startRuleMonth) {
		if ((afterStart || month > startRuleMonth) && (beforeEnd || month < endRuleMonth))
			return rawOffset + dstSavings;
	} else {
		if ((afterStart || month > startRuleMonth || month <= endRuleMonth) &&
				(beforeEnd || month < endRuleMonth || month >= startRuleMonth))
			return rawOffset + dstSavings;
	}
	return rawOffset;
}

/**
 * Gets the offset from GMT of this SimpleTimeZone for the specified date. The
 * offset includes daylight savings time if the specified date is within
 * the daylight savings time period.
 *
 * @param		time 	the date in milliseconds since January 1, 1970 00:00:00 GMT
 * @return		the offset from GMT in milliseconds
 */
int getOffset(long time) {
	if (!useDaylightTime()) return rawOffset;
	if (daylightSavings == null) {
		 GregorianCalendar cal = new GregorianCalendar(false);
		 cal.setTimeZone(this);
		 daylightSavings = cal;
	}
	return daylightSavings.getOffset(time + rawOffset) ;
}

/**
 * Gets the offset for standard time from GMT for this SimpleTimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the offset from GMT of standard time in milliseconds
 */
public int getRawOffset() {
	return rawOffset;
}

/**
 * Answers if the specified Date is in the daylight savings time period
 * for this SimpleTimeZone.
 *
 * @param		time	a Date
 * @return		true when the Date is in the daylight savings time period, false otherwise
 */
boolean inDaylightTime(Date time) {
	// check for null pointer
	long millis = time.getTime();
	if (!useDaylightTime()) return false;
	if (daylightSavings == null) {
		GregorianCalendar cal = new GregorianCalendar(false);
		cal.setTimeZone(this);
		daylightSavings = cal;
	}
	return daylightSavings.getOffset(millis + rawOffset) != rawOffset;
}

private boolean isLeapYear(int year) {
	if (year > 1582)
		return year % 4 == 0 && (year % 100 != 0 || year % 400 == 0);
	else
		return year % 4 == 0;
}
private int mod7(int num1) {
	int rem = num1 % 7;
	return (num1 < 0 && rem < 0) ? 7 + rem : rem;
}

private void checkRange(int month, int dayOfWeek, int time) {
	if (month < Calendar.JANUARY || month > Calendar.DECEMBER)
		// K00e5 = month: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e5", month));
	if (dayOfWeek < Calendar.SUNDAY || dayOfWeek > Calendar.SATURDAY)
		// K00e7 = day of week: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e7", dayOfWeek));
	if (time < 0 || time >= 24 * 3600000)
		// K00e8 = time: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e8", time));
}

private void checkDay(int month, int day) {
	if (day <= 0 || day > GregorianCalendar.DaysInMonth[month])
		// K00e6 = day of month: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00e6", day));
}

private void setEndMode() {
	if (endDayOfWeek == 0) endMode = DOM_MODE;
	else if (endDayOfWeek < 0) {
		endDayOfWeek = -endDayOfWeek;
		if (endDay < 0) {
			endDay = -endDay;
			endMode = DOW_LE_DOM_MODE;
		} else endMode = DOW_GE_DOM_MODE;
	} else endMode = DOW_IN_MONTH_MODE;
	useDaylight = startDay != 0 && endDay != 0;
	if (endDay != 0) {
		checkRange(endMonth, endMode == DOM_MODE ? 1 : endDayOfWeek, endTime);
		if (endMode != DOW_IN_MONTH_MODE)
			checkDay(endMonth, endDay);
		else {
			if (endDay < -5 || endDay > 5)
				// K00f8 = day of week in month: {0}
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00f8", endDay));
		}
	}
	if (endMode != DOM_MODE) endDayOfWeek--;
}

/**
 * Sets the rule which specifies the end of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time ends
 * @param		dayOfMonth	the Calendar day of the month on which daylight savings time ends
 * @param		time	the time of day in milliseconds standard time on which daylight savings time ends
 */
public void setEndRule(int month, int dayOfMonth, int time) {
	endMonth = month;
	endDay = dayOfMonth;
	endDayOfWeek = 0;	// Initialize this value for hasSameRules()
	endTime = time;
	endTimeMode = WALL_TIME;
	setEndMode();
}
/**
 * Sets the rule which specifies the end of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time ends
 * @param		day	the occurrence of the day of the week on which daylight savings time ends
 * @param		dayOfWeek	the Calendar day of the week on which daylight savings time ends
 * @param		time	the time of day in milliseconds standard time on which daylight savings time ends
 */
public void setEndRule(int month, int day, int dayOfWeek, int time) {
	endMonth = month;
	endDay = day;
	endDayOfWeek = dayOfWeek;
	endTime = time;
	endTimeMode = WALL_TIME;
	setEndMode();
}
/**
 * Sets the rule which specifies the end of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time ends
 * @param		day	the Calendar day of the month
 * @param		dayOfWeek	the Calendar day of the week on which daylight savings time ends
 * @param		time	the time of day in milliseconds on which daylight savings time ends
 * @param		after	selects the day after or before the day of month
 */
public void setEndRule(int month, int day, int dayOfWeek, int time, boolean after) {
	endMonth = month;
	endDay = after ? day : -day;
	endDayOfWeek = -dayOfWeek;
	endTime = time;
	endTimeMode = WALL_TIME;
	setEndMode();
}

private void setStartMode() {
	if (startDayOfWeek == 0) startMode = DOM_MODE;
	else if (startDayOfWeek < 0) {
		startDayOfWeek = -startDayOfWeek;
		if (startDay < 0) {
			startDay = -startDay;
			startMode = DOW_LE_DOM_MODE;
		} else startMode = DOW_GE_DOM_MODE;
	} else startMode = DOW_IN_MONTH_MODE;
	useDaylight = startDay != 0 && endDay != 0;
	if (startDay != 0) {
		checkRange(startMonth, startMode == DOM_MODE ? 1 : startDayOfWeek, startTime);
		if (startMode != DOW_IN_MONTH_MODE)
			checkDay(startMonth, startDay);
		else {
			if (startDay < -5 || startDay > 5)
				// K00f8 = day of week in month: {0}
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00f8", startDay));
		}
	}
	if (startMode != DOM_MODE) startDayOfWeek--;
}

/**
 * Sets the rule which specifies the start of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time starts
 * @param		dayOfMonth	the Calendar day of the month on which daylight savings time starts
 * @param		time	the time of day in milliseconds on which daylight savings time starts
 */
public void setStartRule(int month, int dayOfMonth, int time) {
	startMonth = month;
	startDay = dayOfMonth;
	startDayOfWeek = 0;	// Initialize this value for hasSameRules()
	startTime = time;
	startTimeMode = WALL_TIME;
	setStartMode();
}

/**
 * Sets the rule which specifies the start of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time starts
 * @param		day	the occurrence of the day of the week on which daylight savings time starts
 * @param		dayOfWeek	the Calendar day of the week on which daylight savings time starts
 * @param		time	the time of day in milliseconds on which daylight savings time starts
 */
public void setStartRule(int month, int day, int dayOfWeek, int time) {
	startMonth = month;
	startDay = day;
	startDayOfWeek = dayOfWeek;
	startTime = time;
	startTimeMode = WALL_TIME;
	setStartMode();
}
/**
 * Sets the rule which specifies the start of daylight savings time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		month	the Calendar month in which daylight savings time starts
 * @param		day	the Calendar day of the month
 * @param		dayOfWeek	the Calendar day of the week on which daylight savings time starts
 * @param		time	the time of day in milliseconds on which daylight savings time starts
 * @param		after	selects the day after or before the day of month
 */
public void setStartRule(int month, int day, int dayOfWeek, int time, boolean after) {
	startMonth = month;
	startDay = after ? day : -day;
	startDayOfWeek = -dayOfWeek;
	startTime = time;
	startTimeMode = WALL_TIME;
	setStartMode();
}

/**
 * Answers if this TimeZone has a daylight savings time period.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		true if this time zone has a daylight savings time period, false otherwise
 */
public boolean useDaylightTime() {
	return useDaylight;
}

/**
 * Gets the ID of this TimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the time zone ID string
 */
public String getID() {
	return ID;
}

}
