package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2006  All Rights Reserved
 */

class TimeZoneTable extends TimeZone {
	private int rawOffset;
	private SimpleTimeZone[] timezones;
	private int[][] dateOffsets;
	private long[] longOffsets;
	private String ID;

/**
 * Creates a TimeZoneTable with an ordered array of SimpleTimeZones.
 * The TimeZones must all have the same ID. The dateOffsets and longOffsets
 * are arrays of int[] and long[] which specify the localized date and
 * gmt date of the time to change between SimpleTimeZones.
 *
 * @param timezones a non-null array of SimpleTimeZone, ordered by start year
 * @param dateOffsets an int[] containing the localized date to change to the next SimpleTimeZone
 * @param longOffsets a long[] containing the gmt date to change to the next SimpleTimezone
 */
public TimeZoneTable(SimpleTimeZone[] timezones, int[][] dateOffsets, long[] longOffsets) {
	super();
	this.timezones = timezones;
	this.dateOffsets = dateOffsets;
	this.longOffsets = longOffsets;
	ID = timezones[0].getID();
	rawOffset = timezones[timezones.length-1].getRawOffset();
}

/**
 * Gets the offset from GMT of this TimeZone for the specified date and time. The
 * offset includes daylight savings time if the specified date and time are within
 * the daylight savings time period.
 *
 * @param		era	the GregorianCalendar era, either GregorianCalendar.BC or
 *					GregorianCalendar.AD
 * @param		year	the year
 * @param		month	the Calendar month
 * @param		day	the day of the month
 * @param		dayOfWeek	the Calendar day of the week
 * @param		time	the time of day in milliseconds
 * @return		the offset from GMT in milliseconds
 */
public int getOffset(int era, int year, int month, int day, int dayOfWeek, int time) {
	if (era == GregorianCalendar.AD) {
		for (int i=dateOffsets.length-1; i>=0; i--) {
			if (year < dateOffsets[i][0]) continue;
			if (year == dateOffsets[i][0]) {
				if (month < dateOffsets[i][1]) continue;
				if (month == dateOffsets[i][1]) {
					if (day < dateOffsets[i][2]) continue;
					if (day == dateOffsets[i][2]) {
						if (time < dateOffsets[i][3]) continue;
					}
				}
			}
			return timezones[i+1].getOffset(era, year, month, day, dayOfWeek, time);
		}
	}
	return timezones[0].getOffset(era, year, month, day, dayOfWeek, time);
}

/**
 * Answers if this TimeZone has a daylight savings time period.
 *
 * @return		true if this time zone has a daylight savings time period, false otherwise
 */
public boolean useDaylightTime() {
	return timezones[timezones.length-1].useDaylightTime();
}

/**
 * Gets the offset for standard time from GMT for this TimeZone.
 *
 * @return		the offset from GMT of standard time in milliseconds
 */
public int getRawOffset() {
	return rawOffset;
}

/**
 * Answers if the specified Date is in the daylight savings time period
 * for this TimeZone.
 *
 * @param		time	a Date
 * @return		true when the Date is in the daylight savings time period, false otherwise
 */
public boolean inDaylightTime(Date time) {
	// check for null pointer
	long millis = time.getTime();
	for (int i=longOffsets.length-1; i>=0; i--) {
		if (millis < longOffsets[i]) continue;
		return timezones[i+1].inDaylightTime(time);
	}
	return timezones[0].inDaylightTime(time);
}

/**
 * Gets the offset from GMT of this TimeZone for the specified date. The
 * offset includes daylight savings time if the specified date is within
 * the daylight savings time period.
 *
 * @param		time 	the date in milliseconds since January 1, 1970 00:00:00 GMT
 * @return		the offset from GMT in milliseconds
 */
public int getOffset(long time) {
	for (int i=longOffsets.length-1; i>=0; i--) {
		if (time < longOffsets[i]) continue;
		return timezones[i+1].getOffset(time);
	}
	return timezones[0].getOffset(time);
}

/**
 * Gets the ID of this TimeZone.
 *
 * @return		the time zone ID string
 */
public String getID() {
	return ID;
}

/**
 * Gets the daylight savings offset in milliseconds for this TimeZone.
 *
 * @return		the daylight savings offset in milliseconds
 */
public int getDSTSavings() {
	for (int i=longOffsets.length-1; i>=0; i--) {
		if (timezones[i].useDaylightTime())
			return timezones[i].getDSTSavings();
	}
	return 0;
}

}
