package javax.microedition.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

import java.io.*;
import com.ibm.oti.connection.*;

public class Connector {

	public static final int READ = 1;
	public static final int WRITE = 2;
	public static final int READ_WRITE = 3;

private Connector() {
}

public static Connection open(String spec) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return open(spec, READ_WRITE, false, scheme, connectionClass);
}

public static Connection open(String spec, int access) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return open(spec, access, false, scheme, connectionClass);
}

public static Connection open(String spec, int access, boolean timeout) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return open(spec, access, timeout, scheme, connectionClass);
}

public static DataInputStream openDataInputStream(String spec) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return new DataInputStream(openInputStream(spec, scheme, connectionClass));
}

public static DataOutputStream openDataOutputStream(String spec) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return new DataOutputStream(openOutputStream(spec, scheme, connectionClass));
}

public static InputStream openInputStream(String spec) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return openInputStream(spec, scheme, connectionClass);
}

private static InputStream openInputStream(String spec, String scheme, Class connectionClass) throws IOException {
	Connection connection = open(spec, READ, false, scheme, connectionClass);
	try {
		if (!(connection instanceof InputConnection))
			// K0004 = Does not support input stream
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0004"));
		return ((InputConnection)connection).openInputStream();
	} finally {connection.close();}
}

public static OutputStream openOutputStream(String spec) throws IOException {
	String scheme = getScheme(spec);
	Class connectionClass = findConnectionClass(scheme.toLowerCase());
	return openOutputStream(spec, scheme, connectionClass);
}

private static OutputStream openOutputStream(String spec, String scheme, Class connectionClass) throws IOException {
	Connection connection = open(spec, WRITE, false, scheme, connectionClass);
	try {
		if (!(connection instanceof OutputConnection))
			// K0005 = Does not support output stream
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0005"));
		return ((OutputConnection)connection).openOutputStream();
	} finally {connection.close();}
}

private static Connection open(String spec, int access, boolean timeout, String scheme, Class connectionClass) throws IOException {
	if (access != READ && access != WRITE && access != READ_WRITE)
		// K0198 = Invalid access mode
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0198"));

	CreateConnection connection = null;
	try {
		connection = (CreateConnection)connectionClass.newInstance();
	} catch (InstantiationException e) {
		// K0003 = Cannot instantiate scheme: {0}
		throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K0003", connectionClass));
	} catch (IllegalAccessException e) {
		// K0003 = Cannot instantiate scheme: {0}
		throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K0003", connectionClass));
	}

	spec = spec.substring(scheme.length()+1);
	return connection.setParameters2(spec, access, timeout);
}

private static String getScheme(String spec) throws IllegalArgumentException {
	// K0067 = String is null
	if (spec == null) throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0067"));
	int index = spec.indexOf(':');
	// K0001 = Cannot parse scheme: {0}
	if (index == -1) throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0001", spec));
	String scheme = spec.substring(0, index);
	if (scheme.length() == 0)
		// K0001 = Cannot parse scheme: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0001", spec));
	return scheme;
}

/**
 * Finds the connection class for the specified scheme.
 *
 * @author		OTI
 * @version		initial
 *
 * @param	scheme the scheme to find
 * @return	the handler class for the scheme, or null
 *
 */
private static Class findConnectionClass(String scheme) throws ConnectionNotFoundException {
	// Check if there is a list of packages which can provide handlers.
	// If so, then walk this list looking for an applicable one.
	String packageList = System.getProperty("microedition.connection.pkgs");
	if (packageList != null) {
		int last = 0, index = packageList.indexOf('|'), end = packageList.length();
		while (last < end) {
			if (index == -1) index = end;
			String className = packageList.substring(last, index) + "." + scheme + ".Connection";
			try {
				return (Class)Class.forName(className);
			} catch (Exception e) {}
			last = index + 1;
			index = packageList.indexOf('|', last);
		}
	}

	// No one else has provided a handler, so try our internal one.
	try {
		String className = "com.ibm.oti.connection." + scheme + ".Connection";
		return (Class)Class.forName(className);
	} catch (ClassNotFoundException e) {
		if (scheme.equals("file")) {
			try {
				String className = "com.ibm.oti.connection.file.base.Connection";
				return (Class)Class.forName(className);
			} catch (ClassNotFoundException e2) {}
		}
	}
	// K0002 = Scheme not found: {0}
	throw new ConnectionNotFoundException(com.ibm.oti.util.Msg.getString("K0002", scheme));
}

}
