package javax.microedition.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import java.io.IOException;

/**
 * The interface for socket connections, allowing socket settings to be read/modified
 * and allowing the local and remote addresses and ports to be retrieved.
 *
 * @author OTI
 * @since MIDP 2.0
 * @version initial
 *
 */
public interface SocketConnection extends StreamConnection {

	/**
	 * The socket option constant for enabling the Nagle algorithm.
	 * A non-zero value enables it.
	 */
	public static final byte DELAY = 0;

	/**
	 * The socket option constant for enabling keepalive.
	 * A non-zero value enables it.
	 */
	public static final byte KEEPALIVE = 2;

	/**
	 * The socket option for the linger time in seconds.
	 * Changes how long the connection waits before closing when there is output pending.
	 */
	public static final byte LINGER = 1;

	/**
	 * The socket option for the size of receiving buffer.
	 */
	public static final byte RCVBUF = 3;

	/**
	 * The socket option for the size of sending buffer.
	 */
	public static final byte SNDBUF = 4;

	/**
	 * Answers the address of the remote host that the connection accesses.
	 * @return String	the address of the remote host
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public String getAddress() throws IOException;

	/**
	 * Answers the local address for the connection.
	 * @return String	the local address
	 * @throws IOException	if the connection has already been closed
	 *

	 */
	public String getLocalAddress() throws IOException;

	/**
	 * Answers the number of the local port that the connection is on.
	 *
	 * @return int	the local port number
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getLocalPort() throws IOException;

	/**
	 * Answers the port number of the connection on the remote host.
	 *
	 * @return int	the port number
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getPort() throws IOException;

	/**
	 * Answers the value of the selected option.  The option passed to the function
	 * must be one of the constants DELAY, KEEPALIVE, LINGER, RCVBUF, SNDBUF.
	 *
	 * @param optname	the option to return
	 * @return int	the option value to retrieve
	 * @throws IllegalArgumentException	if the option requested is not one of the legal constants
	 * @throws IOException	if the connection has already been closed
	 *
	 */
	public int getSocketOption(byte optname) throws IllegalArgumentException, IOException;

	/**
	 * Sets the value of the selected option.  The choice of option passed to the function
	 * must be one of the constants DELAY, KEEPALIVE, LINGER, RCVBUF, SNDBUF.
	 *
	 * @param optname	the option to return
	 * @param optval	the value to set
	 *
	 * @throws IllegalArgumentException	if the new option value is invalid
	 * or the option is not one of the legal constants
	 * @throws IOException	if the connection has already been closed
	 *
	 *
	 */
	public void setSocketOption(byte optname, int optval) throws IllegalArgumentException, IOException;

}
