package javax.microedition.lcdui;

import com.ibm.ive.midp.*;

/*
 * Licensed Materials - Property of IBM.
 * (c) Copyright IBM Corp 2000, 2006.
 *
 */

public class Alert extends Screen {

	public static final int FOREVER = -2;

	private static final int DEFAULT_TIMEOUT = FOREVER;

	// used when fImage is mutable
	private Image fSnapshot;

	private static CommandListener gDefaultListener = new DefaultCommandListener();

	Image fImage;
	private String fText;
	AlertType fType;
	private int fTimeout;
	private Alarm fAlarm;
	Item fItems[];
	int fItemsLength;
	ImageItem fImageItem;
	StringItem fStringItem;
	Gauge fIndicator;
	private int fIndicatorPosition;

	/*
	 * There are three independent factors which affect the
	 * modality of Alerts.
	 *
	 * (1) the timeout
	 * (2) the length of the content at display time
	 * (3) the presence of 2 or more application-defined commands
	 *
	 * By default, Alerts are modal due to the fact that their
	 * default timeout value is FOREVER.
	 */
	private boolean fCommandModal;

	private boolean fHasDismissCommand;

	public static final Command DISMISS_COMMAND = new Command(AlertPeer.getDismissLabel(), Command.OK, 0);

	public Alert(String title) {
		this(title, null, null, null);
	}

	public Alert(String title, String text, Image image, AlertType type) {
		super(title);

		fImage = image;
		fText = text;
		fType = type;
		fItems = new Item[3];
		fIndicator = null;

		if (image != null && image.isMutable()) {
			fSnapshot = Image.createImage(image);
			fImageItem = new ImageItem(null, fSnapshot, 0, null);
		} else {
			fImageItem = new ImageItem(null, image, 0, null);
		}

		fImageItem.setLayout(Item.LAYOUT_LEFT | Item.LAYOUT_SHRINK | Item.LAYOUT_2);

		fStringItem = new StringItem(null, fText);
		fStringItem.setLayout(Item.LAYOUT_VCENTER | Item.LAYOUT_LEFT | Item.LAYOUT_EXPAND | Item.LAYOUT_2);

		addItem(fImageItem);
		addItem(fStringItem);

		fIndicatorPosition = fItemsLength;

		super.addCommand(DISMISS_COMMAND);
		super.setCommandListener(gDefaultListener);

		// Alert is modal by default
		_setTimeout(DEFAULT_TIMEOUT);
		fHasDismissCommand = true;
	}

	public void addCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			if (command == null) throw new NullPointerException();
			if (command == DISMISS_COMMAND) return;

			if (fHasDismissCommand) {
				fHasDismissCommand = false;
				super.removeCommand(DISMISS_COMMAND);
			}

			super.addCommand(command);

			if (numApplicationCommands() == 2) {
				fCommandModal = true;
				cancelAlarm();
			}
		}
	}

	void cancelAlarm() {
		if (fAlarm != null) fAlarm.cancel();
	}

	public int getDefaultTimeout() {
		return DEFAULT_TIMEOUT;
	}

	public Image getImage() {
		synchronized (Device.gDisplayableLock) {
			return fImage;
		}
	}

	public String getString() {
		synchronized (Device.gDisplayableLock) {
			return fText;
		}
	}

	public int getTimeout() {
		synchronized (Device.gDisplayableLock) {
			return fTimeout;
		}
	}

	public AlertType getType() {
		synchronized (Device.gDisplayableLock) {
			return fType;
		}
	}

	public void setCommandListener(CommandListener listener) {
		synchronized (Device.gDisplayableLock) {
			if (listener == null) {
				// restore default listener
				super.setCommandListener(gDefaultListener);
			} else {
				super.setCommandListener(listener);
			}
		}
	}

	public void setImage(Image image) {
		synchronized (Device.gDisplayableLock) {
			// Since the isMutable() check will cause the right thing to happen,
			// we don't need to check (image == fImage).
			if (image != null && image.isMutable()) {
				fImage = image;
				if (fSnapshot != null) fSnapshot.dispose();
				fSnapshot = Image.createImage(image);
				fImageItem.setImage(fSnapshot);
			} else {
				fImage = image;
				fImageItem.setImage(fImage);
			}
		}
	}

	public void setString(String text) {
		synchronized (Device.gDisplayableLock) {
			fText = text;
			fStringItem.setText(text);
		}
	}

	public void setTimeout(int timeout) {
		synchronized (Device.gDisplayableLock) {
			_setTimeout(timeout);
		}
	}

	void _setTimeout(int timeout) {
		if (timeout == FOREVER || timeout > 0) {
			fTimeout = timeout;
			//setModal(fTimeout == FOREVER);
		} else {
			// timeout must be positive or equal to FOREVER
			throw new IllegalArgumentException();
		}
	}

	public void setType(AlertType type) {
		synchronized (Device.gDisplayableLock) {
			fType = type;
			if (fPeer != null) ((AlertPeer) fPeer).changeType();
		}
	}

	public void removeCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			if (command == DISMISS_COMMAND) return;
			super.removeCommand(command);

			if (numApplicationCommands() == 0) {
				fHasDismissCommand = true;
				super.addCommand(DISMISS_COMMAND);
			}

			if (numApplicationCommands() <= 1) fCommandModal = false;

			/*
			 * What happens if there were 2 commands, one is being
			 * removed, and the timeout != FOREVER?
			 *
			 * From the spec for Alert.removeCommand() :
			 *
			 * "If the Alert has one Command (whether it is DISMISS_COMMAND
			 * or it is one provided by the application), the Alert may have
			 * the timed behavior as described above."
			 *
			 * So do we restart the alarm or do nothing?  For now, do nothing.
			 */
		}
	}

	public void setIndicator(Gauge gauge) {
		synchronized (Device.gDisplayableLock) {
			if (gauge == null) {
				if (fIndicator != null) {
					// remove existing indicator
					fIndicator.setScreen(null);
					fIndicator.setLayout(Item.LAYOUT_DEFAULT);
					removeItem(fIndicatorPosition);
					fIndicator = null;
					// Keep peer in sync with indicator change
					if (fPeer != null) ((AlertPeer)fPeer).scheduleUpdate(FormPeer.ITEMS_DELETED);
				}
			} else {
				/*If it is the same instance, do nothing*/
				if (fIndicator == gauge) return;

				if (!gauge.isAlertCompatible()) throw new IllegalArgumentException();

				Item oldIndicator = fIndicator;
				fIndicator = gauge;

				if (oldIndicator != null) {
					removeItem(fIndicatorPosition);
					oldIndicator.setScreen(null);
					oldIndicator.setLayout(Item.LAYOUT_DEFAULT);
				}

				addItem(fIndicator);
				fIndicator.setLayout(Item.LAYOUT_NEWLINE_BEFORE | Item.LAYOUT_2 | Item.LAYOUT_CENTER | Item.LAYOUT_EXPAND);
				fIndicator.setScreen(this);

				// Keep peer in sync with indicator change
				if (fPeer != null){
					if (oldIndicator != null){
						((AlertPeer)fPeer).scheduleUpdate(FormPeer.ITEMS_APPENDED | FormPeer.ITEMS_DELETED);
					} else {
						((AlertPeer)fPeer).scheduleUpdate(FormPeer.ITEMS_APPENDED);
					}
				}
			}
		}
	}

	public Gauge getIndicator() {
		synchronized (Device.gDisplayableLock) {
			return fIndicator;
		}
	}

	int getDisplayableType() {
		return TYPE_ALERT;
	}

	void addItem(Item item) {
		if (fItemsLength == fItems.length) {
			Item[] newArray = new Item[fItems.length * 2];
			System.arraycopy(fItems, 0, newArray, 0, fItems.length);
			fItems = newArray;
		}
		fItems[fItemsLength++] = item;
	}

	void removeItem(int index) {
		fItemsLength--;
		for (int i = index; i < fItemsLength; i++) {
			fItems[i] = fItems[i + 1];
		}
	}

	private void timeoutOccurred() {
		if (fCommands.size() == 1) {
			super.sendCommand((Command)fCommands.elementAt(0));
		}
	}

	/*
	 * Advance to the next Displayable.
	 */
	void advance() {
		// need to set the next one current
		if (fPeer == null) return;
		Display display = fPeer.getDisplay();
		display.setCurrent(display.getPrevious());
	}

	void showNotify() {
		if (fType != null) fType.playSound(getDisplay());

		/* Check the three types of modality before starting an alarm. */
		if (fTimeout != FOREVER && !fCommandModal) {
			fAlarm = new Alarm(fTimeout, this);
			fAlarm.start();
		}
	}

	/*
	 * Called by the AlertPeer where the value passed is based
	 * on whether or not the contents required scrolling.
	 */
	void setContentModal(boolean modal) {
		if (modal) {
			cancelAlarm();
			_setTimeout(FOREVER);
		}
	}

	static class Alarm extends Thread {

		private int timeout;
		private Alert alert;
		private boolean cancelled=false;

		Alarm(int timeout, Alert alert) {
			this.timeout = timeout;
			this.alert = alert;
		}

		public void cancel() {
			cancelled = true;
		}

		public void run() {
			try {
				Thread.sleep(timeout);
			} catch (InterruptedException e) {
				// cannot be interrupted ...
			}

			// the timeout is over, remove alert now ...
			if (!cancelled) alert.timeoutOccurred();
		}
	}

	static class DefaultCommandListener implements CommandListener {
		public void commandAction(Command command, Displayable displayable) {
			// always perform the automatic advance behavior
			((Alert)displayable).cancelAlarm();
			((Alert)displayable).advance();
		}
	}
}
