package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.*;
import javax.microedition.midlet.*;
import com.ibm.ive.midp.*;
import com.ibm.oti.util.*;
import com.ibm.oti.vm.*;
import com.ibm.ive.midp.util.*;

class AppManager implements IAppManager, NetworkActivityListener {

	static AppManager gAppManager;
	boolean fDisposed = false;
	private int fInstanceCount;

	static {
		try {
			VM.loadLibrary("ivemidp20_23"); //$NON-NLS-1$
			VM.addShutdownClass(new Runnable() {
				public void run() {
					if (gAppManager != null) gAppManager.dispose();
				}
			});
		} catch (IOException e) {
		  	e.printStackTrace();
	    }
	}

	public AppManager(String jadFile, String jarFile, String midletName) {
		MIDletManager.initialize(this, jadFile, jarFile, midletName);
		NetworkActivityHook.setListener(this);
	}

	public static void main(String[] args) {
		if (!Device.isAllowedToRunMIDletSuite()) return;

		String[] jadAndJar = parseArgs(args);
		try {
			gAppManager = new AppManager(jadAndJar[0], jadAndJar[1], jadAndJar[2]);
		} catch (NullPointerException npe) {
			System.err.println(MidpMsg.getString("AppManager.noJadOrJarSpecified"));
			return;
		}

		Device.init();
		gAppManager.open(jadAndJar[2]);
		/* Open will not return - do not put any code below the open() call. */
	}

	static String[] parseArgs(String[] args) {
		String[] jadAndJar = new String[3];
		for (int i = 0; i < args.length && i < 3; i++) {
			String arg = args[i];
			if (arg.length() > 5 && arg.startsWith("-jar:")) {
				if (jadAndJar[1] == null) jadAndJar[1] = arg.substring(5, arg.length());
			} else if (arg.length() > 8 && arg.startsWith("-midlet:")) {
				if (jadAndJar[2] == null) jadAndJar[2] = arg.substring(8, arg.length());
			} else if (jadAndJar[0] == null) {
				jadAndJar[0] = arg;
			}
		}
		return jadAndJar;
	}

	public void open(String midletName) {
		MIDletManager.setSecurity(new UserPermission(MIDletManager.gVendorName));
		if (Device.isAlreadyRunningThisSuite()) return;
		VM.stopExit();

		if (midletName != null) {
			int index = MIDletManager.getIndexFromClassName(midletName);
			if (index == -1) {
				System.err.println(MidpMsg.getString("AppManager.noMidletSpecifiedInJadOrManifest")); //$NON-NLS-1$
				VM.allowMidpExit();
				System.exit(0);
				return;
			}
			launch(index - 1);
		} else {
			if (MIDletManager.getClassName(1) == null) {
				System.err.println(MidpMsg.getString("AppManager.noMidletSpecifiedInJadOrManifest")); //$NON-NLS-1$
				VM.allowMidpExit();
				System.exit(0);
				return;
			}

			if (MIDletManager.getClassName(2) == null) {
				launch(0);
			} else {
				launchMIDletSuite();
			}
		}

		Device.runEventLoop();

		/*
		 * Force the exit of this process because it is possible that
		 * a midlet's thread might still be open but if the lifecycle of the
		 * midlet is over then everything must go away.  Performing the
		 * System.exit after the launch method has completed insures that
		 * midlets with or without a UI will be terminated properly.
		 */
		VM.allowMidpExit();

		/*In case of MIDP 1.0 midlets we have to restart the midlet when
		 *Window resize occurs by screen rotation or by any other way before
		 *this current process exits.
		 */
		if (MIDletManager.gReLaunchMidlet) Device.relaunchMidlet();

		System.exit(0);
	}

	void launchMIDletSuite() {
		allowInstantiation();
		launch(new MIDletSuite(gAppManager));
	}

	public void launch(final MIDlet midlet) {
		MIDletManager.launchingMIDlet();
		(new Thread() {
			public void run() {
				/* Show the user a list and let them choose the midlet. */
				IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
				MIDletManager.addMIDlet(midlet, accessor.getMidletIndex());
				try {
					resumeRequest(midlet);
				} catch (Exception e) {
					e.printStackTrace();
					midletDestroyed(midlet);
				}
			}
		}).start();
	}

	public void launch(final int index) {
		MIDletManager.launchingMIDlet();
		(new Thread() {
			public void run() {
				synchronized (Device.gDisplayableLock) {
					while (!Device.gRunning) {
						try {
							Device.gDisplayableLock.wait();
						} catch (InterruptedException e) {
						}
					}
				}
				MIDlet midlet = MIDletManager.getMIDlet(index);
				if (midlet == null) {
					 /* getClassName() is 1-indexed because the jad file stores the first midlet as MIDlet-1 */
					String className = MIDletManager.getClassName(index + 1);
					try {
						allowInstantiation();
						midlet = ((MIDlet) (Class.forName(className).newInstance()));
						IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
						accessor.setMidletIndex(index);
						MIDletManager.addMIDlet(midlet, index);
						try {
							resumeRequest(midlet);
						} catch (Exception e) {
							e.printStackTrace();
							midletDestroyed(midlet);
						}
					} catch (Throwable t) {
						t.printStackTrace();
						MIDletManager.cancelLaunchingMIDlet();
						shutdownIfNoActiveMIDlets();
						return;
					}
				} else {
					/*  The midlet has either (1) already been launched (2) been destroyed */
					MIDletManager.cancelLaunchingMIDlet(); // decrement "MIDlets being launched" count
					IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
					if (accessor.getState() == MIDletManager.MIDLET_DESTROYED) {
						/* relaunch */
						try {
							resumeRequest(midlet);
						} catch (Exception e) {
							e.printStackTrace();
							midletDestroyed(midlet);
						}
					} else {
						/* The midlet has already been launched.  Bring it to the front. */
						Display display = Display.getDisplay(midlet);
						display.fPeer.bringToTop();
					}
				}
			}
		}).start();
	}

	public boolean instantiationAllowed() {
		return fInstanceCount <= 1;
	}

	public void incrementInstanceCount() {
		fInstanceCount++;
	}

	public void allowInstantiation() {
		fInstanceCount = 0;
	}

	void dispose() {
		if (fDisposed) return;
		fDisposed = true;

		try {Graphics.disposeAll();} catch (Throwable t) {}
		try {Image.disposeAll();} catch (Throwable t) {}
		try {Font.disposeAll();} catch (Throwable t) {}
		try {Device.dispose();} catch (Throwable t) {}
	}

	/*
	 * This function gets called only as a result of implementation code.
	 * It must call destroyApp() to notify the midlet that it has been
	 * closed.
	 */
	public void midletDestroyed(MIDlet midlet) {
		IMIDletAccessor midletAccessor = MIDletManager.getAccessor(midlet);

		if (midletAccessor == null) {
			// We are already in the process of shutting down.
			// MIDlet already called notifyDestroyed and its accessor was removed.
			return;
		}

		try {
			midletAccessor.destroyApp(true);
		} catch (MIDletStateChangeException e) {
			e.printStackTrace();
		}
		handleMidletShutdown(midlet, midletAccessor);
	}

	/*
	 * This function gets called only as a result of user code. Do NOT call
	 * destroyApp() when this method is invoked.
	 */
	public void notifyDestroyed(final MIDlet midlet) {
		/*
		 * Do not run this code directly. The midlet shutdown process needs to be
		 * sync'ed with the serial thread and can deadlock if called directly in
		 * user code.
		 */
		Device.postRunnable(new Runnable() {
			public void run() {
				IMIDletAccessor midletAccessor = MIDletManager.getAccessor(midlet);

				if (midletAccessor == null) {
					// We are already in the process of shutting down.
					// This can happen when MIDlets do a notifyDestroyed() in their destroyApp() method.
					return;
				}

				handleMidletShutdown(midlet, midletAccessor);
			}
		});
	}

	void handleMidletShutdown(MIDlet midlet, IMIDletAccessor midletAccessor) {
		midletAccessor.setState(MIDletManager.MIDLET_DESTROYED);

		Display display = MIDletManager.getDisplay(midlet);
		if (display != null) display.dispose();
		MIDletManager.deleteMIDlet(midlet);

		shutdownIfNoActiveMIDlets();
	}

	void shutdownIfNoActiveMIDlets() {
		if (MIDletManager.getMIDletCount() > 0) return; // we still have active MIDlets

		Device.syncExec(new Runnable() {
			public void run() {
				Device.stopEventLoop();
				Device.flushRunnableQueue();
			}
		});
	}

	/*
	 * This function gets called only as a result of implementation code.
	 * It must call pauseApp() to notify the midlet that it has been
	 * paused.
	 */
	public void pauseApp(MIDlet midlet) {
		IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
		accessor.setState(MIDletManager.MIDLET_PAUSED);
		accessor.pauseApp();
	}

	/*
	 * This function gets called only as a result of user code. Do NOT call
	 * pauseApp() when this method is invoked.
	 */
	public void notifyPaused(MIDlet midlet) {
		IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
		accessor.setState(MIDletManager.MIDLET_PAUSED);
	}

	/*
	 * This method can be called either from user code or the implementation
	 * to invoke a midlet's startApp() call.
	 */
	public void resumeRequest(MIDlet midlet) {
		try {
			IMIDletAccessor accessor = MIDletManager.getAccessor(midlet);
			accessor.setState(MIDletManager.MIDLET_ACTIVE);
			accessor.startApp();
		} catch (RuntimeException e) {
			try {
				e.printStackTrace();
				IMIDletAccessor midletAccessor = MIDletManager.getAccessor(midlet);
				midletAccessor.destroyApp(true);
				midletAccessor.setState(MIDletManager.MIDLET_DESTROYED);
			} catch (MIDletStateChangeException exp) {
			}
		} catch (MIDletStateChangeException exp) {
		}
	}

	public void networkActivityStarted(String operation) {
		Device.networkActivityStarted(operation);
	}

	public void networkActivityComplete(String operation) {
		Device.networkActivityComplete(operation);
	}
}
