package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;

class CustomItemPeer extends Component {

	static final int BORDER_WIDTH = gBorder.getWidth(Item.TYPE_CUSTOM);
	static final int BORDER_HEIGHT = gBorder.getHeight(Item.TYPE_CUSTOM);
	static final int TOTAL_BORDER_WIDTH = 2 * BORDER_WIDTH;
	static final int TOTAL_BORDER_HEIGHT = 2 * BORDER_HEIGHT;

	private static final boolean SUPPORT_H_TRAVERSAL = true;
	private static final boolean SUPPORT_V_TRAVERSAL = true;
//	private static final boolean gSupportTraversal = (SUPPORT_H_TRAVERSAL || SUPPORT_V_TRAVERSAL);

	/*
	 * This ensures that we trigger the show & hide notifications based on
	 * whether the CustomItem is actually in the viewport or not.
	 */
	boolean fNotify;

	CustomItem fCustomItem;
	private static int gInteractionModes;
	boolean fHighlighted;

	CustomItemPeer(Composite parent, CustomItem item) {
		super(parent);
		fCustomItem = item;
		fCustomItem.fPeer = this;
	}

	FormPeer getForm() {
		return (FormPeer) getWindow();
	}

	void scheduleRepaint(int x, int y, int w, int h) {
		x += BORDER_WIDTH;
		y += BORDER_HEIGHT;
		repaint(x, y, w, h);
	}

	boolean isShowingOnScreen() {
		WindowContent content = getWindow().fContentComponent;

		int startY = getOriginY() - content.fY;
		int endY = startY + fHeight;

		return startY < content.fHeight && endY > 0;
	}

	void paint(Graphics g) {
		// do notify if necessary regardless or whether the CustomItem is in view
		if (isShowingOnScreen()) {
			if (!fNotify) {
				fNotify = true;
				showNotify();
			}
		} else {
			if (fNotify) {
				hideNotify();
				fNotify = false;
			}
			return; // not showing: don't continue with paint
		}

		if (fHighlighted) paintBorder(g, true, 0, 0, fWidth - 1, fHeight - 1);

		Rectangle contentBounds = new Rectangle(BORDER_WIDTH, BORDER_HEIGHT, fWidth - TOTAL_BORDER_WIDTH, fHeight - TOTAL_BORDER_HEIGHT);
		Rectangle paintBounds = contentBounds.intersection(g.fClipRect);
		int paintX = paintBounds.x - BORDER_WIDTH;
		int paintY = paintBounds.y - BORDER_HEIGHT;
		int paintWidth = paintBounds.width;
		int paintHeight = paintBounds.height;
		int translateX = BORDER_WIDTH;
		int translateY = BORDER_HEIGHT;

		g.translate(translateX, translateY);
		g.setClip(paintX, paintY, paintWidth, paintHeight);
		g.restrictClip(g.fRestrictedClipX + BORDER_WIDTH, g.fRestrictedClipY + BORDER_HEIGHT, g.fRestrictedClipWidth - TOTAL_BORDER_WIDTH, g.fRestrictedClipHeight - TOTAL_BORDER_HEIGHT);

		g.setColor(0x000000);
		g.setStrokeStyle(Graphics.SOLID);
		g.setFont(Font.getDefaultFont());

		try {
			fCustomItem.paint(g, paintWidth, paintHeight);
		} catch (Exception e) {
			System.err.println(MsgDrawnCommon.getString("CustomItemPeer.UncaughtExceptionInPaint")); //$NON-NLS-1$
			e.printStackTrace();
		}

		g.setStrokeStyle(Graphics.SOLID);
	}

	static int getGameAction(int keyCode) {
		return CanvasPeer.getGameAction(keyCode);
	}

	/*
	 * The value returned will be based on device constraints and
	 * user preferences.
	 */
	static int getInteractionModes() {

		if (gInteractionModes == 0) {

			/* Default settings which should be available everywhere. */
			gInteractionModes |= CustomItem.KEY_PRESS | CustomItem.KEY_RELEASE;

			/* Device specific settings. */
			if (Device.hasPointerEvents()) {
				gInteractionModes |= CustomItem.POINTER_PRESS | CustomItem.POINTER_RELEASE;
			}
			if (Device.hasPointerMotionEvents()) {
				gInteractionModes |= CustomItem.POINTER_DRAG;
			}
			if (Device.hasKeyRepeatEvents()) {
				gInteractionModes |= CustomItem.KEY_REPEAT;
			}

			/* Settings which depend on the implementation. */
			if (SUPPORT_H_TRAVERSAL) {
				gInteractionModes |= CustomItem.TRAVERSE_HORIZONTAL;
			}
			if (SUPPORT_V_TRAVERSAL) {
				gInteractionModes |= CustomItem.TRAVERSE_VERTICAL;
			}
		}

		return gInteractionModes;
	}

	boolean pointerPressed(int x, int y) {
		x -= BORDER_WIDTH;
		y -= BORDER_HEIGHT;
		try {
			fCustomItem.pointerPressed(x,y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean pointerReleased(int x, int y) {
		x -= BORDER_WIDTH;
		y -= BORDER_HEIGHT;
		try {
			fCustomItem.pointerReleased(x,y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean pointerDragged(int x, int y) {
		x -= BORDER_WIDTH;
		y -= BORDER_HEIGHT;
		try {
			fCustomItem.pointerDragged(x,y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean keyPressed(int keyCode) {
		try {
			fCustomItem.keyPressed(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean keyReleased(int keyCode) {
		try {
			fCustomItem.keyReleased(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean keyRepeated(int keyCode) {
		try {
			fCustomItem.keyRepeated(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	void dispose() {
		fCustomItem.fPeer = null;
		super.dispose();
	}

	GraphicsThreadsafe getDisplayGraphics() {
		GraphicsThreadsafe g = getWindow().getDisplayGraphics();
		setOriginAndClip(g);

		int paintX = fX + BORDER_WIDTH;
		int paintY = fY + BORDER_HEIGHT;
		int paintWidth = fWidth - TOTAL_BORDER_WIDTH;
		int paintHeight = fHeight - TOTAL_BORDER_HEIGHT;
		g.translate(BORDER_WIDTH, BORDER_HEIGHT);
		g.restrictClip(g.fRestrictedClipX + paintX, g.fRestrictedClipY + paintY, paintWidth, paintHeight);
		g.setClip(paintX, paintY, paintWidth, paintHeight);

		return g;
	}

	int getMinimumHeight() {
		try {
			return fCustomItem.getMinContentHeight() + TOTAL_BORDER_HEIGHT;
		} catch (Exception e) {
			e.printStackTrace();
			return 0;
		}
	}

	int getMinimumWidth() {
		try {
			return fCustomItem.getMinContentWidth() + TOTAL_BORDER_WIDTH;
		} catch (Exception e) {
			e.printStackTrace();
			return 0;
		}
	}

	int getPreferredHeight() {
		try {
			return fCustomItem.getPrefContentHeight(fCustomItem.fPreferredWidth) + TOTAL_BORDER_HEIGHT;
		} catch (Exception e) {
			e.printStackTrace();
			return 0;
		}
	}

	int getPreferredWidth() {
		try {
			return fCustomItem.getPrefContentWidth(fCustomItem.fPreferredHeight) + TOTAL_BORDER_WIDTH;
		} catch (Exception e) {
			e.printStackTrace();
			return 0;
		}
	}

	int getType() {
		return Item.TYPE_CUSTOM;
	}

	void sizeChanged(int w, int h) {
		w -= TOTAL_BORDER_WIDTH;
		h -= TOTAL_BORDER_HEIGHT;
		try {
			fCustomItem.sizeChanged(w, h);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	protected boolean traverse(int dir, int viewportWidth, int viewportHeight, int[] visRect_inout, int x, int y) {

		boolean itemTakesTraverse = false;
		boolean focus = hasFocus();

		try {
			itemTakesTraverse = fCustomItem.traverse(dir, viewportWidth, viewportHeight, visRect_inout);
		} catch (Exception e) {
			e.printStackTrace();
		}

		if (!focus) {
			fHighlighted = true;

			/* Workaround for CustomItem#SizeChanged TCK failure issue (seen only on RHEL)
			 *
			 * Avoid calling repaint(), if the traverse is received as a result of
			 * call to setInitialFocus(). There is a subsequent repaint() in Window.show().
			 * The CustomItem#SizeChanged TCK seems to be counting the number of paint()
			 * it receives and tries to compare values as soon as it receives the second
			 * paint(). So, generating an extra paint(), though not against the spec.,
			 * might fail this TCK case depending on how fast the paint() calls are generated.
			 */
			if (!(fCustomItem.fScreen.fPeer.fFocusComponent == null && dir == Canvas.DOWN)) repaint();
		}
		return itemTakesTraverse ? itemTakesTraverse : !focus;
	}

	protected void traverseOut() {
		fHighlighted = false;
		try {
			fCustomItem.traverseOut();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	void showNotify() {
		try {
			if (fNotify) fCustomItem.showNotify();
		} catch (Exception e) {
			e.printStackTrace();
		}

		if (fCustomItem.fIsMediaItem) {
			((IMediaContainerListener)fCustomItem).showMedia(0, 0, fWidth, fHeight);
		}
	}

	void hideNotify() {
		try {
			if (fNotify) fCustomItem.hideNotify();
		} catch (Exception e) {
			e.printStackTrace();
		}

		if (fCustomItem.fIsMediaItem) {
			((IMediaContainerListener)fCustomItem).hideMedia();
		}
	}
}
