package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import javax.microedition.midlet.*;
import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;

/**
 * A <code>Display</code> is the abstraction for a device
 * actual screen.
 */
public class Display {

	public static final int ALERT = 3;
	public static final int CHOICE_GROUP_ELEMENT = 2;
	public static final int LIST_ELEMENT = 1;

	public final static int COLOR_BACKGROUND = 0;
	public final static int COLOR_BORDER = 4;
	public final static int COLOR_FOREGROUND = 1;
	public final static int COLOR_HIGHLIGHTED_BACKGROUND = 2;
	public final static int COLOR_HIGHLIGHTED_BORDER = 5;
	public final static int COLOR_HIGHLIGHTED_FOREGROUND = 3;

	Displayable fCurrent = null;
	Displayable fPrevious = null;

	private IMIDletAccessor fAccessor;

	DisplayPeer fPeer;
	Object fLock = new Object();
	static Display gCurrentDisplay;
	boolean fPauseSetCurrent = false;
	Runnable fSetCurrentRunnable;

	Display(MIDlet m) {
		if (m != null) fAccessor = MIDletManager.getAccessor(m);
		fPeer = new DisplayPeer(this);
		gCurrentDisplay = this;
	}

	public Displayable getCurrent() {
		synchronized (fLock) {
			return fCurrent;
		}
	}

	public int getBorderStyle(boolean highlighted) {
		/*
		 * Somewhat questionable that these are the values
		 * that the TCK expects.  Test should probably be
		 * excluded as it tests the RI not the spec.
		 */
		return (highlighted ? Graphics.SOLID : Graphics.DOTTED);
	}

	public int getColor(int specifier) {
		if (specifier < 0 || specifier > 5) throw new IllegalArgumentException();
		// valid values of specifier are in the range [0,5]
		if (fPeer == null) return 0;
		return DisplayPeer.getSpecifiedColor(specifier);
	}

	public boolean flashBacklight(int millis) {
		if (millis < 0) throw new IllegalArgumentException();
		if (fPeer == null) return false;
		if (fAccessor.getState() != MIDletManager.MIDLET_ACTIVE) return false;
		return Device.flashBacklight(millis);
	}

	public boolean vibrate(int millis) {
		if (millis < 0) throw new IllegalArgumentException();
		if (fPeer == null) return false;
		if (fAccessor.getState() != MIDletManager.MIDLET_ACTIVE) return false;
		return Device.vibrate(millis);
	}

	public int numAlphaLevels() {
		return Device.numAlphaLevels();
	}

	public void setCurrentItem(Item item) {
		if (item == null) throw new NullPointerException();
		if ((item.getScreen() == null) || (item.getScreen().getDisplayableType() == Displayable.TYPE_ALERT)) {
			throw new IllegalStateException(MidpMsg.getString("Display.setCurrentItem.NonItem")); //$NON-NLS-1$
		}

		Form form = (Form)item.getScreen();
		form.setCurrentItem(item);
		setCurrent(form);
	}

	public static Display getDisplay(MIDlet midlet) {
		if (midlet == null) throw new NullPointerException();

		synchronized (MIDletManager.gLock) {
			Display display = MIDletManager.getDisplay(midlet);
			if (display == null) {
				display = new Display(midlet);
				MIDletManager.registerDisplay(midlet, display);
			}
			return display;
		}
	}

	public int getBestImageHeight(int imageType) {
		// Valid values of imageType are in the range [1,3]
		if (imageType < 1 || imageType > 3) throw new IllegalArgumentException();

		switch (imageType) {
			case ALERT:
				return Device.getDisplayHeight() / 3;
			case CHOICE_GROUP_ELEMENT:
			case LIST_ELEMENT:
				return Device.getIconHeight();
		}
		return 0;
	}

	public int getBestImageWidth(int imageType) {
		// Valid values of imageType are in the range [1,3]
		if (imageType < 1 || imageType > 3) throw new IllegalArgumentException();

		switch (imageType) {
			case ALERT:
				return Device.getDisplayWidth()/2;
			case CHOICE_GROUP_ELEMENT:
			case LIST_ELEMENT:
				return Device.getIconWidth();
		}
		return 0;
	}

	public boolean isColor() {
		return Device.isColor();
	}

	public int numColors() {
		return Device.getNumColors();
	}

	public void setCurrent(final Alert alert, final Displayable nextDisplayable) {
		if (nextDisplayable == null || alert == null) throw new NullPointerException();
		if (nextDisplayable instanceof Alert) throw new IllegalArgumentException();

		synchronized (fLock) {
			if (fPauseSetCurrent) {
				fSetCurrentRunnable = new Runnable() {
					public void run() {
						setCurrentImpl(alert, nextDisplayable);
					}
				};
			} else {
				setCurrentImpl(alert, nextDisplayable);
			}
		}
	}

	void setCurrentImpl(Alert alert, final Displayable nextDisplayable) {
		final Displayable last = fCurrent;
		fPrevious = nextDisplayable;
		fCurrent = alert;
		Device.postRunnable(new Runnable() {
			public void run() {
				synchronized (Device.gRepaintLock) {
					synchronized (Device.gDisplayableLock) {
						if (last != null && last.fPeer != null && last != nextDisplayable) {
							last.fPeer.dispose();
						}
					}
				}
				/*
				 * We cannot hold any locks while calling hideNotify. Note that we
				 * must also call hideNotify in the serial thread.
				 */
				if (last != null) last.callHideNotify();
			}
		});
		if (fPeer != null) fPeer.setCurrent(alert);
	}

	public void setCurrent(final Displayable newDisplayable) {
		synchronized (fLock) {
			if (fPauseSetCurrent) {
				fSetCurrentRunnable = new Runnable() {
					public void run() {
						setCurrentImpl(newDisplayable);
					}
				};
			} else {
				setCurrentImpl(newDisplayable);
			}
		}
	}

	void setCurrentImpl(final Displayable newDisplayable) {
		synchronized (fLock) {
			if (newDisplayable != null) {
				/*
				 * If the passed in Displayable is same as the current
				 * one, treat it as a request to bring the MIDlet to the
				 * foreground.
				 */
				if (newDisplayable == fCurrent) {
					Device.bringMidletToForeground(fPeer.fHandle);
					return;
				}
				// Cache the previous Displayable
				final Displayable cachedPreviousDisplayable = fPrevious;
				final Displayable last = fPrevious = fCurrent;
				fCurrent = newDisplayable;
				Device.postRunnable(new Runnable() {
					public void run() {
						synchronized (Device.gRepaintLock) {
							synchronized (Device.gDisplayableLock) {
								if (last != null && last.fPeer != null && newDisplayable.getDisplayableType() != Displayable.TYPE_ALERT) {
									last.fPeer.dispose();
								}
								/*
								 * If the last Displayable is an Alert, then there is a possibility
								 * that we might not have disposed the prior Displayable
								 */
								if (last != null && last.getDisplayableType() == Displayable.TYPE_ALERT) {
									if (cachedPreviousDisplayable != null && cachedPreviousDisplayable != newDisplayable && cachedPreviousDisplayable.fPeer != null) {
										cachedPreviousDisplayable.fPeer.dispose();
									}
								}
							}
						}
						/*
						 * We cannot hold any locks while calling hideNotify. Note that we
						 * must also call hideNotify in the serial thread.
						 */
						if (fPrevious != null) fPrevious.callHideNotify();
						if (cachedPreviousDisplayable != null && cachedPreviousDisplayable != newDisplayable && last.getDisplayableType() == Displayable.TYPE_ALERT) {
							cachedPreviousDisplayable.hideNotify();
						}
					}
				});
				if (fPeer != null) fPeer.setCurrent(newDisplayable);
			} else { // Displayable is null. Request for sending the MIDlet to the background
				Device.sendMidletToBackground(fPeer.fHandle);
			}
		}
	}

	void pauseSetCurrent() {
		synchronized (fLock) {
			fPauseSetCurrent = true;
		}
	}

	void unpauseSetCurrent() {
		synchronized (fLock) {
			fPauseSetCurrent = false;
			if (fSetCurrentRunnable == null) return;
			Device.postRunnable(fSetCurrentRunnable);
			fSetCurrentRunnable = null;
		}
	}

	public void callSerially(Runnable runnable) {
		if (runnable == null) return;
		Device.postRunnable(runnable);
	}

	Displayable getPrevious() {
		return fPrevious;
	}

	void dispose() {
		synchronized (Device.gRepaintLock) {
			synchronized (Device.gDisplayableLock) {
				if (fPeer != null) {
					fPeer.dispose();
					fPeer = null;
				}
			}
		}
	}

	void close() {
		if (fPeer != null) fPeer.close();
	}
}
