package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;

class DisplayablePeer extends Window {

//	Graphics fGraphics;			/* Accessed by INL. Do not move. */

	Displayable fDisplayable;
	Widget fTitleComponent;
	Widget fTickerComponent;
	CommandManager fCommandManager;

	public DisplayablePeer(Display display, Displayable displayable) {
		super(null, display);

		fInvalidated = true;
		fDisplay = display;
		fDisplayable = displayable;
		fX = 0;
		fY = 0;
		fWidth = display.fPeer.getWidth();
		fHeight = display.fPeer.getHeight();

		fDisplayable.fPeer = this;
	}

	void create() {
		fTickerComponent = new TickerPeer(this);
		fTitleComponent = generateTitleType();
		createContents();
		fCommandManager = new CommandManager(fDisplayable, this);
		createGraphics();
		if (ScrollBarComponent.isAlwaysVisible()) fContentComponent.createScrollBar();
	}

	void createContents() {
		switch (fDisplayable.getDisplayableType()) {
			case Displayable.TYPE_ALERT:
			case Displayable.TYPE_FORM:
				fContentComponent = new FormContentComponent((FormPeer) this);
				break;
			case Displayable.TYPE_LIST:
				fContentComponent = new ListContentComponent(this);
				break;
			default:
				fContentComponent = new FillContentComponent(this);
				break;
		}
	}

	Widget generateTitleType() {
		if (fDisplayable.getDisplayableType() == Displayable.TYPE_MENU) return null;
		return new TitleComponent(this);
	}

	void dispose() {
		super.dispose();
		if (fCommandManager != null) {
			fCommandManager.dispose();
			fCommandManager = null;
		}
		fDisplayable.fPeer = null;
	}

	boolean isMidp10CanvasPeer() {
		return (getType() == Displayable.TYPE_CANVAS) && MIDletManager.isMidp10();
	}

	boolean isFullScreenCanvasPeer() {
		return (getType() == Displayable.TYPE_CANVAS) && ((CanvasPeer)this).fFullScreen;
	}

	int getType() {
		return fDisplayable.getDisplayableType();
	}

	void setTitle(String title) {
		scheduleUpdate(TITLE_UPDATED);
	}

	void setTicker(Ticker ticker) {
		scheduleUpdate(TICKER_UPDATED);
	}

	void _setTicker(Ticker ticker) {
		fTickerComponent.update();
	}

	void _setTitle(String title) {
		fTitleComponent.update();
	}

	public void addCommand(Command c, int index) {
		scheduleUpdate(COMMANDS_CHANGED);
	}

	public void removeCommand(Command c, int index) {
		scheduleUpdate(COMMANDS_CHANGED);
	}

	void dispatchCommand(Command c) {
	   fDisplayable.sendCommand(c);
	}

	boolean dispatchEvent(Event e) {
		switch (e.fType) {
			case Event.WINDOW_PAINT:
				if (fGraphics != null) fGraphics.reset();
				if (isShown()) {
					Device.syncExec(fRepaintRunnable);
					if (fDisplayable.getDisplayableType() == Displayable.TYPE_CANVAS) {
						((Canvas)fDisplayable).repaint();
					}
				}
				return false;
			case Event.WINDOW_ACTIVATE:
				if (fGraphics != null) fGraphics.reset();
				setInitialFocus();
				if (isShown()) {
					Device.syncExec(fRepaintRunnable);
					if (fDisplayable.getDisplayableType() == Displayable.TYPE_CANVAS) {
						((Canvas)fDisplayable).repaint();
					}
				}
				return false;
			case Event.WINDOW_DEACTIVATE:
				if(fGraphics != null) {
					fGraphics.reset();
				}
				if (fFocusComponent != null) {
					fFocusComponent.traverseOut();
				}
				return false;
			case Event.MENU_OPENING:
			case Event.MENU_CLOSING:
			case Event.MENU_SELECTED:
				fCommandManager.dispatchEvent(e);
				return false;
			case Event.WINDOW_FOCUSIN:
//				if (fFocusComponent != null) fFocusComponent.traverse(CustomItem.NONE, fFocusComponent.fWidth, fFocusComponent.fHeight, getVisibleRect(fFocusComponent), 0, 0);
				return false;
		}
		return false;
	}

	int getMaxWidth(Component component) {
		if (fContentComponent == null) return Device.getDisplayWidth();
		return fContentComponent.fWidth;
	}

	public Graphics getGraphics() {
		return fGraphics;
	}

	Displayable getDisplayable() {
		return fDisplayable;
	}

	public void resize(int newWidth, int newHeight) {
//		System.err.println("Resize: " + newWidth + "," + newHeight);
		if (fDisposed || newWidth <= 0 || newHeight <= 0) return;
		if (newWidth == fWidth && newHeight == fHeight) return;

		synchronized (Device.gRepaintLock) {
			int oldWidth = fWidth;
			int oldHeight = fHeight;
			fWidth = newWidth;
			fHeight = newHeight;

			if (!isShown()) return;

			clearCachedSizes();
			layout();

			/*
			 * Need to call CommandManager.update() after layout() call
			 * so that CommandComponent width is correct
			 */

			if (fCommandManager != null) fCommandManager.update();
			if (fGraphics != null) fGraphics.resetClipRestriction();

			fContentComponent.resizeWithDeltas(newWidth - oldWidth, newHeight - oldHeight);
			//if this is an Alert then reposition the window accordingly
			if (fDisplayable.getDisplayableType() == Displayable.TYPE_ALERT) ((AlertPeer)this).alertResize(newWidth,newHeight);

			repaint();
		}
	}

	DisplayablePeer getDisplayablePeer() {
		return this;
	}

	int getMinimumWidth() {
		return Device.getDisplayWidth();
	}

	int getMinimumHeight() {
		return Device.getDisplayHeight();
	}

	static int getDisplayableHeight() {
		return Device.getDisplayHeight();
	}

	static int getDisplayableWidth() {
		return Device.getDisplayWidth();
	}

	void layout() {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		super.layout();
		fDisplayable.sizeChanged(fContentComponent.fWidth, fContentComponent.fHeight);
	}

	void invalidate() {
		scheduleUpdate(LAYOUT_CHANGED);
	}

	void layoutChanged(Component child) {
		if (!fInvalidated) scheduleUpdate(LAYOUT_CHANGED);
	}

	void update() {
		if (!fUpdateScheduled) return;
		fUpdateScheduled = false;

		synchronized (Device.gDisplayableLock) {
			if (fDisposed) return;

			// Added/Removed items
			if ((fUpdateType & ITEMS_CHANGED) != 0) {
				fInvalidated = true;
				disposeComponents();
				/*
				 * Platform native MenuBarComponent does not extend Component.
				 * So disposeComponents() does not dispose it.
				 * CommandManager.dispose() disposes it.
				 */
				if (fCommandManager != null) fCommandManager.dispose();
				fUpdateType &= ~ITEMS_CHANGED;

				show();
			} else {
				// Changed Item Values
				if ((fUpdateType & ITEM_VALUE_CHANGED) != 0) {
					fUpdateType &= ~ITEM_VALUE_CHANGED;
					fContentComponent.updateItemValues();
				}

				if ((fUpdateType & COMMANDS_CHANGED) != 0) {
					fUpdateType &= ~COMMANDS_CHANGED;
					if (fCommandManager != null) fCommandManager.update();
				}

				if ((fUpdateType & TICKER_UPDATED) != 0) {
					fUpdateType &= ~TICKER_UPDATED;
					_setTicker(fDisplayable.fTicker);
				}

				if ((fUpdateType & TITLE_UPDATED) != 0) {
					fUpdateType &= ~TITLE_UPDATED;
					_setTitle(fDisplayable.fTitle);
				}
			}
		}

		super.updateLayout();
	}

	void focusChanged() {
		/* No-op */
	}
}
