package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;

/**
 * <code>Form</code> is the high-level MIDP UI abstraction.<p>
 * A <code>Form</code> displays a title and contains one or
 * several <code>Item</code>s.
 *
 * @see Item
 * @see ChoiceGroup
 * @see DateField
 * @see Gauge
 * @see ImageItem
 * @see StringItem
 * @see TextField
 **/
public class Form extends Screen {

	private ItemStateListener fItemListener;

	Item fItems[] = new Item[10];
	int fItemsLength;
	Item fCurrentItem;

	public Form(String formTitle) {
		this(formTitle, null);
	}

	public Form(String formTitle, Item[] initItems) {
		super(formTitle);

		// insert items into internal vector
		if (initItems != null) {
			for (int i = 0; i < initItems.length; i++) {
				Item currentItem = initItems[i];

				if (currentItem == null) {
					// if one element of initItems is null, reset the items vector
					// and throw a NullPointerException
					fItemsLength = 0;
					throw new NullPointerException();
				} else {
					if (currentItem.getScreen() != null) throw new IllegalStateException();

					currentItem.setScreen(this);
					addItem(currentItem);
				}
			}
		}
	}

	/*
	 * overrides Displayable.isClosing()
	 *
	 * Call hideNotify() on CustomItems
	 */
	void isClosing(Display display) {
		if (fPeer != null) fPeer.hideNotify();
	}

	public int append(Image image) {
		synchronized (Device.gDisplayableLock) {
			if (image == null) throw new NullPointerException(MidpMsg.getString("Form.append.NullImage")); //$NON-NLS-1$

			return append(new ImageItem(null, image, ImageItem.LAYOUT_DEFAULT, null));
		}
	}

	public int append(Item item) {
		synchronized (Device.gDisplayableLock) {
			// the following throws NullPointerException first if item is null
			if (item.fScreen != null) throw new IllegalStateException(MidpMsg.getString("Form.Used.Item")); //$NON-NLS-1$

			// Mark this Item as being held by a Screen
			item.fScreen = this;
			addItem(item);

			// Keep peer in sync
			if (fPeer != null) fPeer.scheduleUpdate(FormPeer.ITEMS_APPENDED);
			return fItemsLength - 1;
		}
	}

	public int append(String string) {
		if (string == null) throw new NullPointerException(MidpMsg.getString("Form.append.NullString")); //$NON-NLS-1$
		return append(new StringItem(null, string));
	}

	public void delete(int index) {
		synchronized (Device.gDisplayableLock) {
			if (index < 0 || index >= fItemsLength) throw new IndexOutOfBoundsException();
			Item anItem = fItems[index];
			removeItem(index);
			if (anItem == fCurrentItem) fCurrentItem = null;
			// Mark the item as unused by any screen
			anItem.fScreen = null;
			// Keep peer in sync
			if (fPeer != null) fPeer.scheduleUpdate(FormPeer.ITEMS_DELETED);
		}
	}

	public void deleteAll() {
		synchronized (Device.gDisplayableLock) {
			if (fItems == null) return;

			// Each item needs to be marked as unused so that it can be
			// placed onto a different form
			for (int i = 0; i < fItemsLength; i++) {
				fItems[i].fScreen = null;
			}

			// Clear any item that was marked as the current item
			fCurrentItem = null;
			fItemsLength = 0;
			// Keep peer in sync
			if (fPeer != null) fPeer.scheduleUpdate(FormPeer.ITEMS_DELETED);
		}
	}

	public Item get(int index) {
		synchronized (Device.gDisplayableLock) {
			if (fItems == null || index < 0 || index >= fItemsLength) throw new IndexOutOfBoundsException();
			return fItems[index];
		}
	}

	public void insert(int index, Item item) {
		synchronized (Device.gDisplayableLock) {
			// the following throws NullPointerException first if item is null
			if (item.getScreen()!= null) throw new IllegalStateException(MidpMsg.getString("Form.Used.Item")); //$NON-NLS-1$
			if (index < 0 || index > fItemsLength) throw new IndexOutOfBoundsException();

			if (index == size()) {
				append(item);
				return;
			}

			// Mark the item as used by a Screen
			item.setScreen(this);
			// IndexOutOfBoundsException will be thrown if the index is invalid
			insertItem(index, item);
			// Keep the peer in sync
			if (fPeer != null) fPeer.scheduleUpdate(FormPeer.ITEMS_INSERTED);
		}
	}

	public void set(int index, Item item) {
		synchronized (Device.gDisplayableLock) {
			// the following throws NullPointerException first if item is null
			if (item.getScreen()!= null) throw new IllegalStateException(MidpMsg.getString("Form.Used.Item")); //$NON-NLS-1$
			if (index < 0 || index >= fItemsLength) throw new IndexOutOfBoundsException();

			// Find the item to be replaced
			Item oldItem = fItems[index];
			if (oldItem == fCurrentItem) fCurrentItem = null;
			// Settting the same item should do nothing
			if (oldItem == item) return;
			// Free the old item so it can be reused
			oldItem.setScreen(null);
			// Mark the new item as used
			item.setScreen(this);
			fItems[index] = item;
			// Keep the peer in sync
			if (fPeer != null) fPeer.scheduleUpdate(FormPeer.ITEMS_SET);
		}
	}

	public void setItemStateListener(ItemStateListener listener) {
		synchronized (Device.gDisplayableLock) {
			fItemListener = listener;
		}
	}

	public int size() {
		synchronized (Device.gDisplayableLock) {
			return fItemsLength;
		}
	}

	void callItemStateChanged(Item item) {
		if (fItemListener != null) {
			try {
				fItemListener.itemStateChanged(item);
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}

	void setCurrentItem(Item item) {
		fCurrentItem = item;
		if (fPeer != null) {
			((FormPeer)fPeer).setCurrentItem(fCurrentItem);
		}
	}

	int getDisplayableType() {
		return TYPE_FORM;
	}

	void addItem(Item item) {
		if (fItemsLength == fItems.length) {
			Item[] newArray = new Item[fItems.length * 2];
			System.arraycopy(fItems, 0, newArray, 0, fItems.length);
			fItems = newArray;
		}
		fItems[fItemsLength] = item;
		fItemsLength++;
	}

	void removeItem(int index) {
		fItemsLength--;
		for (int i = index; i < fItemsLength; i++) {
			fItems[i] = fItems[i + 1];
		}
	}

	void insertItem(int index, Item item) {
		fItemsLength++;
		if (fItemsLength > fItems.length) {
			Item[] copy = new Item[fItems.length * 2];
			System.arraycopy(fItems, 0, copy, 0, fItems.length);
			fItems = copy;
		}
		if (index < 0) index = fItemsLength - 1;
		for (int i = fItemsLength - 1; i > index; i--) {
			fItems[i] = fItems[i - 1];
		}
		fItems[index] = item;
	}
}
