package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.IOException;
import com.ibm.ive.midp.*;

class GaugePeer extends Component {

	/* font height seems like a good measure to keep things proportionate */
	static final int GAUGE_ITEM_HEIGHT = Math.max(17, ((Font.getDefaultFont()._getHeight() + 1 ) / 2) * 2 - 1);

	static final int BORDER_WIDTH = gBorder.getWidth(Item.TYPE_GAUGE);
	static final int BORDER_HEIGHT = gBorder.getHeight(Item.TYPE_GAUGE);
	static final int TOTAL_BORDER_WIDTH = 2 * BORDER_WIDTH;
	static final int TOTAL_BORDER_HEIGHT = 2 * BORDER_HEIGHT;
	static final int CONTINUOUS_GAUGE_ITEM_HEIGHT = GAUGE_ITEM_HEIGHT * 2;
	static final int GAUGE_ITEM_WIDTH = Device.getDisplayWidth() / 2;
	static final int ARROW_WIDTH = GAUGE_ITEM_HEIGHT - TOTAL_BORDER_WIDTH;
	static final int GAUGE_HEIGHT = GAUGE_ITEM_HEIGHT - TOTAL_BORDER_HEIGHT;
	static final int GAUGE_VALUE_HEIGHT = GAUGE_HEIGHT - 2;

	static final int GAUGE_X = TOTAL_BORDER_WIDTH + ARROW_WIDTH;
	static final int NON_INTERACTIVE_GAUGE_X = BORDER_WIDTH;
	static final int GAUGE_VALUE_X = GAUGE_X + 1;
	static final int GAUGE_Y = BORDER_HEIGHT;
	static final int GAUGE_VALUE_Y = GAUGE_Y + 1;
	static final int ARROW_X = GAUGE_X - BORDER_WIDTH;
	static final int V_MIDPOINT = GAUGE_Y + (GAUGE_HEIGHT / 2);
	static final int ARROW_BOTTOM = GAUGE_Y + GAUGE_HEIGHT - 1;

	static final int GAUGE_BACKGROUND_RGB = 0xE5E5E5;
	static final int GAUGE_FOREGROUND_RGB = 0xB0B0FF;

	static final int MAX_INCREMENTAL_ANIMATION_STATE = 10;

	static Image[] gHourglassImages;
	static Image gGaugeImage = null;
	static boolean gHourglassLoadAttempted = false;
	static boolean gHourglassLoaded = false;
	static boolean gGaugeImageLoadAttempted = false;
	static final String INCREMENTAL_GAUGE_FILE = "images/gauge.png";  //$NON-NLS-1$
	static final String CONTINUOUS_GAUGE_FILE = "images/hourglass";  //$NON-NLS-1$

	int fGaugeWidth = 0;
	int fHighlightBoxWidth = 0;
	int fGaugeValueWidth = 0;
	int fArrow2X = 0;
	int fOldGaugeValue;
	int fIncrementalAnimationState = 0;
	int fContinuousAnimationState = 0;

	Gauge fGauge;
	GaugeThread fGaugeThread;

	boolean fHighlighted = false;
	boolean fLeftPressed = false;
	boolean fRightPressed = false;

	/* Hourglass values */
	int fHourglassX;
	static final int HOURGLASS_Y = 2 + BORDER_HEIGHT;
	int fHourglassW;
	int fHourglassH;

	int fHourglassX1;
	int fHourglassX2;
	int fHourglassX3;
	int fHourglassX4;
	int fHourglass5;
	int fHourglassX6;

	int fHourglassY1;
	int fHourglassY15;
	int fHourglassY2;
	int fHourglassY3;
	int fHourglassY4;
	int fHourglassY45;
	int fHourglassY5;
	int fHourglassY6;
	/* End Hourglass Values */

	GaugePeer(Composite parent, Gauge gauge) {
		super(parent);
		fGauge = gauge;
		fGauge.fPeer = this;
		if (isContinuousRunning()) startGaugeThread(false);
	}

	void dispose() {
		super.dispose();

		stopGaugeThread();
		fGauge.fPeer = null;
	}

	/*
	 * Direction is only applicable if it is an interactive gauge
	 * false = left, true = right
	 * Otherwise the hourglass spins and direction is ignored
	 */
	void startGaugeThread(boolean direction) {
		fGaugeThread = new GaugeThread(direction);
		fGaugeThread.start();
	}

	void stopGaugeThread() {
		if (fGaugeThread != null) {
			synchronized (fGaugeThread) {
				fGaugeThread.fRunning = false;
				fGaugeThread.notify();
				fGaugeThread = null;
			}
		}
	}

	void suspendGaugeThread() {
		if (fGaugeThread != null) fGaugeThread.fSuspended = true;
	}

	void resumeGaugeThread() {
		if (fGaugeThread != null && fGaugeThread.fSuspended) {
			fGaugeThread.fSuspended = false;
			synchronized (fGaugeThread) {
				fGaugeThread.notify();
			}
		}
	}

	void newMaxValue(int maxValue) {
		if (!indefiniteGaugeStylesApply()) {
			stopGaugeThread();
			setGaugeValueWidth();
		}
		repaint();
	}

	void incrementAnimationState() {
		if (++fIncrementalAnimationState >= MAX_INCREMENTAL_ANIMATION_STATE) fIncrementalAnimationState = 0;
	}

	void sizeChanged(int w, int h) {
		fGaugeWidth = fWidth - TOTAL_BORDER_WIDTH - 2 * GAUGE_X;
		fHighlightBoxWidth = fWidth;
		fArrow2X = GAUGE_X + fGaugeWidth + BORDER_WIDTH - 1;
		setGaugeValueWidth();
		recalculateHourglassValues();
	}

	void setGaugeValueWidth() {
		if (fGauge.fMax == Gauge.INDEFINITE) {
			fGaugeValueWidth = fGaugeWidth - 2;
		} else if (fGauge.fInteractive) {
			fGaugeValueWidth = ((fGaugeWidth - 2) * fGauge.fValue) / fGauge.fMax;
		} else {
			fGaugeValueWidth = (((fWidth - TOTAL_BORDER_WIDTH - 2 * NON_INTERACTIVE_GAUGE_X) - 1) * fGauge.fValue) / fGauge.fMax;
		}
	}

	protected void paint(Graphics g) {
		int fGaugeYAdd;
		if (fGauge.fLayout == Item.LAYOUT_DEFAULT || (fGauge.fLayout & Item.LAYOUT_VCENTER) == Item.LAYOUT_VCENTER) {
			fGaugeYAdd = (fHeight - GAUGE_HEIGHT - TOTAL_BORDER_HEIGHT) / 2;
		} else if ((fGauge.fLayout & Item.LAYOUT_BOTTOM) > 0) {
			fGaugeYAdd = fHeight - GAUGE_HEIGHT - BORDER_HEIGHT;
		} else {
			fGaugeYAdd = BORDER_HEIGHT;
		}

		g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BACKGROUND_RGB);
		g.fillRect(0, 0, fWidth, fHeight);
		if (fHighlighted) paintBorder(g, true, 0, 0, fHighlightBoxWidth - 1, fHeight - 1);
		g.setColor(DisplayPeer.COLOR_FOREGROUND_RGB);

		if (indefiniteGaugeStylesApply()) {
			if (isContinuous()) {
				loadHourglassImages();
				if (gHourglassLoaded) {
					if (fContinuousAnimationState < 4) {
						g.drawImage(gHourglassImages[fContinuousAnimationState], fHourglassX1, HOURGLASS_Y, 0);
					}
				}
			} else {
				loadIncrementalGaugeImage();
				if (gGaugeImage != null) {
					int offset = fGauge.fValue == Gauge.INCREMENTAL_UPDATING ? (((fWidth - TOTAL_BORDER_WIDTH) * fIncrementalAnimationState) / MAX_INCREMENTAL_ANIMATION_STATE) : 0;
					int imageHeight = gGaugeImage.getHeight();
					int y = (fHeight - imageHeight) / 2;

					g.setClip(BORDER_WIDTH + BORDER_WIDTH, y, fWidth - TOTAL_BORDER_WIDTH - TOTAL_BORDER_WIDTH, imageHeight);
					g.drawImage(gGaugeImage, BORDER_WIDTH + BORDER_WIDTH + offset, y, 0);
					g.setColor(GAUGE_FOREGROUND_RGB);
					g.fillRect(BORDER_WIDTH + BORDER_WIDTH, y, offset, imageHeight);
					g.setColor(GAUGE_BACKGROUND_RGB);
					int imageWidth = gGaugeImage._getWidth();
					g.fillRect(BORDER_WIDTH + BORDER_WIDTH + offset + imageWidth, y, fWidth - (TOTAL_BORDER_WIDTH + TOTAL_BORDER_WIDTH + offset + imageWidth), imageHeight);
					g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BORDER_RGB);
					g.drawRect(BORDER_WIDTH + BORDER_WIDTH, y, fWidth - TOTAL_BORDER_WIDTH - TOTAL_BORDER_WIDTH - 1, imageHeight - 1);
				}
			}
		} else {
			g.setColor(GAUGE_BACKGROUND_RGB);
			int x = 0;
			int y = fGaugeYAdd + GAUGE_Y;
			if (fGauge.fInteractive) {
				x = BORDER_WIDTH + GAUGE_X;
				g._fillRoundRect(x + 1, y, fGaugeWidth - 2, GAUGE_HEIGHT - 1, GAUGE_HEIGHT, GAUGE_HEIGHT);
				g.setColor(GAUGE_FOREGROUND_RGB);
				g._fillRoundRect(x + 1, y + 1, fGaugeValueWidth, GAUGE_VALUE_HEIGHT, GAUGE_VALUE_HEIGHT, GAUGE_HEIGHT);
				g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BORDER_RGB);
				g._drawRoundRect(x, y, fGaugeWidth - 1, GAUGE_HEIGHT - 1, GAUGE_HEIGHT, GAUGE_HEIGHT);

				x = BORDER_WIDTH + ARROW_X - ARROW_WIDTH;
				g._drawRoundRect(x, y, ARROW_WIDTH - 1, ARROW_WIDTH - 1, ARROW_WIDTH - 1, ARROW_WIDTH - 1);
				g.drawLine(x + 2, y + (ARROW_WIDTH / 2), x + ARROW_WIDTH - 3, y + (ARROW_WIDTH / 2));

				x = fArrow2X + BORDER_WIDTH;
				g._drawRoundRect(x + 1, y, ARROW_WIDTH - 1, ARROW_WIDTH - 1, ARROW_WIDTH - 1, ARROW_WIDTH - 1);

				g.drawLine(x + 3, y + (ARROW_WIDTH / 2), x + ARROW_WIDTH - 2, y + (ARROW_WIDTH / 2));
				g.drawLine(x + (ARROW_WIDTH / 2) + 1, y + 2, x + (ARROW_WIDTH / 2) + 1, y + ARROW_WIDTH - 3);
			} else {
				x = BORDER_WIDTH + BORDER_WIDTH;
				int width = fWidth - TOTAL_BORDER_WIDTH - TOTAL_BORDER_WIDTH;
				g._fillRoundRect(x, y, width, GAUGE_HEIGHT - 1, GAUGE_HEIGHT, GAUGE_HEIGHT);
				g.setColor(GAUGE_FOREGROUND_RGB);

				x = BORDER_WIDTH + NON_INTERACTIVE_GAUGE_X;
				g._fillRoundRect(x, y + 1, fGaugeValueWidth, GAUGE_VALUE_HEIGHT, GAUGE_VALUE_HEIGHT, GAUGE_HEIGHT);
				g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BORDER_RGB);
				g._drawRoundRect(x - 1, y, width + 1, GAUGE_HEIGHT - 1, GAUGE_HEIGHT, GAUGE_HEIGHT);
			}
		}
	}

	void recalculateHourglassValues() {
		fHourglassW = CONTINUOUS_GAUGE_ITEM_HEIGHT - 4;
		fHourglassH = fHourglassW;
		fHourglassX = (fWidth - fHourglassW) / 2;

		fHourglassX1 = fHourglassX + (fHourglassW/6);
		fHourglassX2 = fHourglassX + (fHourglassW/3);
		fHourglassX3 = fHourglassX + (fHourglassW/2);
		fHourglassX4 = fHourglassX + (fHourglassW*2/3);
		fHourglass5 = fHourglassX + (fHourglassW*5/6);
		fHourglassX6 = fHourglassX + fHourglassW;

		fHourglassY1 = HOURGLASS_Y + (fHourglassH/6);
		fHourglassY15 = HOURGLASS_Y + (fHourglassH/4);
		fHourglassY2 = HOURGLASS_Y + (fHourglassH/3);
		fHourglassY3 = HOURGLASS_Y + (fHourglassH/2);
		fHourglassY4 = HOURGLASS_Y + (fHourglassH*2/3);
		fHourglassY45 = HOURGLASS_Y + (fHourglassH*3/4);
		fHourglassY5 = HOURGLASS_Y + (fHourglassH*5/6);
		fHourglassY6 = HOURGLASS_Y + fHourglassH;
	}

	boolean indefiniteGaugeStylesApply() {
		return fGauge.fMax == Gauge.INDEFINITE && !fGauge.fInteractive;
	}

	boolean isContinuous() {
		return fGauge.fValue == Gauge.CONTINUOUS_IDLE || fGauge.fValue == Gauge.CONTINUOUS_RUNNING;
	}

	boolean isContinuousRunning() {
		return indefiniteGaugeStylesApply() && fGauge.fValue == Gauge.CONTINUOUS_RUNNING;
	}

	int getMinimumHeight() {
		if (indefiniteGaugeStylesApply() && isContinuous()) {
			return CONTINUOUS_GAUGE_ITEM_HEIGHT;
		} else {
			return GAUGE_ITEM_HEIGHT;
		}
	}

	int getMinimumWidth() {
		return GAUGE_ITEM_WIDTH + TOTAL_BORDER_WIDTH;
	}

	boolean keyPressed(int keyCode) {
		if (!fGauge.fInteractive) return false;
		int oldValue = fGauge.fValue;
		switch (CustomItemPeer.getGameAction(keyCode)) {
			case Canvas.LEFT: fGauge._setValue(fGauge.fValue - 1); break;
			case Canvas.RIGHT: fGauge._setValue(fGauge.fValue + 1); break;
			default: return true; // don't do the check for notifySateChanged
		}
		if (fGauge.fValue != oldValue) fGauge.notifyStateChanged();
		return true;
	}

	boolean keyRepeated(int keyCode) {
		return keyPressed(keyCode);
	}

	boolean pointerPressed(int x, int y) {
		if (fGauge.fInteractive) {
			fOldGaugeValue = fGauge.fValue;
			if (isInLeftArrowField(x, y)) {
				decrementGaugeValue();
				fLeftPressed = true;
				startGaugeThread(false);
			} else if (isInRightArrowField(x, y)) {
				incrementGaugeValue();
				fRightPressed = true;
				startGaugeThread(true);
			} else if (handleOnGaugePress(x, y)) {
				if (fGauge.fValue != fOldGaugeValue) fGauge.notifyStateChanged();
			}
		}
		return true;
	}

	boolean pointerDragged(int x, int y) {
		if (fGauge.fInteractive) {
			if ((fLeftPressed && isInLeftArrowField(x, y)) || (fRightPressed && isInRightArrowField(x, y))) {
				resumeGaugeThread();
			} else if (handleOnGaugePress(x, y)) {
				if (fGauge.fValue != fOldGaugeValue) fGauge.notifyStateChanged();
			} else if (fGaugeThread != null) {
				suspendGaugeThread();
			}
		}
		return true;
	}

	boolean pointerReleased(int x, int y) {
		if (fGauge.fInteractive && fGaugeThread != null) {
			stopGaugeThread();
			fLeftPressed = fRightPressed = false;
			if (fGauge.fValue != fOldGaugeValue) fGauge.notifyStateChanged();
		}
		return true;
	}

	boolean handleOnGaugePress(int x, int y) {
		/* set the value of the gauge if the user pressed inside the gauge itself */
		int gaugeLeft = BORDER_WIDTH + BORDER_WIDTH + ARROW_WIDTH;
		int gaugeRight = fWidth - BORDER_WIDTH - ARROW_WIDTH - BORDER_WIDTH;
		if (x > gaugeRight || x < gaugeLeft) return false;
		/* user pressed inside the gauge */
		int gaugeWidth = gaugeRight - gaugeLeft;
		int halfIncrementWidth = (gaugeWidth / fGauge.fMax) / 2;
		int newGaugeValue = fGauge.fMax * (x + halfIncrementWidth - gaugeLeft) / gaugeWidth;
		fGauge._setValue(newGaugeValue);
		return true;
	}

	boolean isInRightArrowField(int x, int y) {
		return x >= fWidth - BORDER_WIDTH - ARROW_WIDTH - BORDER_WIDTH && x < fWidth - BORDER_WIDTH - BORDER_WIDTH;
	}

	boolean isInLeftArrowField(int x, int y) {
		return x >= BORDER_WIDTH + BORDER_WIDTH && x < BORDER_WIDTH + BORDER_WIDTH + ARROW_WIDTH;
	}

	class GaugeThread extends Thread {
		boolean fRunning = true;
		boolean fSuspended = false;

		/* right is true, left is false */
		boolean fDirection;

		public GaugeThread(boolean direction) {
			fDirection = direction;
		}

		public void run() {
			while (fRunning) {
				try {
					while (!fSuspended) {
						Thread.sleep(300);
						synchronized (Device.gDisplayableLock) {
							if (!fRunning) return;
							if (fSuspended) break;
							if (isContinuousRunning()) {
								if (++fContinuousAnimationState >= 4) fContinuousAnimationState = 0;
								repaint();
							} else {
								if (fDirection) {
									incrementGaugeValue();
								} else {
									decrementGaugeValue();
								}
							}
						}
					}
					synchronized (this) {
						wait();
					}
				} catch (InterruptedException e1) {
				}
			}
		}
	}

	void incrementGaugeValue() {
		fGauge._setValue(fGauge.fValue + 1);
	}

	void decrementGaugeValue() {
		fGauge._setValue(fGauge.fValue - 1);
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		/*
		 * The Gauge should display its full contents
		 * when given focus.
		 */
		visibleRectangle[0] = 0;
		visibleRectangle[1] = 0;
		visibleRectangle[2] = viewWidth;
		visibleRectangle[3] = viewHeight;

		if (!hasFocus()) {
			fHighlighted = true;
			repaint();
			return true;
		}

		/*
		 * This is a subsequent traversal event.
		 */
		if (fGauge.fInteractive) {
			switch (direction) {
				case Canvas.RIGHT: keyPressed(Canvas.KEYCODE_RIGHT); return true;
				case Canvas.LEFT: keyPressed(Canvas.KEYCODE_LEFT); return true;
			}
		}

		return false;
	}

	void traverseOut() {
		fHighlighted = false;
		super.traverseOut();
	}

	int getPreferredWidth() {
		int dWidth = getWindow().fContentComponent.fWidth;
		ItemComponent c = (ItemComponent) fParent;
		if (c.fLabelComponent != null) {
			int remainingPixels = dWidth - c.fLabelComponent.fWidth - ItemComponentLayout.LABEL_RIGHT_MARGIN;
			if (remainingPixels > getMinimumWidth()) {
				return remainingPixels;
			}
		}
		return dWidth;
	}

	int getType() {
		return Item.TYPE_GAUGE;
	}

	boolean isEditable() {
		return fGauge.fInteractive;
	}

	static void loadHourglassImages() {
		/* We only try to load the images once. If it fails we'll fall back to the drawn version */
		if (!gHourglassLoadAttempted) {
			gHourglassLoadAttempted = true;
			gHourglassImages = new Image[4];
			int i = 0;
			try {
				for (i = 0; i < gHourglassImages.length; i++) {
					gHourglassImages[i] = Image.createImage(CONTINUOUS_GAUGE_FILE + (i + 1) + ".png"); //$NON-NLS-1$
				}
				gHourglassLoaded = true;
			} catch (IOException ioe) {
				/* silent fail */
				for (int j = 0; j < i; j++) {
					/* Images will be disposed of later using WeakReference */
					gHourglassImages[j] = null;
				}
			}
		}
	}

	static void loadIncrementalGaugeImage() {
		/* We only try to load the image once. If it fails we'll fall back to the drawn version */
		if (!gGaugeImageLoadAttempted) {
			gGaugeImageLoadAttempted = true;
			try {
				gGaugeImage = Image.createImage(INCREMENTAL_GAUGE_FILE);
			} catch (IOException ioe) {
				/* silent fail */
			}
		}
	}

	void update() {
		int value = fGauge.fValue;
		if (indefiniteGaugeStylesApply()) {
			if (value == Gauge.CONTINUOUS_RUNNING) {
				if (fGaugeThread == null || !fGaugeThread.fRunning) startGaugeThread(false);
			} else {
				stopGaugeThread();
			}
		} else {
			setGaugeValueWidth();
		}
		if (getMinimumHeight() > fHeight) {
			invalidate();
		} else {
			repaint();
		}
	}
}
