package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

class ImageItemPeer extends Component implements IButtonListener {

	ImageItem fImageItem;
	static final int BORDER_WIDTH = gBorder.getWidth(Item.TYPE_IMAGE);
	static final int BORDER_HEIGHT = gBorder.getHeight(Item.TYPE_IMAGE);
	static final int TOTAL_BORDER_WIDTH = 2 * BORDER_WIDTH;
	static final int TOTAL_BORDER_HEIGHT = 2 * BORDER_HEIGHT;

	static final int MARGIN = 4;
	static final int TOTAL_MARGIN = 8;
	static int ARC = 8;

	boolean fInteractive;
	boolean fPressed;
	boolean fHighlighted;

	int fImageX;
	int fImageY;
	ImageButton fImageButton;

	public ImageItemPeer(Composite parent, ImageItem item) {
		super(parent);
		fImageItem = item;
		fImageItem.fPeer = this;

		fInteractive = item.fDefaultCommand != null && item.fCommandListener != null;

		if (usesButton()) {
			fImageButton = new ImageButton(parent, item.fImage, null);
			fImageButton.setListener(this);
			fImageButton.fHeight = getMinimumHeight();
			fImageButton.fWidth = getMinimumWidth();
			fImageButton.setEnabled(fInteractive);
		}
	}

	boolean usesButton() {
		return fImageItem.fAppearanceMode == Item.BUTTON
			|| fImageItem.fAppearanceMode == Item.PLAIN && fImageItem.fDefaultCommand != null;
	}

	void dispose() {
		super.dispose();

		fImageItem.fPeer = null;
	}

	public void setImage(Image image) {
		if (usesButton()) fImageButton.fImage = image;
		invalidate();
	}

	protected void paint(Graphics g) {
		if (fImageItem.fImage == null) return;

		if (usesButton()) {
			fImageButton.paint(g);
			return;
		}

		boolean hasFocus = hasFocus();
		g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BACKGROUND_RGB);
		g.fillRect(0, 0, fWidth, fHeight);

		if (fImageItem.fAppearanceMode == Item.HYPERLINK){
			if (fHighlighted) {
				g.setColor(DisplayPeer.COLOR_HIGHLIGHTED_BACKGROUND_RGB);
				int width = fWidth - TOTAL_BORDER_WIDTH;
				int height = fHeight - TOTAL_BORDER_HEIGHT;
				g.fillRect(BORDER_WIDTH, BORDER_HEIGHT, width, height);
			}
			if (hasFocus && fInteractive) paintBorder(g, true, 0, 0, fWidth - 1, fHeight - 1);
		}
		g.drawImage(fImageItem.fImage, fImageX, fImageY, 0);
	}

	int getMinimumHeight() {
		if (fImageItem == null || fImageItem.fImage == null) return 0;
		return TOTAL_MARGIN + fImageItem.fImage._getHeight() + TOTAL_BORDER_HEIGHT;
	}

	int getMinimumWidth() {
		if (fImageItem == null || fImageItem.fImage == null) return 0;
		return TOTAL_MARGIN + fImageItem.fImage._getWidth() + TOTAL_BORDER_WIDTH;
	}

	int getType() {
		return Item.TYPE_IMAGE;
	}

	boolean keyPressed(int keyCode) {
		if (CanvasPeer.getGameAction(keyCode) == Canvas.FIRE && fInteractive) {
			fPressed = fHighlighted = true;
			if (fImageItem.fAppearanceMode == Item.HYPERLINK) {
				fImageItem.fireCommand(fImageItem.fDefaultCommand);
				repaint();
			} else if (usesButton()) {
				fImageButton.keyPressed(keyCode);
			}
			return true;
		}
		return false;
	}

	boolean keyReleased(int keyCode) {
		if (CanvasPeer.getGameAction(keyCode) == Canvas.FIRE && fInteractive) {
			fPressed = fHighlighted = false;
			if (fImageItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fImageButton.keyReleased(keyCode);
			}
			return true;
		}
		return false;
	}

	boolean pointerPressed(int x, int y) {
		if (fInteractive) {
			fPressed = fHighlighted = true;
			if (fImageItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fImageButton.pointerPressed(x, y);
			}
			return true;
		}
		return false;
	}

	boolean pointerDragged(int x, int y) {
		if (!fInteractive) return false;
		if (fPressed != contains(x, y)) {
			fPressed = !fPressed;
			fHighlighted = fPressed;
			if (fImageItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fImageButton.pointerDragged(x, y);
			}
			return true;
		}
		return false;
	}

	boolean pointerReleased(int x, int y) {
		if (fPressed) {
			fImageItem.fireCommand(fImageItem.fDefaultCommand);
			fPressed = fHighlighted = false;
			if (fImageItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fImageButton.pointerReleased(x, y);
			}
			return true;
		}
		return false;
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		if (usesButton()) {
			fImageButton.traverse(direction, viewWidth, viewHeight, visibleRectangle, x, y);
		}

		/*
		 * If the whole item is visible:
		 * - If the item either has focus or is not interactive, return false.
		 * - Otherwise repaint to show focus.
		 */
		if (visibleRectangle[2] == fWidth && visibleRectangle[3] == fHeight) {
			if (hasFocus())	return false;
			if (fInteractive) {
				fHighlighted = true;
				repaint();
				return true;
			} else if (fImageItem.fCommands != null) {
				return true;
			}
			return false;
		}

		if (!hasFocus()) {
			switch (direction) {
				case Canvas.LEFT:
				case Canvas.UP:
					if (visibleRectangle[2] < viewWidth) {
						visibleRectangle[2] = viewWidth;
					}
					if (visibleRectangle[3] < viewHeight) {
						visibleRectangle[3] = viewHeight;
					}
					visibleRectangle[0] = fWidth - viewWidth;
					visibleRectangle[1] = fHeight - viewHeight;

					if (fInteractive) fHighlighted = true;

					return true;
				case Canvas.DOWN:
				case Canvas.RIGHT:
				case CustomItem.NONE:
					if (visibleRectangle[2] < viewWidth) {
						visibleRectangle[2] = viewWidth;
					}
					if (visibleRectangle[3] < viewHeight) {
						visibleRectangle[3] = viewHeight;
					}
					visibleRectangle[0] = 0;
					visibleRectangle[1] = 0;

					if (fInteractive) fHighlighted = true;

					return true;
			}
		}

		switch (direction) {
			case Canvas.LEFT:
			case Canvas.UP:
				/*
				 * If the upper left corner of the rectangle is at zero, the
				 * rectangle can't be scrolled anymore.
				 */
				if (visibleRectangle[1] == 0) return false;
				visibleRectangle[1] = Math.max(visibleRectangle[1] - viewHeight / 2, 0);
				visibleRectangle[3] = viewHeight;
				return true;
			case Canvas.RIGHT:
			case Canvas.DOWN:
				/*
				 * If the bottom of the rectangle is the viewHeight, the
				 * rectangle can't be scrolled anymore.
				 */
				if (visibleRectangle[1] + visibleRectangle[3] == fHeight) {
					return false;
				}
				visibleRectangle[1] = Math.min(visibleRectangle[1] + viewHeight / 2,
												   fHeight - viewHeight);
				visibleRectangle[3] = viewHeight;

				return true;
			case CustomItem.NONE:
				return true;
		}
		return false;
	}

	void traverseOut() {
		fHighlighted = false;
		if (usesButton()) fImageButton.traverseOut();
		super.traverseOut();
	}

	void updateCommands() {
		fInteractive = fImageItem.fDefaultCommand != null && fImageItem.fCommandListener != null;
		if (usesButton()) {
			fImageButton.setEnabled(fInteractive);
		}
	}

	void sizeChanged(int w, int h) {
		Image image = fImageItem.fImage;

		if (image == null) {
			fImageX = 0;
			fImageY = 0;
			return;
		}

		int width = image.getWidth();
		int height = image.getHeight();

		if ((fImageItem.fLayout & Item.LAYOUT_CENTER) == Item.LAYOUT_CENTER) {
			fImageX = (fWidth - TOTAL_BORDER_WIDTH - width) / 2;
		} else if ((fImageItem.fLayout & Item.LAYOUT_RIGHT) > 0) {
			fImageX = fWidth - width - BORDER_WIDTH;
		} else {
			fImageX = BORDER_WIDTH;
		}

		if ((fImageItem.fLayout & Item.LAYOUT_VCENTER) == Item.LAYOUT_VCENTER) {
			fImageY = (fHeight - TOTAL_BORDER_HEIGHT - height) / 2;
		} else if ((fImageItem.fLayout & Item.LAYOUT_BOTTOM) > 0) {
			fImageY = fHeight - height - BORDER_HEIGHT;
		} else {
			fImageY = BORDER_HEIGHT;
		}
		if (usesButton()) fImageButton.setBounds(fX, fY, fImageButton.fWidth, fImageButton.fHeight);
	}

	boolean isEditable() {
		return fInteractive;
	}

	boolean isHyperlink() {
		return fImageItem.fAppearanceMode == Item.HYPERLINK;
	}

	public void buttonFired(ButtonWidget button) {
		fImageItem.fireCommand(fImageItem.fDefaultCommand);
	}
}
