package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;

/**
 * <code>Item</code> presents the behaviour common to
 * the objects that can be inserted into a <code>Form</code>,
 * such as <code>ChoiceGroup</code>,<code>DateField</code>,
 * <code>Gauge</code>, <code>ImageItem</code>,
 * <code>StringItem</code>, or <code>TextField</code>.<p>
 * All <code>Item</code> subclasses are preceeded by a label.
 *
 * @see Form
 */
public abstract class Item {

	// The following are for quick lookup of type without instanceof checks
	final static int TYPE_UNUSED	  = -1; // for component class
	final static int TYPE_STRING      = 0;
	final static int TYPE_IMAGE       = 1;
	final static int TYPE_SPACER      = 2;
	final static int TYPE_CUSTOM      = 3;
	final static int TYPE_CHOICEGROUP = 4;
	final static int TYPE_DATEFIELD   = 5;
	final static int TYPE_TEXTFIELD   = 6;
	final static int TYPE_GAUGE       = 7;
	final static int TYPE_ITEMCOMMAND = 8;
	final static int TYPE_DISPLAYABLECOMMAND = 9;
	final static int TYPE_BUTTON = 10;
	final static int TYPE_LABEL = 11;

	public final static int PLAIN = 0;
	public final static int BUTTON = 2;
	public final static int HYPERLINK = 1;
	public final static int LAYOUT_2 = 0x4000;
	public final static int LAYOUT_BOTTOM = 0x20;
	public final static int LAYOUT_CENTER = 3;
	public final static int LAYOUT_DEFAULT = 0;
	public final static int LAYOUT_EXPAND = 0x800;
	public final static int LAYOUT_LEFT = 1;
	public final static int LAYOUT_NEWLINE_AFTER = 0x200;
	public final static int LAYOUT_NEWLINE_BEFORE = 0x100;
	public final static int LAYOUT_RIGHT = 2;
	public final static int LAYOUT_SHRINK = 0x400;
	public final static int LAYOUT_TOP = 0x10;
	public final static int LAYOUT_VCENTER = 0x30;
	public final static int LAYOUT_VEXPAND = 0x2000;
	public final static int LAYOUT_VSHRINK = 0x1000;

	static final int ALL_LAYOUT_DIRECTIVES =
		LAYOUT_2 | LAYOUT_BOTTOM | LAYOUT_CENTER | LAYOUT_DEFAULT |
		LAYOUT_EXPAND | LAYOUT_LEFT | LAYOUT_NEWLINE_AFTER |
		LAYOUT_NEWLINE_BEFORE | LAYOUT_RIGHT | LAYOUT_SHRINK | LAYOUT_TOP |
		LAYOUT_VCENTER | LAYOUT_VEXPAND | LAYOUT_VSHRINK;
	static final int ALL_LAYOUT_DIRECTIVES_INVERTED = ~ALL_LAYOUT_DIRECTIVES;
	static final int HLAYOUT = LAYOUT_LEFT | LAYOUT_CENTER | LAYOUT_RIGHT;

	String fLabel;

	// Items can be added to a form or an alert, and can only be attached to
	// one at a time.  This can be null if the item is not currently attached
	// to any form
	Screen fScreen;

	// The commands for this item.  The field will be null if no commands
	// are currently associated with this item.
	FastVector fCommands;

	// The default command for this item.  If no default command has been
	// set than this field might be null
	Command fDefaultCommand;

	// The current ItemCommandListener.  This field may be null if no listener
	// has been set
	ItemCommandListener fCommandListener;

	// The prefered width and height of this Item
	int fPreferredWidth;
	int fPreferredHeight;

	// The layout flags for this Item
	int fLayout;

	boolean fIsMediaItem = false;

	Item(String label) {
		fLabel = label;
		fPreferredWidth = -1;
		fPreferredHeight = -1;
	}

	abstract ItemComponent getItemComponent();

	public String getLabel() {
		synchronized (Device.gDisplayableLock) {
			return fLabel;
		}
	}

	public void setLabel(String label) {
		synchronized (Device.gDisplayableLock) {
			// Labels are not allowed on items in an Alert
			checkForAlert();

			// exit if nothing has changed
			if ((fLabel!=null && fLabel.equals(label)) || (fLabel==null&&label==null)) return;
			fLabel=label;

			// Keep peer in sync with label changes
			ItemComponent peer = getItemComponent();
			if (peer != null) peer.setLabel(label);
		}
	}

	void forceLayout() {
		ItemComponent peer = getItemComponent();
		if (peer != null) peer.invalidate();
	}

	public void addCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			// Commands cannot be added to items that are contained in an Alert
			checkForAlert();

			// The command cannot be null
			if (command == null) throw new NullPointerException(MidpMsg.getString("Item.addCommand.NullCommand")); //$NON-NLS-1$
			if (fCommands == null) fCommands = new FastVector();

			// Adding the same command to an item twice should do nothing
			if (fCommands.contains(command)) return;
			insertCommandSorted(command);
			syncCommandsWithPeer();
		}
	}

	public int getLayout() {
		synchronized (Device.gDisplayableLock) {
			return fLayout;
		}
	}

	public int getMinimumHeight() {
		synchronized (Device.gDisplayableLock) {
			ItemComponent peer = getItemComponent();
			if (peer != null) return peer.getMinimumHeight();
			return 0;
		}
	}

	public int getMinimumWidth() {
		synchronized (Device.gDisplayableLock) {
			ItemComponent peer = getItemComponent();
			if (peer != null) return peer.getMinimumWidth();
			return 0;
		}
	}

	public int getPreferredHeight() {
		synchronized (Device.gDisplayableLock) {
			if (fPreferredHeight != -1) return fPreferredHeight;

			ItemComponent peer = getItemComponent();
			return peer != null ? peer.getPreferredHeight() : 0;
		}
	}

	public int getPreferredWidth() {
		synchronized (Device.gDisplayableLock) {
			if (fPreferredWidth != -1) return fPreferredWidth;

			ItemComponent peer = getItemComponent();
			return peer != null ? peer.getPreferredWidth() : 0;
		}
	}

	public void removeCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			if (command == null || fCommands == null) return;

			// If the command removed is the default command, then the default
			// command should be reset.
			if (command == fDefaultCommand) fDefaultCommand = null;

			int index = fCommands.indexOf(command);
			if (index != -1) {
				fCommands.removeElementAt(index);
				if (fCommands.size() == 0) fCommands = null; // GC the unused FastVector

				syncCommandsWithPeer();
			}
		}
	}

	public void setItemCommandListener(ItemCommandListener listener) {
		synchronized (Device.gDisplayableLock) {
			checkForAlert();
			fCommandListener = listener;
			syncCommandsWithPeer();
		}
	}

	void checkValidity(int aLayout) {
		if ((aLayout & ALL_LAYOUT_DIRECTIVES_INVERTED) != 0)
			throw new IllegalArgumentException(MidpMsg.getString("Item.setLayout.LayoutFlags")); //$NON-NLS-1$
	}

	public void setLayout(int layout) {
		synchronized (Device.gDisplayableLock) {
			checkForAlert();

			// The layout flag can only be a Bitwise combination of all the
			// valid layout flags
			checkValidity(layout);
			fLayout = layout;

			// Keep the peer in sync
			if (fScreen != null && fScreen.fPeer != null) ((FormPeer) fScreen.fPeer).scheduleUpdate(FormPeer.LAYOUT_CHANGED);
		}
	}

	public void setPreferredSize(int width, int height) {
		synchronized (Device.gDisplayableLock) {
			checkForAlert();
			if (width < -1 || height < -1)
				throw new IllegalArgumentException(MidpMsg.getString("Item.setPreferredSize.WidthHeight")); //$NON-NLS-1$

			if (fPreferredHeight != height || fPreferredWidth != width) {
				fPreferredHeight = height;
				fPreferredWidth = width;
				// Keep peer in sync
				// TODO: this could be done better if every method is implemented
				// to only layout the items below this item
				forceLayout();
			}
		}
	}

	public void notifyStateChanged() {
		synchronized (Device.gDisplayableLock) {
			if ((fScreen == null) || (fScreen.getDisplayableType() != Displayable.TYPE_FORM))
				throw new IllegalStateException(MidpMsg.getString("Item.notifyStateChanged.FormItems")); //$NON-NLS-1$

			/*
			 * According to the MIDP 2.0 Javadoc for Item.notifyStateChanged(),
			 * this method does not block waiting for itemStateChanged() to complete.
			 */
			final Item item = this;
			Device.postRunnable(new Runnable() {
				public void run() {
					((Form)fScreen).callItemStateChanged(item);
				}
			});
		}
	}

	public void setDefaultCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			checkForAlert();
			if (fDefaultCommand == command) return;

			Command oldDefault = fDefaultCommand;
			fDefaultCommand = command;

			// Non-null default commands should be added to the item as regular
			// commands also
			if (fDefaultCommand != null) {
				// calling addCommand will perform some redundant checking
				if (fCommands == null) fCommands = new FastVector();
				if (fCommands.contains(command)) {
					// remove it
					fCommands.removeElement(command);
				}
				// place the default command at the front of the list
				fCommands.insertElementAt(command,0);
				// If there was already a default command present, it should
				// be removed from the list and re-inserted.
				if (oldDefault != null) {
					fCommands.removeElementAt(1);
					insertCommandSorted(oldDefault);
				}
				syncCommandsWithPeer();
			}
		}
	}

	/*
	 * Insert a command into the FastVector, sorting based on
	 * its status as default and its priority.
	 */
	private void insertCommandSorted(Command command) {
		// Add the item into the list so that the Commands are always
		// sorted and resorting does not need to happen everytime the commands
		// should be displayed
		int pos = 0;
		while (pos < fCommands.size()) {
			Command tempCommand = (Command)fCommands.elementAt(pos);

			// The default command should always be at the front of the list,
			// but we cannot change its priority directly.
			if (tempCommand == fDefaultCommand) {
				pos++;
				continue;
			}

			if (tempCommand.getPriority() > command.getPriority()) break;
			pos++;
		}
		fCommands.insertElementAt(command, pos);
	}

	private void syncCommandsWithPeer() {
		if (getItemComponent() != null) getItemComponent().updateCommands();
	}

	Screen getScreen() {
		return fScreen;
	}

	void setScreen(Screen screen) {
		fScreen = screen;
	}

	abstract int getType();

	void checkForAlert() {
		if ((fScreen != null) && (fScreen.getDisplayableType() == Displayable.TYPE_ALERT))
			throw new IllegalStateException();
	}

	int getHAlignment() {
		return fLayout & HLAYOUT;
	}

	void fireCommand(final Command command) {
		final Item item = this;
		Device.postRunnable(new Runnable() {
			public void run() {
				if (fCommandListener != null) {
					try {
						fCommandListener.commandAction(command, item);
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}
			}
		});
	}

	void checkAppearanceMode(int appearanceMode) {
		switch (appearanceMode) {
			case PLAIN: return;
			case BUTTON: return;
			case HYPERLINK: return;
			default: throw new IllegalArgumentException(MidpMsg.getString("Item.checkAppearanceMode.invalid_mode", appearanceMode)); //$NON-NLS-1$
		}
	}
}
