package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import java.util.Hashtable;
import com.ibm.ive.midp.*;

class MenuBarComponent extends Widget {

	static final int ID_PPC = 100;
	static final int ID_SP_BM = 103;
	static final int ID_SP_BB = 105;

	static final int ID_SP_LEFT = 106;
	static final int ID_SP_RIGHT = 107;
	static final int ID_START = 108;
	Menu fMenuData[] = new Menu[4];
	int fMenuCount = 0;
	static Menu[] fMenuRegistry;
	Hashtable fCommandLookup = new Hashtable(10);
	Hashtable fItemCommandLookup = new Hashtable(10);
	int fHandle = 0;
	Menu fItemMenu;
	Item fItemMenuItem;
	int fItemMenuPosition = -1;
	FastVector fCommands;
	FastVector fItemCommands;

	//final static int WIN32_MENU_HEIGHT = OS.GetMenuBarSize();
	static int gWin32MenuHeight = -1;

	public MenuBarComponent(DisplayablePeer peer) {
		super(peer);
	}

	void dispose() {
		super.dispose();

		if (fHandle != 0) {
			Device.syncExec(new Runnable() {
				public void run() {
					int windowHandle = getDisplayablePeer().getWindowHandle();
					for (int i = 0; i < fMenuData.length; i++) {
						if (fMenuData[i] != null) fMenuData[i].dispose();
					}
					if (fItemMenu != null) {
						OS.RemoveMenu(fHandle, fItemMenuPosition, OS.MF_BYPOSITION);
					}
					if (OS.IsWinCE) {
						OS.DestroyWindow(fHandle);
					} else {
						OS.DestroyMenu(fHandle);
					}
					OS.DrawMenuBar(windowHandle);
					fHandle = 0;
				}
			});
		}
	}

	boolean isShowing() {
		return false;
	}

	int getMinimumWidth() {
		return 0;
	}

	int getMinimumHeight() {
		if (OS.IsWinCE) return 0;
		if (fMenuCount == 0) return getMenuHeight();
		return 0;
	}

	static int getMenuHeight() {
		if (gWin32MenuHeight == -1) {
			Device.syncExec(new Runnable() {
				public void run() {
				  gWin32MenuHeight = OS.GetMenuBarSize();
				}
			});
		}
		return gWin32MenuHeight;
	}

	int getMenuBarId() {
		if (!OS.IsWinCE) return 0; // win32 does not use the id

		if (!OS.usesSoftKeys) return ID_PPC;

		int numberOfCommands = 0;
		if (fCommands != null) numberOfCommands += fCommands.size();
		if (fItemMenuItem != null && fItemMenuItem.fCommands != null) numberOfCommands += fItemMenuItem.fCommands.size();

		if (numberOfCommands <= 2) {
			return ID_SP_BB;
		} else {
			return ID_SP_BM;
		}
	}

	void createSmartphoneMenus() {
		int hMenu = OS.SendMessageW(fHandle, OS.SHCMBM_GETSUBMENU, 0, ID_SP_RIGHT);
		while (OS.RemoveMenu(hMenu, 0, OS.MF_BYPOSITION) != 0);

		FastVector allCommands = new FastVector();
		if (fCommands != null) allCommands.addAll(fCommands);
		if (fItemCommands != null || fItemCommands.size() != 0) allCommands.addAll(fItemCommands);

		if (allCommands.size() > 0) {
			Command command = (Command) allCommands.elementAt(0);
			String label = command.fLongLabel;
			if (label == null) label = command.fLabel;
			OS.SetButtonInfo(fHandle, ID_SP_LEFT, label);

			if (allCommands.size() == 1 && fCommands.size() == 0) {
				OS.SetButtonInfo(fHandle, ID_SP_RIGHT, "");
				return;
			}
			if (allCommands.size() == 2) {
				command = (Command) allCommands.elementAt(1);
				label = command.fLongLabel;
				if (label == null) label = command.fLabel;
				OS.SetButtonInfo(fHandle, ID_SP_RIGHT, label);
				return;
			}
		} else {
			OS.SetButtonInfo(fHandle, ID_SP_LEFT, ""); //$NON-NLS-1$
		}

		if (fMenuData[Menu.ACTIONS] != null) {
			Menu menu = fMenuData[Menu.ACTIONS];
			OS.SetButtonInfo(fHandle, ID_SP_RIGHT, menu.fMenuText);
			for (int i = 0; i < menu.fMenuItemCount; i++) {
				Command command = menu.fMenuItems[i];
				String label = command.fLongLabel;
				if (label == null) label = command.fLabel;
				OS.InsertMenu(hMenu, i, OS.MF_BYPOSITION, registerCommand(menu.fID, i, command), label);
			}
		} else {
			OS.SetButtonInfo(fHandle, ID_SP_RIGHT, ""); //$NON-NLS-1$
		}
	}

	void createMenus() {
		final MenuBarComponent menuBar = this;

		Device.syncExec(new Runnable() {
			public void run() {
				int windowHandle = getDisplayablePeer().getWindowHandle();
				OS.SendMessageW(windowHandle, OS.WM_SETREDRAW, 0, 0);
				dispose();

				fHandle = OS.createMenuBar(windowHandle, getMenuBarId());

				if (OS.usesSoftKeys) {
					createSmartphoneMenus();
				} else {
					int index = 0;
					for (int i = 0; i < fMenuData.length; i++) {
						Menu menu = fMenuData[i];
						if (menu == null) continue;
						addMenu(menu);
						menu.create(menuBar);
						if (OS.IsWinCE) { // WM2003 PPC, DellAxim X51 & X51v devices
							try {
								OS.AddSubMenu(fHandle, menu.fHandle, index++, menu.fID, menu.fMenuText);
							} catch (Throwable t) {
								t.printStackTrace();
							}
						} else {
							OS.InsertMenu(fHandle, i, OS.MF_BYPOSITION | OS.MF_POPUP, menu.fHandle, menu.fMenuText);
						}
					}
				}
				OS.SendMessageW(windowHandle, OS.WM_SETREDRAW, 1, 0);
				OS.DrawMenuBar(windowHandle);
			}
		});
	}

	void dispatchMenuSelection(Event e) {
		if (OS.usesSoftKeys) {
			int totalCommands = fCommands.size();
			if (totalCommands == 0 && (fItemMenuItem == null || fItemMenuItem.fCommands == null || fItemMenuItem.fCommands.size() == 0)) return;
			if (e.fScrollOrSelectionIndex == ID_SP_LEFT) {
				Command c;
				/*
				 * If there are one or more Displayable Commands then
				 * place the first one in the left menu. If not, if
				 * there are one or more Item Commands then place the
				 * first Item Command in the menu.
				 */
				if (totalCommands > 0) {
					c = (Command) fCommands.elementAt(0);
					getDisplayablePeer().dispatchCommand(c);
				} else {
					c = (Command) fItemMenuItem.fCommands.elementAt(0);
					fItemMenuItem.fireCommand(c);
				}
				return;
			} else if (e.fScrollOrSelectionIndex == ID_SP_RIGHT) {
				if (totalCommands < 2) {
					int totalItemCommands = fItemMenuItem == null ? 0 : fItemMenuItem.fCommands.size();
					if (totalItemCommands > 0) {
						Command c = null;
						/*
						 * If there is no Displayable Command then the second Item Command,
						 * if available, is placed in right menu. If there is one Displayable
						 * Command then the first Item Command, if available, is placed in
						 * the right menu.
						 */
						if (totalCommands == 0 && totalItemCommands > 1) {
							c = (Command) fItemMenuItem.fCommands.elementAt(1);
						} else if (totalCommands == 1 && totalItemCommands == 1) {
							c = (Command) fItemMenuItem.fCommands.elementAt(0);
						}
						if (c != null) fItemMenuItem.fireCommand(c);
					}
				} else {
					Command c = (Command) fCommands.elementAt(1);
					getDisplayablePeer().dispatchCommand(c);
				}
				return;
			}
			getDisplayablePeer().repaint();
		}
		Command c = (Command) fCommandLookup.get(new Integer(e.fScrollOrSelectionIndex));
		if (c == null) c = (Command) fItemCommandLookup.get(new Integer(e.fScrollOrSelectionIndex));
		if (c == null) return;

		if (fCommands != null) {
			for (int i = 0; i < fCommands.size(); i++) {
				if (c == fCommands.elementAt(i)) {
					getDisplayablePeer().dispatchCommand(c);
					return;
				}
			}
		}
		/* Not a displayable command, fire as an item command. */
		fItemMenuItem.fireCommand(c);
	}

	void showNotify() {
	}

	void hideNotify() {
	}

	void setCommands(FastVector displayableCommands, FastVector itemCommands) {
		/*
		 * If there is no change in the displayablecommands or itemCommands
		 * there is no need to make any changes to the menubar.
		 */
		if (displayableCommands.equals(fCommands) && itemCommands.equals(fItemCommands)) return;

		fCommands = displayableCommands;
		fItemCommands = itemCommands;
		int size = displayableCommands.size();
		fMenuData = new Menu[4];
		fMenuCount = 0;
		fCommandLookup.clear();

		/* Sort commands into menus */
		if (OS.usesSoftKeys) {
			for (int i = 1; i < size; i++) {
				Command c = (Command) displayableCommands.elementAt(i);
				if (fMenuData[Menu.ACTIONS] == null) {
					fMenuData[Menu.ACTIONS] = new Menu(this, Menu.ACTIONS);
					fMenuCount++;
				}
				fMenuData[Menu.ACTIONS].addCommand(c);
			}
			if (fItemMenuItem != null && fItemMenuItem.fCommands != null) {
				int i = 0;
				if (fItemCommands.size() == 0) i = 1;
				size = fItemCommands.size();
				for (; i < size; i++) {
					Command c = (Command) fItemCommands.elementAt(i);
					if (fMenuData[Menu.ACTIONS] == null) {
						fMenuData[Menu.ACTIONS] = new Menu(this, Menu.ACTIONS);
						fMenuCount++;
					}
					fMenuData[Menu.ACTIONS].addCommand(c);
				}
			}
		} else {
			for (int i = 0; i < size; i++) {
				Command c = (Command)displayableCommands.elementAt(i);
				switch (c.getCommandType()) {
					case Command.OK:
					case Command.SCREEN:
					case Command.EXIT:
					case Command.ITEM:
						if (fMenuData[Menu.ACTIONS] == null) {
							fMenuData[Menu.ACTIONS] = new Menu(this, Menu.ACTIONS);
							fMenuCount++;
						}
						fMenuData[Menu.ACTIONS].addCommand(c);
						break;

					case Command.HELP:
						if (fMenuData[Menu.HELP] == null) {
							fMenuData[Menu.HELP] = new Menu(this, Menu.HELP);
							fMenuCount++;
						}
						fMenuData[Menu.HELP].addCommand(c);
						break;

					case Command.BACK:
					case Command.CANCEL:
					case Command.STOP:
						if (fMenuData[Menu.NAVIGATION] == null) {
							fMenuData[Menu.NAVIGATION] = new Menu(this, Menu.NAVIGATION);
							fMenuCount++;
						}
						fMenuData[Menu.NAVIGATION].addCommand(c);
						break;
				}
			}
		}

		createMenus();
		if (!OS.usesSoftKeys && fItemMenuItem != null) updateItemMenu(fItemMenuItem);
	}

	static void addMenu(Menu menu) {
		if (fMenuRegistry == null) fMenuRegistry = new Menu[12];
		for (int i = 0; i < fMenuRegistry.length; i++) {
			if (fMenuRegistry[i] == null) {
				menu.fID = i + ID_START;
				fMenuRegistry[i] = menu;
				return;
			}
		}
		menu.fID = fMenuRegistry.length + ID_START;
		Menu[] newItems = new Menu[fMenuRegistry.length + 12];
		newItems[fMenuRegistry.length] = menu;
		System.arraycopy(fMenuRegistry, 0, newItems, 0, fMenuRegistry.length);
		fMenuRegistry = newItems;
	}

	static void removeMenu(Menu menu) {
		if (fMenuRegistry == null) return;
		fMenuRegistry[menu.fID - ID_START] = null;
		menu.fID = -1;
	}

	/**
	 * Generates a unique id based on submenu index and item index
	 *
	 * @param menuIndex the index of the submenu
	 * @param itemIndex the index of the item
	 * @return the id that should be set on the native widget for this command
	 */
	int registerCommand(int menuIndex, int itemIndex, Command command) {
		/*
		 * WinCE: Commands with id = 0 will not generate WM_COMMAND messages.
		 */
		itemIndex += 1;
		int id = ((menuIndex & 0x00FF) << 8) + (itemIndex & 0x00FF);

		// Store each command in the hashtable by a short value
		// since each command can only have a 16 bit id in windows
		if (fItemMenu != null && menuIndex == fItemMenu.fID) {
			fItemCommandLookup.put(new Integer(id), command);
		} else {
			fCommandLookup.put(new Integer(id), command);
		}
		return id;
	}

	Command getCommand(int id) {
		return (Command) fCommandLookup.get(new Integer(id));
	}

	void updateItemMenu(final Item item) {
		if (OS.usesSoftKeys) {
			fItemMenuItem = item;
			if (item == null || item.fCommands == null) {
				setCommands(fCommands, new FastVector());
			} else {
				setCommands(fCommands, item.fCommands);
			}
			return;
		}
		if (item != null && item.fCommands != null) {
			final MenuBarComponent menuBar = this;
			Device.syncExec(new Runnable() {
				public void run() {
					DisplayPeer display = getDisplayablePeer().fDisplay.fPeer;
					fItemMenuItem = item;
					if (fHandle == 0) fHandle = OS.createMenuBar(getDisplayablePeer().getWindowHandle(), getMenuBarId());

					fItemMenu = new Menu(menuBar, Menu.ITEM);
					for (int i = 0; i < item.fCommands.size(); i++) {
						fItemMenu.addCommand((Command)item.fCommands.elementAt(i));
					}
					addMenu(fItemMenu);
					fItemMenu.create(menuBar);
					if (OS.IsWinCE) { // WM2003 PPC, DellAxim X51 & X51v devices
						try {
							OS.AddSubMenu(fHandle, fItemMenu.fHandle, fMenuCount,fItemMenu.fID, fItemMenu.fMenuText);
						} catch (Throwable t) {
							t.printStackTrace();
						}
					} else {
						OS.InsertMenu(fHandle, fMenuCount, OS.MF_BYPOSITION | OS.MF_POPUP, fItemMenu.fHandle, fItemMenu.fMenuText);
					}
					fItemMenuPosition = fMenuCount++;
					OS.SendMessageW(display.fHandle, OS.WM_SETREDRAW, 1, 0);
					OS.DrawMenuBar(display.fHandle);
				}
			});
		} else {
			Device.syncExec(new Runnable() {
				public void run() {
					DisplayPeer display = getDisplayablePeer().fDisplay.fPeer;
					if (OS.IsWinCE) { // WM2003 PPC, DellAxim X51 & X51v devices
						try {
							OS.RemoveSubMenu(fHandle, fItemMenu.fHandle, fItemMenuPosition);
						} catch (Throwable t) {
							t.printStackTrace();
						}
					} else {
						OS.RemoveMenu(fHandle, fItemMenuPosition, OS.MF_BYPOSITION);
					}
					//fItemMenu.dispose();
					fItemMenu = null;
					/*
					 * previously it was fItemMenuItem = null
					 * This is wrong in the case where the focus may still be on the item
					 * but the commands associated with it just becomes null
					 */
					fItemMenuItem = item;
					fItemCommandLookup.clear();
					if(--fMenuCount > 0) {
						OS.SendMessageW(display.fHandle, OS.WM_SETREDRAW, 1, 0);
						OS.DrawMenuBar(display.fHandle);
					} else {
						dispose();
					}
				}
			});
		}
	}
}
