package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;

class PopupChoiceContentComponent extends Scrollable implements IEventListener {

	Component fComponent;
	Graphics fGraphics;
	Image fContentBuffer;
	int fHandle = 0;
	int fContentWidth = -1;

	boolean fMousePressed = false;

	static final int MARGIN = 2;

	PopupChoiceContentComponent(final Window parent) {
		super(parent);
		Device.syncExec(new Runnable() {
			public void run() {
				fHandle = OS.CreateChildWindow(parent.getWindowHandle(),getContentFlags());
			}
		});
		addContentWindowEventListener();

		/*
		 * We have to set this field before creating items because they reference
		 * the content component when they're created.
		 */
		parent.fContentComponent = this;

		fFocusComponent = fComponent = createContents();

		fFocusComponent.fLayoutData |= Item.LAYOUT_2 | Item.LAYOUT_VEXPAND;
		fExpandVertically = true;

		int largestDimension = Math.max(Device.getShellWidth(), Device.getShellHeight());
		fContentBuffer = Image.createImage(largestDimension, largestDimension);
		fGraphics = new Graphics(this);
	}

	Component createContents() {
		return  new ChoiceComponent(this, ((PopupChoice) getWindow()).getChoice());
	}

	Rectangle getVisibleBounds() {
		return new Rectangle(fX, fY, fParent.fWidth, fParent.fHeight);
	}

	Image getOffscreenBuffer() {
		return fContentBuffer;
	}

	int getOriginX() {
		return fX;
	}

	int getOriginY() {
		return fY;
	}

	int getOriginX(Component child) {
		return child.fX - fScrollX;
	}

	int getOriginY(Component child) {
		return child.fY - fScrollY;
	}

	Layout createLayout() {
		return new CompositeLayout(this, MARGIN, MARGIN, MARGIN, 0, 0);
	}

	boolean isScrollable() {
//		return fLayout.fLayoutHeight > fHeight;
		return fComponent.getPreferredHeight() > fParent.fHeight;
	}

	boolean canScrollUp() {
		return fScrollY > 0;
	}

	boolean canScrollDown() {
//		return fScrollY + fParent.fHeight < fLayout.fLayoutHeight;
		return fScrollY + fParent.fHeight < fComponent.getPreferredHeight();
	}

	int getContentFlags() {
		int bits = OS.WS_CHILD | OS.WS_VISIBLE | OS.WS_CLIPSIBLINGS | OS.WS_BORDER;
		bits |= OS.WS_CLIPCHILDREN;
		return bits;
	}

	public void addContentWindowEventListener() {
		Device.addEventListener(this, fHandle);
	}

	public boolean dispatchEvent(Event e) {
		switch (e.fType) {
			case Event.WINDOW_PAINT:
				paintComponents(fGraphics);
				return false;
			case Event.VERTICAL_SCROLL:
				if (fComponent.isShown()) {
					int scrollY = e.fScrollOrSelectionIndex - fScrollY;
					scroll(0, scrollY);
				}
				return false;
			case Event.WINDOW_RESIZE:
				if (e.fX <= 0) return false;
				clearCachedSizes();
				int newWidth = e.fX;
				if (newWidth == fContentWidth) return false;

				fContentWidth = newWidth;
				fWidth = newWidth;

				layout();
				fParent.sizeChanged(e.fX, e.fY);
				fParent.layout();

				updateScrollBar();
				paintComponents(fGraphics);
				return false;
		}
		return false;
	}

	public boolean dispatchPointerEvent(Event e) {
		switch (e.fType) {
			case Event.POINTER_PRESSED:
				fMousePressed = true;
				if (Device.canGrabPointer()) Device.gCanUpdate = false;
				if (fComponent.isShown()) {
					return pointerPressed(e.fX, e.fY);
				}
				return false;
			case Event.POINTER_RELEASED:
				fMousePressed = false;
				if (Device.canGrabPointer()) Device.gCanUpdate = true;
				if (fComponent.isShown()) {
					return pointerReleased(e.fX, e.fY);
				}
				return false;
			case Event.POINTER_MOVED:
				if (!fMousePressed) return false;
				if (fComponent.isShown()) {
					return pointerDragged(e.fX, e.fY);
				}
				return false;
			default:
				return false;
		}
	}

	public boolean getAllowMenuEvents() {
		return false;
	}

	public void dispose() {
		super.dispose();

		Device.removeEventListener(this, fHandle);
		Device.syncExec(new Runnable() {
			public void run() {
				if (fHandle != 0) {
					OS.DestroyWindow(fHandle);
					fHandle = 0;
				}
			}
		});
		if (fGraphics != null) {
			fGraphics.dispose();
			fGraphics = null;
		}
		if (fContentBuffer != null) {
			fContentBuffer.dispose();
			fContentBuffer = null;
		}
	}

	public void setVisible(boolean visible) {
		int flags = visible ? OS.SW_SHOW : OS.SW_HIDE;
		OS.ShowWindow(fHandle, flags);
	}

	public Graphics getGraphics() {
		return fGraphics;
	}

	public GraphicsThreadsafe getDisplayGraphics() {
		final GraphicsThreadsafe[] g = new GraphicsThreadsafe[1];
		final PopupChoiceContentComponent content = this;
		Device.syncExec(new Runnable() {
			public void run() {
				g[0] = new GraphicsThreadsafe(content);
			}
		});
		return g[0];
	}

	void paintComponents(Graphics g) {
		// Clear the graphics
		g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BACKGROUND_RGB);
		g.fillRect(0, 0, fParent.fWidth, fParent.fHeight);

		super.paintComponents(g);

		g.flush();
	}

	void layout() {
		super.layout();
		if (fLayout.fLayoutHeight - fScrollY < fHeight) {
			fScrollY = Math.max(0, fLayout.fLayoutHeight - fHeight);
		}

		Device.syncExec(new Runnable() {
			public void run() {
				OS.SetWindowPos(fHandle, 0, fParent.fX, fParent.fY, fParent.fWidth, fParent.fHeight, 0);
			}
		});
	}

	void hideNotify() {
		((CompositeLayout) fLayout).hideNotify();
	}

	int getWindowHandle() {
		return fHandle;
	}

	int getContentWidth() {
		if (fContentWidth == -1) return fParent.fWidth;
		return fContentWidth;
	}
}
