package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2005  All Rights Reserved
 */

abstract class Scrollable extends WindowContent {

	int fScrollY;
	int fScrollX;
	boolean fScrolled = false;

	ScrollBarComponent fScrollBarComponent;
	int fContentWidth = -1;

	Scrollable(Composite parent) {
		super(parent);
	}

	void dispose() {
		if (fScrollBarComponent != null) fScrollBarComponent.hide();
		super.dispose();
	}

	void createScrollBar() {
		fScrollBarComponent = new ScrollBarComponent(this);
	}

	int getOriginX(Component child) {
		return getOriginX() + child.fX - fScrollX;
	}

	int getOriginY(Component child) {
		return getOriginY() + child.fY - fScrollY;
	}

	int translateX(Component c, int x) {
		return x - fX + fScrollX - c.fX;
	}

	int translateY(Component c, int y) {
		return y - fY + fScrollY - c.fY;
	}

	int toChildX(Component child, int x) {
		return x - child.fX + fScrollX;
	}

	int toChildY(Component child, int y) {
		return y - child.fY + fScrollY;
	}

	void resizeWithDeltas(int deltaX, int deltaY) {
		super.resizeWithDeltas(deltaX, deltaY);

		/* Scroll to the bottom if there is room */
		if (deltaY > 0) {
			/* the rest of the height that one would have to scroll downwards to see */
			int rest = getPreferredHeight() - fHeight - fScrollY;
			if (fScrollY > 0 && rest < deltaY) {
				fScrollY -= deltaY - rest;
			}
		}

		/* deltaX unused for now */

		updateScrollBar();
	}

	void scrollTo(int scrollXLocation, int scrollYLocation) {
		scroll(scrollXLocation - fScrollX, scrollYLocation - fScrollY);
	}

	boolean canScrollUp() {
		return false;
	}

	boolean canScrollDown() {
		return false;
	}

	abstract boolean isScrollable();

	void scrollContents() {
		/* No-op */
	}

	boolean scrollToTop() {
		/* Ignore horizontal scrolling for now. */
		if (/*fContentScrollX == 0 && */ fScrollY == 0) return false;

		/*fContentScrollX = 0 */
		fScrollY = 0;
		repaint();
		return true;
	}

	boolean scroll(int deltaX, int deltaY) {
		if (!getWindow().fAllowScroll) return false;

		/* Ignore horizontal scrolling for now. */
		/*fContentScrollX += deltaX;*/
		int oldValue = fScrollY;
		int contentHeight = fLayout.fLayoutHeight;
		if (contentHeight < fHeight) {
			fScrollY = 0;
		} else {
			fScrollY += deltaY;
			fScrollY = Math.max(0, fScrollY);
			fScrollY = Math.min(contentHeight - fHeight, fScrollY);
		}
//		System.err.println("Scroll setting new ScrollY: " + fScrollY);

		if (oldValue != fScrollY) {
			scrollContents();
			if (isShown()) repaint();
			if (fScrollBarComponent != null) fScrollBarComponent.setValues(contentHeight, fScrollY, fHeight);
			return true;
		}
		return false;
	}

	ScrollBarComponent getScrollBar() {
		return fScrollBarComponent;
	}

	void ensureVisible(Component c, int[] rect) {
		/*
		 * This algorithm ignores the possibility of horizontal
		 * scrolling. If a platform supports horizontal scrolling,
		 * this code will need to be revisited.
		 */

		int y = rect[1];
		int height = Math.min(fHeight, Math.max(0, rect[3]));
		int scrollY = 0;

		if (y < fScrollY) {
			scrollY = y - fScrollY;
		} else {
			int desiredBottom = y + height;
			int currentBottom = fScrollY + fHeight;
			if (desiredBottom > currentBottom) scrollY = desiredBottom - currentBottom;
		}

		if (scrollY != 0) {
			/* Hack. Magical +10s are needed to ensure that we scroll all the way
			 * to the top or bottom of the scrollable when using the keys to traverse
			 * (there could be margins that aren't traversed).
			 *
			 * Need a better way to do this.
			 */
			if (scrollY > 0) {
				scrollY = scrollY + 10;
			} else {
				scrollY = scrollY - 10;
			}
			scroll(0, scrollY);
		}
	}

	int[] translateComponentRectToChildRect(Component child, int[] visRect) {
		int[] cVisRect = new int[4];
//		int visLeft = visRect[0] + fScrollX;
		int visTop = visRect[1] + fScrollY;
//		int visRight = visLeft + visRect[2];
		int visBottom = visTop + visRect[3];

//		int cLeft = child.fX;
		int cTop = child.fY;
//		int cRight = child.fX + child.fWidth;
		int cBottom = child.fY + child.fHeight;

		int cVisX = 0;
		int cVisY = 0;
		int cVisWidth = child.fWidth;
		int cVisHeight = child.fHeight;

		if (cVisHeight > visRect[3]) {
			cVisHeight = visRect[3];
		} else if (cTop < visTop) {
			cVisHeight = cBottom - visTop;
		} else if (cBottom > visBottom) {
			cVisHeight = visBottom - cTop;
		}

		if (cTop < visTop) {
			cVisY = visTop - cTop;
		}

		cVisRect[0] = cVisX;
		cVisRect[1] = cVisY;
		cVisRect[2] = cVisWidth;
		cVisRect[3] = cVisHeight;

		return cVisRect;
	}

	void translateChildRectToComponentRect(Component child, int[] childRect, int viewWidth, int viewHeight, int[] visRect) {
		visRect[0] = childRect[0] + child.fX;
		visRect[1] = childRect[1] + child.fY;
		visRect[2] = Math.min(childRect[2], viewWidth);
		visRect[3] = Math.min(childRect[3], viewHeight);

		visRect[1] += fScrollY;
	}

	void updateScrollBar() {
		if (fScrollBarComponent == null) return;
		fScrollBarComponent.setValues(fLayout.fLayoutHeight, fScrollY, fHeight);
	}

	void layout() {
		super.layout();

		if (fLayout.fLayoutHeight - fScrollY < fHeight) {
			fScrollY = Math.max(0, fLayout.fLayoutHeight - fHeight);
		}

		if (getWindow().getType() != Displayable.TYPE_CANVAS && !ScrollBarComponent.isAlwaysVisible()) {
			boolean scrollable = isScrollable();
			if (scrollable) {
				if (fScrollBarComponent == null) createScrollBar();
				updateScrollBar();
			} else if (!scrollable && fScrollBarComponent != null) {
				fScrollBarComponent.dispose();
				fScrollBarComponent = null;
			}
		}
	}

	int getContentWidth() {
		if (fContentWidth == -1) return fWidth;
		return fContentWidth;
	}

	/*
	 * Called from the vertical ScrollBarCompoenent impl
	 */
	void setContentWidth() {
		if (fContentWidth != fWidth  && isScrollable()) {
			fContentWidth = fWidth - ScrollBarComponent.TOTAL_WIDTH;
			fWidth = fContentWidth;
			clearCachedSizes();
			layout();
		}
	}
}
