package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;

class TextPeer extends Win32Component implements IEventListener {

	static final int TEXTFIELD_TYPE = 0;
	static final int TEXTBOX_TYPE = 1;

	TextImpl fTextImpl;
	TextField fTextField;
	DisplayablePeer fDisplayablePeer;
	boolean fIsTextBox;
	boolean fIsNumeric;

	private final static int STATE_START       = 0;
	private final static int STATE_AFTER_MINUS = 1;
	private final static int STATE_FRACTION    = 2;
	private final static int STATE_NUMBER      = 3;
	private final static int KEY_PRINTABLE = 31; // Printable characters hava ASCII > 31

	TextPeer(Composite parent, TextField textField, TextImpl textImpl, int type) {
		super(parent);
		fIsTextBox = type == TEXTBOX_TYPE;
		fTextImpl = textImpl;
		fTextField = textField;

		fIsNumeric = isNumericFlagSet();

		fDisplayablePeer = textField.fScreen.fPeer;
		createHandle();
		setText(fTextImpl.getString());
		setConstraints(fTextImpl.getConstraints());

		textImpl.fPeer = this;
		Device.addEventListener(this, fHandle);
	}

	void createHandle() {
		Device.syncExec(new Runnable() {
			public void run() {
				if (fIsTextBox) {
					fHandle = createText(fDisplayablePeer.getWindowHandle(), getFlags());
				} else {
					int formHandle = getWindowHandle();
					fHandle = createText(formHandle, getFlags());
				}
				OS.SendMessageW(fHandle, OS.EM_SETLIMITTEXT, fTextImpl.getMaxSize(), 0);
			}
		});
	}

	boolean isNumericFlagSet() {
		return (fTextImpl.getConstraints() & TextField.NUMERIC) != 0;
	}

	int getFlags() {
		int bits = OS.WS_CHILD | OS.WS_VISIBLE | OS.WS_BORDER;

//		if ((fTextImpl.getConstraints() & TextField.NUMERIC) != 0) {
//			bits |= OS.ES_NUMBER;
//		}

		if (fIsTextBox) {
			bits |= OS.ES_MULTILINE | OS.ES_NOHIDESEL | OS.ES_AUTOVSCROLL
					| OS.WS_VSCROLL;
		} else {
			bits |= OS.ES_AUTOHSCROLL;
		}

		return bits;
	}

	void dispose() {
		super.dispose();
		final TextPeer peer = this;
		Device.syncExec(new Runnable() {
			public void run() {
				if (fHandle != 0) {
					fTextImpl.fContents = getText();
					fTextImpl.fPeer = null;
					OS.DestroyWindow(fHandle);
					Device.removeEventListener(peer, fHandle);
					fHandle = 0;
				}
			}
		});
	}

	int getMinimumHeight() {
		return getEditControlHeight(fHandle);
	}

	int getMinimumWidth() {
		//TODO: implement
		return 120;
	}

	int getPreferredHeight() {
		//TODO: implement properly
		fWidth = getWindow().getPreferredWidth();
		int maxHeight = ((ItemComponent) fParent).getMaxContentHeight();
		if (fIsTextBox) {
			return maxHeight;
		} else { /* TextField */
			return getMinimumHeight();
		}
	}

	int getPreferredWidth() {
		//TODO: implement
		if (fIsTextBox) return getWindow().getPreferredWidth();
		return getWindow().getPreferredWidth() * 3 / 4;
	}

	public int getCaretPosition() {
		final String[] textRet = new String[1];
		final int[] start = new int[1];
		final int[] end = new int[1];
		Device.syncExec(new Runnable() {
			public void run() {
				textRet[0] = OS.GetWindowText(fHandle);
				OS.SendMessageW(fHandle, OS.EM_GETSEL, start, end);
			}
		});

		/*
		 * Need to account for the fact that because of "\r\n" line delimeter replacement,
		 * the index could be off.
		 */
		String text = textRet[0];
		int caretPos = start[0];
		int index = text.indexOf("\r\n");
		while (index != -1 && index < caretPos) {
			caretPos--;
			index = text.indexOf("\r\n", index + 1);
		}
		return caretPos;
	}

	void setConstraints(final int constraints) {
		Device.syncExec(new Runnable() {
			public void run() {
				int readonlyValue = 0;

				if (isNumericFlagSet() != fIsNumeric) {
					dispose();
					createHandle();
					setText(fTextImpl.getString());
					show();
					sizeChanged(fWidth, fHeight);
					if (hasFocus()) OS.SetFocus(fHandle);
					fIsNumeric = !fIsNumeric;
				}

				if ((constraints & TextField.UNEDITABLE) != 0) {
					readonlyValue = 1;
				}
				OS.SendMessageW(fHandle, OS.EM_SETREADONLY, readonlyValue,0);

				if ((constraints & TextField.PASSWORD) != 0) {
					OS.SendMessageW(fHandle, OS.EM_SETPASSWORDCHAR, '*', 0);
				}

				// On the smartphone it is possible to change the characters
				// that the '1' key cycles through, and on some contraints it
				// makes sense to change the defaults
				if (OS.IsSP) {
					int mode;
					switch (constraints & TextField.CONSTRAINT_MASK) {
						case TextField.URL:
						case TextField.ANY:
						case TextField.EMAILADDR:
							mode = OS.EIM_SPELL; break;

						case TextField.DECIMAL:
						case TextField.NUMERIC:
						case TextField.PHONENUMBER:
							mode = OS.EIM_NUMBERS; break;

						default:
							mode = -1; break;
					}

					if (mode != -1) OS.Edit_SetInputMode(fHandle, mode);
				}
			}
		});
	}

	void setMaxSize(final int size) {
		Device.syncExec(new Runnable() {
			public void run() {
				OS.SendMessageW(fHandle, OS.EM_SETLIMITTEXT, size, 0);
			}
		});
	}

	public void setText(String label) {
		if ((fTextImpl.getConstraints() & TextField.PASSWORD) != 0) {
			/*
			 * Need to do this only for TextBox, as TextField (SingleLine Edit)
			 * password handling is taken care by windows
			 */
			if (fIsTextBox) {
				// If the mode is password then the string to be displayed will
				// be replaced with a '*' for each letter in the string;
				 char[] passwordString = new char[label.length()];
				 for (int i = 0; i < passwordString.length; i++) {
					 passwordString[i] = '*';
				 }
				 label = new String(passwordString);
			}
		} else {
			// Look through the entire string changing all
			// \n to \r\n so that the windows text field will break the
			// lines properly
			int start = label.indexOf('\n');
			while (start != -1) {
				if (start != 0) {
					if (label.charAt(start-1) != '\r') {
						// insert a \r before the \n if there isn't one
						// already there
						label = label.substring(0,start) + '\r' + label.substring(start);
						start+= 2;
					} else {
						// Move past the newline that we just found
						start++;
					}
				} else {
					// Add a \r to the beginning of the line and continue looking
					// through the rest of the string
					label = '\r' + label;
					start = 2;
				}
				start = label.indexOf('\n',start);
			}
		}

		super.setText(label);

		final int caretPos = label.length();
		Device.syncExec(new Runnable() {
			public void run() {
				OS.SendMessageW(fHandle, OS.EM_SETSEL, caretPos, caretPos);
			}
		});
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		if (hasFocus()) {
			switch (direction) {
				case Canvas.UP :
					/* Return false if at the top and not a textbox */
					if (!fIsTextBox) return false;
					return true;
				case Canvas.DOWN :
					/* Return true if at the bottom of the text */
					if (!fIsTextBox) return false;
					return true;
				case Canvas.LEFT:
					if (getCaretPosition() <= 0) {
						// if cursor is positioned before the first char of the textfield then traverse to previous item
						return fIsTextBox;
					}
					return true;
				case Canvas.RIGHT:
					if (getCaretPosition() >= getText().length()) {
						return false;
					}
					return true;
			}
		}

		/*
		 * Dispose the PopupChoice window if it is the one that is curretly showing up.
		 * Need to check if there is a cleaner way to do this. This is more or less
		 * a hack that is good enough for now.
		 */
		if (!fIsTextBox) {
			DisplayPeer peer = fDisplayablePeer.getDisplayPeer();
			if (peer.fCurrentWindow instanceof PopupChoice) {
				peer.fCurrentWindow.dispose();
			}
		}

		visibleRectangle[0] = 0;
		visibleRectangle[1] = 0;
		visibleRectangle[2] = fWidth;
		visibleRectangle[3] = fHeight;
		Device.syncExec(new Runnable() {
			public void run() {
				OS.SetFocus(fHandle);
				OS.InvalidateRect(fHandle, true);
			}
		});
		return true;
	}

	void traverseOut() {
		Device.syncExec(new Runnable() {
			public void run() {
				if (OS.GetFocus() == fHandle) OS.SetFocus(getDisplayablePeer().getWindowHandle());
			}
		});
	}

	boolean checkDecimal(int keyCode) {
		if (keyCode >= '0' && keyCode <= '9') return true;

		String text = getText();
		if (text.length() == 0 && keyCode == '-') return true;

		int state = STATE_START;

		boolean hadWholePart = false;
		boolean hadFractionPart = false;

		int caret = getCaretPosition();
		StringBuffer buffer = new StringBuffer(text);
		buffer.insert(caret, String.valueOf((char) keyCode));
		text = buffer.toString();

		for (int i = 0; i < text.length(); i++) {
			char c = text.charAt(i);
			switch (state) {
				case STATE_START: {
					if (c == '-') {
						state = STATE_AFTER_MINUS;
						break;
					}
				}
				case STATE_NUMBER:
				case STATE_AFTER_MINUS: {
					if (c == '.') {
						state = STATE_FRACTION;
						break;
					}
					if (Character.isDigit(c)) {
						state = STATE_NUMBER;
						hadWholePart = true;
						break;
					}
					return false;
				}
				case STATE_FRACTION: {
					if (Character.isDigit(c)) {
						hadFractionPart = true;
						break;
					}
					return false;
				}
			}
		}

		if (hadFractionPart || hadWholePart) return true;
		return false;
	}

	boolean checkNumeric(int keyCode) {
		String text = getText();
		if (text != null && text.length() == 0 && keyCode == '-') return true;
		if (keyCode >= '0' && keyCode <= '9') return true;
		return false;
	}

	boolean checkConstraints(int keyCode) {
		int constraints = fTextImpl.getConstraints();
		if ((constraints & TextField.UNEDITABLE) > 0) return false;
		switch (fTextImpl.getConstraints() & TextField.CONSTRAINT_MASK) {
			// These three contraints let any text pass
			case TextField.ANY:
			case TextField.PHONENUMBER:
				return true;
			case TextField.EMAILADDR:
				//return checkEmail(keyCode) != 0;
				return true;
			case TextField.URL:
				//return checkURL(keyCode);
				return true;
			case TextField.DECIMAL:
				return checkDecimal(keyCode);
			case TextField.NUMERIC:
				return checkNumeric(keyCode);
		}
		return false;
	}

	/**
	 * Checks whether the character is QCHAR as specified in RFC 821.
	 * QCHAR = Any of 128 ASCII characters except \r, \n, ", \
	 * @param ch
	 * @return 	true if character is a QCHAR
	 * 			false otherwise
	 */
	boolean isQChar(char ch) {
		return (!(ch > 128 || ch == '\"' || ch == '\n' || ch == '\r' || ch == '\\'));
	}

	/**
	 * Checks whether a character is SPECIAL character according to
	 * RFC 821.
	 * SPECIAL = {"<" | ">" | "(" | ")" | "[" | "]" | "\" | "."
                      | "," | ";" | ":" | "@"  """ | the control
                      characters (ASCII codes 0 through 31 inclusive and
                      127) }
	 * @param ch
	 * @return	true if character is a Special character
	 * 			false otherwise
	 */
	boolean isSpecialChar(char ch) {
		return ((ch >= 127 || (ch <= 0 && ch >= 31) || ch == '<' || ch == '>' || ch == '(' || ch == ')' || ch == '[' || ch == ']' || ch == '\\' || ch == '.' || ch == ',' || ch == ';' || ch == ':' || ch == '@' || ch == '\"'));
	}

	/**
	 * This implementation is part of ABNF format specification of RFC 821.
	 * RFC 821 specifies the complete ABNF for SMTP commands. But this implementation
	 * does implement only mailbox specification which specifies the syntax of an Email Address.
	 *
	 * @return  0 Invalid Email address
	 * 			1 Incomplete but valid email characters are present
	 * 			2 Complete Email address
	 */
	int checkEmail(int keyCode) {
		String localName, domain =  null;
		String emailAddress = getText() + String.valueOf((char) keyCode);
		int len = emailAddress.length();
		int index = emailAddress.indexOf('@');
		if (index == -1) {
			localName = emailAddress;
		} else {
			if (index == 0) return 0; // Invalid to have at start

			localName = emailAddress.substring(0, index);
			if (index + 1 < len) domain = emailAddress.substring(index + 1);
		}

		// check for validity of localName
		if (localName.charAt(0) == '\"') { // Quoted String.
			//Invalid as domain is present and this string starts with " and doesn't end with "
			if (!localName.endsWith("\"") && domain != null) return 0;

			index = 1;
			int lengthOfQString = localName.length();
			if (localName.charAt(lengthOfQString - 1) == '\"') lengthOfQString--; // Skipping the last quotation mark

			while (index < lengthOfQString) {
				char ch = localName.charAt(index);
				index++;
				if (ch == '\\') {
					if (index < lengthOfQString) {
						ch = localName.charAt(index);
						index++;
					} else {
						/*
						 * If \ is present and no other character present then it is
						 * invalid. Otherwise it might be incomplete
						 */
						return domain != null ? 0 : 1;
					}
				} else { // It must be a qchar
					if (!isQChar(ch)) return 0; // Invalid char present
				}
			}
		} else { // Dot String
			index = 0;
			int lengthOfDotString = localName.length();
			if (localName.charAt(lengthOfDotString - 1) == '.' && domain != null) return 0;

			while (index < lengthOfDotString) {
				char ch = localName.charAt(index);
				index++;
				if (ch == '\\') {
					if (index < lengthOfDotString) {
						ch = localName.charAt(index);
						index++;
					} else {
						/*
						 * If \ is present and no other character present then it is
						 * invalid. Otherwise it might be incomplete
						 */
						return domain != null ? 0 : 1;
					}
				} else { // It must be qchar
					if (isSpecialChar(ch)) return 0; // Invalid char present
				}
			}
		}
		if (domain != null) { // Check whether it is a valid domain
			index = 0;
			int lengthOfDomain = domain.length();
			while (index < lengthOfDomain) {
				char ch = domain.charAt(index);
				if (isAlpha(ch)) {
					index++;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						if (ch == '.') break; // Can be next domain element
						index++;
						if (!isAlpha(ch) && !isDigit(ch) && ch != '-') return 0; // Invalid char in domain name
					}
					if (ch != '.') { // Completed the domain length
						/*
						 * if ch ends with -, then it is incomplete. Otherwise, it is
						 * a complete E-mail address
						 */
						return ch == '-' ? 1 : 2;
					}
				} else if (ch == '#') { // The Number follows
					index++;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						if (ch =='.') break; // Can be next domain element present
						if (!isDigit(ch)) return 0; // Invalid char where number digits must be present
						index++;
					}
					if (ch != '.') return 2; // End of the domain is reached
				} else if (ch == '[') {
					index++;
					int num = -1;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						index++;
						if (ch == ']') {
							if (index < lengthOfDomain) { // Next char must be dot since next domain element is present.
								ch = domain.charAt(index);
								if (ch != '.') return 0;
							} else {
								return 2;
							}
						}
						if (ch == '.') {
							if (num < 0 || num > 255) return 0; // dots are present at wrong places.
							num = -1;
							continue;
						}
						if (!isDigit(ch)) return 0; // Invalid char other than digit is found

						if (num == -1) {
							num = ch - '0';
						} else {
							num = num * 10 + ch - '0';
						}
					}
				} else {
					return 0;
				} // Invalid char present
				index++;
			}
		}
		return 1; // In-complete Email Address.
	}

	/**
	 * checks whether the character is an alphabet or not
	 * @param c character to be checked
	 * @return	true if c is an alphabet
	 * 			false otherwise
	 */
	boolean isAlpha(char c) {
		return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z');
	}

	/**
	 * checks whether the character is a Digit or not
	 * @param c character to be checked
	 * @return	true if c is a digit
	 * 			false otherwise
	 */
	boolean isDigit(char c) {
		return c >= '0' && c <= '9';
	}

	/**
	 * checks whether the character is Sub-Delimiters for URL
	 * Sub-Delimiters = "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / ";" / "="
	 * @param c character to be checked
	 * @return	true if c is a sub-delimiter
	 * 			false otherwise
	 */
	boolean isSubDelimiters(char c) {
		return c == '!' || c == '$' || c == '\'' || c == '*' || c == '(' || c == ')' || c == '+' || c == ',' || c == ';' || c == '=';
	}

	/**
	 * checks whether the character is a unreserved character according to RFC 3986
	 * Unreserved = ALPHA / DIGIT / "-" / "." / "_" / "~"
	 * @param c
	 * @return	true if c is a unreserved character
	 * 			false otherwise
	 */
	boolean isUnreserved(char c) {
		return isAlpha(c) || isDigit(c) || c == '-' || c =='_' || c == '~' || c =='.';
	}

	/**
	 * Checks whether the query and fragment are valid or not
	 * according to specification of Fragment and Query parts in
	 * syntax of URI in RFC 3986.
	 * @param query 	To check whether it is a valid query part
	 * @param fragment	To Check whether fragment part is a valid fragment part or not
	 * @return	true if both are valid parts
	 * 			false if any one is not valid.
	 */
	boolean isValidQueryAndFragment(String query, String fragment) {
		if ((query == null || query.length() == 0) && (fragment == null || fragment.length() == 0)) return true;

		int val = isPChars(query);
		if (val == 0 || (val == 1 && fragment != null)) return false; // If Fragment part is present, then query cant be incomplete.

		// Valid Query is there. Now process the Fragment
		if (fragment != null) {
			return isPChars(fragment) != 0;
		}
		return true;
	}

	/**
	 * Checks whether a character is a HEX-DIGIT - {0..9, A..F}
	 * @param ch
	 * @return	true if it is a hexa-digit
	 * 			false otherwise
	 */
	boolean isHexDigit(char ch) {
		return (isDigit(ch) || (ch >= 'a' && ch <= 'f') || (ch >= 'A' || ch <= 'F'));
	}

	/**
	 * checks whether the text string contains only pchars
	 * pchar = pct-encoded / unreserved / sub-delimiters / ':' / '@'
	 * @param text
	 * @return 	0 	If it contains invalid character other than pchar
	 * 			1	Partially complete - PCT ENCODED may be incomplete
	 * 			2 	string completely consists of pchars only.
	 */
	int isPChars(String text) {
		if (text == null || text.length() == 0) return 1; // Incomplete

		int index = 0;
		int len = text.length();
		while (index < len) {
			char ch = text.charAt(index);
			if (ch == '%') { // May be PCT Encoded
				if (index + 2 >= len) return 1; // may be partially empty

				index++;
				ch = text.charAt(index);
				char nextCh = text.charAt(index + 1);
				if (!isHexDigit(ch) || !isHexDigit(nextCh)) return 0; // Invalid pchar
				index++;
			} else {
				if (!isUnreserved(ch) && !isSubDelimiters(ch) && ch != ':' && ch != '@') return 0; // Invalid pchar
			}
			index++;
		}
		return 2; // Complete pchar string
	}

	/**
	 * checkURL method is used to check the URL syntax. If checks whether
	 * the text in the TextField/TextBox is valid URL conforming to specification
	 * of RFC 3986. It is implementation of ABNF form of URI syntax in RFC 3986.
	 * @param keyCode - keycode of character pressed.
	 * @return 	true if it is a partial or complete URL
	 * 			false if contains invalid character or syntax
	 */
	boolean checkURL(int keyCode) {
		String query = null, fragment = null;
		boolean isAuthority = false;

		String text = getText() + String.valueOf((char) keyCode);
		if (text.length() == 0) return true; // Empty string is valid URL

		int index = 0;
		int length = text.length();
		char c;
		if (text.indexOf(' ') != -1) { // Check if the text contains space
			StringBuffer textBuffer = new StringBuffer();
			// Convert SPACE into Percent-Encoded form so as to allow space in the URL field
			while (index < length) {
				 c = text.charAt(index);
				 if (c == ' ') {
					 // %20 is Percent Encoded format for space
					 textBuffer.append("%20"); //NON-NLS-$1
				 } else {
					 textBuffer.append(c);
				 }
				 index++;
			 }
			 text = textBuffer.toString();
			 textBuffer = null;
		}

		index = text.indexOf('#'); // Checking whether the Fragment Part is there
		if (index != -1) {
			fragment = text.substring(index + 1); // Extract the fragment part
			text = text.substring(0, index);
		}

		index = text.indexOf('?'); // Checking whether the Query part is there
		if (index != -1) { // Extracting the Query Part of URL
			query = text.substring(index + 1);
			text = text.substring(0, index);
		}

		// Checking whether the fragment and query parts are valid.
		boolean validQueryFragment = isValidQueryAndFragment(query, fragment);
		if (!validQueryFragment) return false;

		// Process now to check for Scheme and hier-part
		index = 0;
		c = text.charAt(index);
		length = text.length();

		boolean isURI = false;
		boolean isURIReference = false;
		if (isAlpha(c)) { // URL scheme is present
			index = 1;
			isURI = true;
			do {
				if (index == length) return true; // Scheme incomplete. But valid URL

				c = text.charAt(index++);
				if (c == ':') break; // Scheme ended

				if (!(isAlpha(c) || isDigit(c) || c == '+' || c == '-' || c == '.')) return false; // SCHEME contains invalid character
			} while (index < length);

			if (index == length) return true; // Scheme incomplete. But valid URL
			c = text.charAt(index);
		}
		if (c == '/') { // If / is there it should contain hierarchical part
			index++;
			if (index == length) return true; // Scheme is complete. But next parts are not full.

			if (text.charAt(index) == '/') {
				isURIReference = true;
				isAuthority = true;
				index++;
			}
		}
		// If not URI or URI-Reference it is invalid scheme
		if (!isURI & !isURIReference) return false;

		if (index == length) return true;

		String remaining = text.substring(index);
		if (isAuthority) { // Authority part is there
			String abEmpty = null, authority = null;
			int indexAbEmpty = remaining.indexOf('/');
			if (indexAbEmpty != -1) { // abEmpty part is there.
				abEmpty = remaining.substring(indexAbEmpty + 1);
				authority = remaining.substring(0, indexAbEmpty);
			} else { // path-abempty is absent
				authority = remaining;
			}

			// Process the authority part
			if (authority != null && authority.length() != 0) {
				String userName = null;
				int uNameInd = authority.indexOf('@');
				if (uNameInd != -1) {
					userName = authority.substring(0, uNameInd);
					authority = authority.substring(uNameInd + 1);
				}

				if (userName != null && userName.length() != 0) {
					if (userName.indexOf('@') != -1) return false;

					int val = isPChars(userName);
					if (val == 0 || (val == 1 && (authority != null || abEmpty != null || query != null || fragment != null))) return false;
				}

				// Process the authority
				if (authority != null && authority.length() != 0) {
					if (authority.indexOf('@') != -1) return false;

					int val = isPChars(authority);
					if (val == 0 || (val == 1 && (abEmpty != null || query != null || fragment != null))) return false;
				}
			}
			// Process the abEmpty.
			if (abEmpty != null && abEmpty.length() != 0) {
				int len = abEmpty.length();
				index = abEmpty.indexOf('/');
				String segment = null;
				while (index < len && index != -1) {
					segment = abEmpty.substring(0, index);
					int val = isPChars(segment);
					if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
					abEmpty = abEmpty.substring(index + 1);
					index = abEmpty.indexOf('/');
				}
				int val = isPChars(abEmpty);
				if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
			}
			return true;
		} else {
			// path-abempty or path-absolute or path-noscheme checking goes here.
			// This is general URI syntax construct.
			int len = remaining.length();
			index = remaining.indexOf('/');
			String segment = null;
			while (index < len && index != -1) {
				segment = remaining.substring(0, index);
				int val = isPChars(segment);
				if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
				remaining = remaining.substring(index + 1);
				index = remaining.indexOf('/');
			}
			int val = isPChars(remaining);
			if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;

			return true;
		}
	}

	/*
	 * Send WM_CHAR message to the TextField
	 */
	void sendWMCharMessage(Event e) {
		final int keycode = e.fKeyCode;
		Device.syncExec(new Runnable() {
			public void run() {
				OS.SendMessageW(fHandle, OS.WM_CHAR, keycode, 0);
			}
		});
	}

	/**
	 * This function gets the selection indices of textbox contents,
	 *  The values returned are stored in array of three elements
	 *
	 * @return 	indices[0] - start of the selection text
	 * 			indices[1] - end of selected text
	 * 			indices[2] - Original length of the text in textbox
	 */
	int[] getSelectionIndices() {
		int start[] = new int[1];
		int end[] = new int[1];
		int indices[] = new int[3];
		String tempContents = OS.GetWindowText(fHandle);

		// Get the selection
		OS.SendMessageW(fHandle, OS.EM_GETSEL, start, end);

		int tempLen = tempContents.length();
		int sInd = start[0];
		int eInd = end[0];
		int index = 0;
		int newLineCount = 0;
		index = tempContents.indexOf("\r\n");
		while (index < sInd && index != -1) {
			start[0]--;
			end[0]--;
			newLineCount++;
			index = tempContents.indexOf("\r\n", index + 1);
		}
		while (index < eInd && index != -1) {
			end[0]--;
			newLineCount++;
			index = tempContents.indexOf("\r\n", index + 1);
		}
		while (index < tempLen && index != -1) {
			newLineCount++;
			index = tempContents.indexOf("\r\n", index + 1);
		}
		indices[0] = start[0];
		indices[1] = end[0];
		indices[2] = tempLen;
		return indices;
	}

	public boolean dispatchEvent(Event e) {
		boolean isPasswordTextBox =  fIsTextBox && (fTextImpl.getConstraints() & TextField.PASSWORD) == TextField.PASSWORD;
		if (e.fType == Event.CHARACTER) {
			switch (e.fKeyCode) {
				case Canvas.KEYCODE_LEFT:
				case Canvas.KEYCODE_RIGHT:
					return false;
				case Canvas.KEYCODE_BACKSPACE:
				case Canvas.KEYCODE_DELETE:
					// We need to process the BACKSPACE and DELETE in the case of a Password TextBox
					if (isPasswordTextBox) {
						int selectedIndices[] = getSelectionIndices();
						int len = fTextImpl.fContents.length();

						// If no text is selected, then delete the previous character
						// if group of char is selected, delete group
						if (selectedIndices[0] == selectedIndices[1]) selectedIndices[0]--;

						if (selectedIndices[1] > len) selectedIndices[1] = len;
						if (selectedIndices[2] <= 0) return false;
						fTextImpl.fContents = fTextImpl.fContents.substring(0, selectedIndices[0]) + fTextImpl.fContents.substring(selectedIndices[1]);
					}
					if (!fIsTextBox) {
						Device.postRunnable(new Runnable(){
							public void run() {
								fTextField.notifyStateChanged();
							}
						});
					}
					return false;
			}

			boolean allow = checkConstraints(e.fKeyCode);
			if (allow && !fIsTextBox) {
				Device.postRunnable(new Runnable(){
					public void run() {
						fTextField.notifyStateChanged();
					}
				});
			}
			if (allow && isPasswordTextBox) {
				int len = fTextImpl.fContents.length();
				int selectedIndices[] = getSelectionIndices();

				int ch = e.fKeyCode;
				if (e.fKeyCode > KEY_PRINTABLE) {
					e.fKeyCode = '*';
				} else if (e.fKeyCode != '\n' && e.fKeyCode != '\r') { // We are getting \r character as new line character event in WinXP.
					return true;
				}
				if (selectedIndices[1] > len) selectedIndices[1] = len;
				// Size of textbox is already reached the maximum size
				if (selectedIndices[2] >= fTextImpl.getMaxSize() && selectedIndices[0] == selectedIndices[1]) {
					return false;
				}
				fTextImpl.fContents = fTextImpl.fContents.substring(0, selectedIndices[0]) + String.valueOf((char)ch) + fTextImpl.fContents.substring(selectedIndices[1]);
				return false;
			}
			return !allow;
		}

		// If any of CLEAR / CUT / PASTE / UNDO operations are intended on password textbox, then dont allow this. We are blocking this here.
		if ((e.fType == Event.TEXTFIELD_CLEAR || e.fType == Event.TEXTFIELD_CUT || e.fType == Event.TEXTFIELD_PASTE || e.fType == Event.TEXTFIELD_UNDO) && isPasswordTextBox) {
			return true;
		}

		// If Delete key is pressed, we dont get WM_CHAR event. Instead we get WM_KEYDOWN Event only.
		// So we have to handle this DELETE key to delete the contents
		if (e.fType == Event.KEY_PRESSED) {
			if (e.fKeyCode == Canvas.KEYCODE_DELETE) { // Handle for TextBox with PASSWORD Field set
				if (isPasswordTextBox) {
					int selectedIndices[] = getSelectionIndices();
					int len = fTextImpl.fContents.length();
					if (selectedIndices[1] > len) selectedIndices[1] = len;
					if (selectedIndices[2] <= 0) return false;

					fTextImpl.fContents = fTextImpl.fContents.substring(0, selectedIndices[0]) + fTextImpl.fContents.substring(selectedIndices[1]);
				}
				return false;
			}
		}
		if (e.fType == Event.KEY_PRESSED || e.fType == Event.KEY_RELEASED) {
			fDisplayablePeer.fDisplay.fPeer.dispatchEvent(e);
		}
		return false;
	}

	public boolean getAllowMenuEvents() {
		return false;
	}

	boolean keyPressed(int keyCode) {
		switch (keyCode) {
			case Canvas.KEYCODE_LEFT:
			case Canvas.KEYCODE_RIGHT:
			case Canvas.KEYCODE_BACKSPACE:
			case Canvas.KEYCODE_DELETE:
				return false;
		}

		return super.keyPressed(keyCode);
	}

	public int size() {
		return getText().length();
	}

	void sizeChanged(int w, int h) {
		super.sizeChanged(w, h);
		/*
		 * The win32 text widget does not adjust the text scrolling when
		 * the size is changed. To ensure that the text is displaying correctly,
		 * we reset the text as a workaround.
		 */
		Device.asyncExec(new Runnable() {
			public void run() {
				setText(getText());
			}
		});
	}

	public void backKeyPressed(final Event e) {
		Device.syncExec(new Runnable() {
			public void run() {
				OS.SHSendBackToFocusWindow(OS.WM_HOTKEY, e.fArg1, e.fArg2);
			}
		});
	}

	public String getText() {
		final String[] textRet = new String[1];
		Device.syncExec(new Runnable() {
			public void run() {
				if (fIsTextBox && (fTextImpl.getConstraints() & TextField.PASSWORD) == TextField.PASSWORD) {
					textRet[0] = fTextField.fText.fContents;
				} else {
					if (fHandle == 0) {
						textRet[0] = fTextField.fText.fContents;
					} else {
						textRet[0] = OS.GetWindowText(fHandle);
					}
				}
			}
		});

		String text = textRet[0];
		int start = text.indexOf("\r\n");
		while (start != -1) {
			text = text.substring(0, start) + text.substring(start + 1);
			start = text.indexOf("\r\n");
		}

		return text;
	}

	static native int createText(int parent, int flags);
	native int getEditControlHeight(int handle);
}
