package javax.microedition.lcdui.game;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import javax.microedition.lcdui.*;
import com.ibm.ive.midp.*;

/**
 * <code>GameCanvas</code> represents MIDP's low-level game UI,
 * where the application developer is provided with a dedicated
 * off-screen graphics buffer and the ability to query the state
 * of a game action key.
 *
 * @since MIDP 2.0
 */
public abstract class GameCanvas extends Canvas {

	/**
	 * this bit indicates if GAME_A key was pressed (value is 0x0200)
	 */
	public static final int GAME_A_PRESSED = 0x0200;

	/**
	 * this bit indicates if GAME_B key was pressed (value is 0x0400)
	 */
	public static final int GAME_B_PRESSED = 0x0400;

	/**
	 * this bit indicates if GAME_C key was pressed (value is 0x0800)
	 */
	public static final int GAME_C_PRESSED = 0x0800;

	/**
	 * this bit indicates if GAME_D key was pressed (value is 0x1000)
	 */
	public static final int GAME_D_PRESSED = 0x1000;

	/**
	 * this bit indicates if LEFT key was pressed (value is 0x0004)
	 */
	public static final int LEFT_PRESSED = 0x0004;

	/**
	 * this bit indicates if RIGHT key was pressed (value is 0x0020)
	 */
	public static final int RIGHT_PRESSED = 0x0020;

	/**
	 * this bit indicates if UP key was pressed (value i 0x0002)
	 */
	public static final int UP_PRESSED = 0x0002;

	/**
	 * this bit indicates if DOWN key was pressed (value is 0x0040)
	 */
	public static final int DOWN_PRESSED = 0x0040;

	/**
	 * this bit indicates if FIRE key was pressed (value is 0x0100)
	 */
	public static final int FIRE_PRESSED = 0x0100;

	private Image fOffscreenBuffer;
	private Graphics fDisplayGraphics;
	private boolean fKeysSuppressed;
	private int fKeyStates;

	long fLastFlush;

	/**
	 * Consrtucts a new <code>GameCanvas</code>. This creates a new
	 * buffer consisting of a white background for use only within this
	 * instance.
	 * @param gameKeyEventsSuppressed <code>true</code> if key events should
	 * not be generated for keys associated with game actions (key events for all
	 * other keys cannot be suppressed).
	 */
	protected GameCanvas(boolean gameKeyEventsSuppressed) {
		super();

		// Create a dedicated off-screen image filled with white
		// pixels (done by the image). The size is the max screen
		// size (size before commands & ticker)
		int maxSize = Math.max(Device.getShellWidth(), Device.getShellHeight());
		// Set to maximum screen size possible to handle screen rotation correctly
		fOffscreenBuffer = Image.createImage(maxSize, maxSize);

		// whether or not regular key event notification is suppressed
		// for the game keys
		fKeysSuppressed = gameKeyEventsSuppressed;
	}

	/*
	 * This method is called from JNI by CanvasPeer.
	 * Returns true if keycode is a game key, false otherwise.
	 */
	private boolean hiddenKeyDown(int keycode) {
		try {
			int gameAction = getGameAction(keycode);
			if (gameAction == 0) return false;

			synchronized (Device.gDisplayableLock) {
				fKeyStates = fKeyStates | (1 << gameAction);
				return true;
			}
		} catch (IllegalArgumentException e) {}

		return false;
	}

	/**
	 * Paints the off-screen buffer onto the display.
	 */
	public void flushGraphics() {
		flushGraphics(0, 0, getWidth(), getHeight());
	}

	/**
	 * Paints the specified location and size of the off-screen buffer to the display.
	 * The buffer is not painted if the <code>width</code> or <code>height</code>
	 * is less than 1.
	 * @param x the horizontal location within the buffer
	 * @param y the vertical location within the buffer
	 * @param width the number of width pixels of the buffer
	 * @param height the number of height pixels for the buffer
	 */
	public void flushGraphics(int x, int y, int width, int height) {

		//System.out.println("flushGraphics("+x+","+y+","+width+","+height+")");

 		// buffer is not painted when width or height are less than 1
		if (width < 1 || height < 1) return;
		synchronized (Device.gDisplayableLock) {
			if (!isShown()) return;
			if (fDisplayGraphics == null) fDisplayGraphics = getDisplayGraphics();
			fDisplayGraphics.setClip(x, y, width, height);
			fDisplayGraphics.drawImage(fOffscreenBuffer, x, y, 0);

			/*
			 * flush to the display for each call, instead of delaying
			 * the flush. This results in smooth animation effects compared
			 * to the delayed flush. But, this also sets us back in JBenchmark2
			 * scores considerably.
			 */
//			Device.gCanvasFlushNeeded = true;
			Device.flushDisplay(System.currentTimeMillis());
		}
	}

	native Graphics getDisplayGraphics();

	/**
	 * Creates and returns a new <code>Graphics</code> object that
	 * draws to the existing off-screen buffer of the receiver.
	 * @return Graphics the surface the receiver will draw on.
	 */
	protected Graphics getGraphics() {
		// The image that creates the new Graphics ensures that:
		// the clipping area is size of the image, color is black,
		// default font, SOLID stoke style, and the origin is
		// located at (0,0) of the image.

		return fOffscreenBuffer.getGraphics();
	}

	/**
	 * Answers the current or previous (since the last time this method
	 * was called) state of the keys associated with game actions. As a
	 * result of calling this method, the key state is cleared.
	 * @return int an integer where the bits are 1 to indicating a
	 * key is down. Returns 0 if the receiver is not showing.
	 */
	public int getKeyStates() {
		synchronized (Device.gDisplayableLock) {
			if (!isShown()) return 0;

			// clear the key states now that they are read
			int readStates = fKeyStates;
			fKeyStates = 0;

			return readStates;
		}
	}

	/**
	 * Paints the off-screen buffer of the receiver on <code>g</code>
	 * while respecting the clip area and the origin translation of
	 * <code>g</code>.
	 * @param g the surface the receiver will draw on.
	 * @throws NullPointerException if <code>g</code> is <code>null</code>.
	 */
	public void paint(Graphics g) {
		hiddenPaint(g);
	}

	/*
	 * If subclasses override paint(), we have to be
	 * able to render the buffer to the display.
	 */
	private void hiddenPaint(Graphics g) {
		if (g == null) throw new NullPointerException();
		g.drawImage(fOffscreenBuffer, 0, 0, 0);
	}

	/*
	 * Peers are destroyed when not being displayed, so
	 * the display graphics must be recreated here.
	 */
	private void hiddenShowNotify() {
		// indicate that all keys are unpressed whenever we
		// become visible
		fKeyStates = 0;
	}

	private void hiddenHideNotify() {
	}
}
