package javax.microedition.lcdui.game;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import javax.microedition.lcdui.*;

/**
 * <code>Layer</code> is an abstract class that provides a framework
 * for visual game elements. Attributes for position, dimension, and
 * visiblity are included within the framework. Implementers must
 * subclass this type, and implement the <code>paint(Graphics)</code>
 * method.
 *
 * @since MIDP 2.0
 */

public abstract class Layer {

	// location of this layer
	private int fX;
	private int fY;

	// size of this layer
	private int fWidth;
	private int fHeight;

	// whether the layer is visible
	private boolean fVisible;

	Object fLock = new Object();

	/*
	 * default constructor cannot be visible
	 */
	Layer() {
		fX = 0;
		fY = 0;
		fVisible = true;
	}

	/**
	 * Answer the number of pixels that are the height of the receiver.
	 * @return int
	 */
	public final int getHeight() {
		synchronized (fLock) {
			return fHeight;
		}
	}

	/**
	 * Answer the number of pixels that are the width of the receiver.
	 * @return int
	 */
	public final int getWidth() {
		synchronized (fLock) {
			return fWidth;
		}
	}

	/**
	 * Answer the horizontal location of the receiver that is relative to the
	 * upper-left corner of the painter's coordinate system.  The default is 0.
	 * @return int
	 */
	public final int getX() {
		synchronized (fLock) {
			return fX;
		}
	}

	/**
	 * Answer the vertical location of the receiver that is relative to the
	 * upper-left corner of the painter's coordinate system. The default is 0.
	 * @return int
	 */
	public final int getY() {
		synchronized (fLock) {
			return fY;
		}
	}

	/**
	 * Answer <code>true</code> if the receiver is visible or <code>false</code>
	 * otherwise.
	 * @return boolean
	 * @see Layer#setVisible
	 */
	public final boolean isVisible() {
		synchronized (fLock) {
			return fVisible;
		}
	}

	/**
	 * Moves the receiver the specified distance in the horizontal and
	 * vertical direction.
	 * @param distanceX (right - positive, left - negative)
	 * @param distanceY (down - positive, up - negative)
	 * @see Layer#setPosition
	 */
	public void move(int distanceX, int distanceY) {
		synchronized (fLock) {
			fX = fX + distanceX;
			fY = fY + distanceY;
		}
	}

	/**
	 * Paints the receiver on <code>g</code>. As this is an
	 * abstract method, implementers must check for visibiity
	 * and do nothing if the receiver is not visible.
	 * @param g the surface the receiver will draw on.
	 * @throws NullPointerException if <code>g</code> is  <code>null</code>.
	 */
	public abstract void paint(Graphics g);

	/**
	 * Sets the horizontal and vertical location of the receiver
	 * relative to the upper-left corner of the painter's
	 * coordinate system.
	 * @param x
	 * @param y
	 */
	public void setPosition(int x, int y) {
		synchronized (fLock) {
			fX = x;
			fY = y;
		}
	}

	/**
	 * Marks the receiver as visible if the argument is <code>true</code>,
	* and marks it invisible otherwise.  The reciever is not painted if it is
	* not visible.
	 * @param visible true if this layer should be visible
	 * @see Layer#isVisible
	 */
	public void setVisible(boolean visible) {
		synchronized (fLock) {
			fVisible = visible;
		}
	}

	/*
	 * allow subclasses to change the width
	 * @param width
	 */
	void setWidth(int width) {
		synchronized (fLock) {
			fWidth = width;
		}
	}

	/*
	 * allow subclasses to change the height
	 * @param height
	 */
	void setHeight(int height) {
		synchronized (fLock) {
			fHeight = height;
		}
	}
}
