package javax.microedition.lcdui.game;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import javax.microedition.lcdui.Graphics;

/**
 * LayerManager assists the user in managing and drawing one or
 * more layers of visual game elements in the specified z-order.
 * Layers are painted from bottom to top from the end list to
 * position 0, respectively.
 *
 * @since MIDP 2.0
 */
public class LayerManager {

	// list of layers
	private FastVector fLayers;
	// view window
	private int fViewWindowX;
	private int fViewWindowY;
	private int fViewWindowW;
	private int fViewWindowH;

	Object fLock = new Object();

	/**
	 * Constructor for LayerManager.
	 */
	public LayerManager() {
		super();

		fLayers = new FastVector();
		// create the default view window
		fViewWindowX = 0;
		fViewWindowY = 0;
		fViewWindowW = Integer.MAX_VALUE;
		fViewWindowH = Integer.MAX_VALUE;
	}

	/**
	 * Appends a <code>layer</code> to the end of the receiver's list
	 * of existing layers. If the <code>layer</code> currently exists in the list, it is
	 * removed and then added to the end. The layer at position 0 is drawn on top.
	 * @param layer the layer to append
	 * @throws NullPointerException if <code>layer</code> is <code>null</code>
	 */
	public void append(Layer layer) {
		if (layer == null)
			throw new NullPointerException(MidpMsg.getString("LayerManager.NullLayer")); //$NON-NLS-1$

		synchronized (fLock) {
			// remove the layer if it already exists then
			// add the layer to the end
			fLayers.removeElement(layer);
			fLayers.addElement(layer);
		}
	}

	/**
	 * Answer the <code>layer</code> at position <code>index</code>.
	* @param index the zero-based position
	* @return Layer the layer at the index position
	 * @throws IndexOutOfBoundsException if <code>index < 0 </code>
	 * or if <code>index >= number of layers</code>
	 */
	public Layer getLayerAt(int index) {
		synchronized (fLock) {
			if (index < 0 || index >= fLayers.size())
				throw new IndexOutOfBoundsException(MidpMsg.getString("LayerManager.InvalidIndex")); //$NON-NLS-1$

			return (Layer) fLayers.elementAt(index);
		}
	}

	/**
	 * Answer the number of layers in the receiver's list of existing layers.
	 * @return int the number of layers
	 */
	public int getSize() {
		synchronized (fLock) {
			return fLayers.size();
		}
	}

	/**
	 * Inserts <code>layer</code> into the receiver's list of existing layers
	 * at position <code>position</code>. If the <code>layer</code>
	 * currently exists in the list, it is removed before adding. The layer
	 * at position 0 is drawn on top.
	 * @param layer the layer to insert
	 * @param position the zero-based position of <code>layer</code>
	 * @throws NullPointerException if <code>layer</code> is null
	 * @throws IndexOutOfBoundsException if <code>position < 0 </code>
	 * or if <code>position > number of layers</code>
	 */
	public void insert(Layer layer, int position) {
		if (layer == null) throw new NullPointerException(MidpMsg.getString("LayerManager.NullLayer")); //$NON-NLS-1$

		synchronized (fLock) {
			/*
			 * As per JSR 248 (MSA), section 6.2.3.3, IndexOutOfBoundsException needs
			 * to be thrown in the following cases,
			 * - if the index is less than 0
			 * - if the index is greater than the number of Layers already added to this
			 *   LayerManager and the Layer has not been added to this LayerManager yet
			 * - if the index is greater than the number of Layers already added to this
			 *   LayerManager minus one and the Layer has already been added to this
			 *   LayerManager
			 */
			int size = fLayers.size();
			int maxAllowedPosition = fLayers.contains(layer) ? size - 1 : size;
			if (position < 0 || position > maxAllowedPosition) {
				throw new IndexOutOfBoundsException(MidpMsg.getString("LayerManager.InvalidIndex")); //$NON-NLS-1$
			}

			/*
			 * remove the layer if it already exists then
			 * add the layer to the end
			 */
			fLayers.removeElement(layer);
			fLayers.insertElementAt(layer, position);
		}
	}

	/**
	 * Paints in descending order the layers within the view window
	 * on <code>g</code> at the location specified by
	 * <code>x</code>, <code>y</code>.
	 * @param g the surface the receiver will draw on.
	 * @param x the point in the horizontal direction for drawing of
	 * the view window on <code>g</code>.
	 * @param y the point in the vertical direction for drawing of
	 * the view window on <code>g</code>.
	 * @throws NullPointerException if <code>g</code> is <code>null</code>.
	 * @see setViewWindow(int, int, int, int)
	 */
	public void paint(Graphics g, int x, int y) {
		if (g == null) throw new NullPointerException(MidpMsg.getString("LayerManager.paint.NullGraphics")); //$NON-NLS-1$

		// save the original translation point and
		// clipping rectangle
		int oldX = g.getTranslateX();
		int oldY = g.getTranslateY();
		int oldClipX = g.getClipX();
		int oldClipY = g.getClipY();
		int oldClipW = g.getClipWidth();
		int oldClipH = g.getClipHeight();

		synchronized (fLock) {
			// g's clipping area must be intersected with a rectangle
			// the size of the view window and located at (x, y).
			g.clipRect(x, y, fViewWindowW, fViewWindowH);

			// shift g's x and y so that the view window is at the point
			// specified by the parmametes x, y. Remeber that translates
			// are cumulative - we want it at an absolute location. Keep
			// the user's translation point as the start.
			g.translate(x - fViewWindowX, y - fViewWindowY);

			for (int index = fLayers.size() - 1; index >= 0; index--) {
				Layer layer = (Layer) fLayers.elementAt(index);
				if (layer.isVisible()) layer.paint(g);
			}
		}

		// restore original translations and clipping
		g.translate(oldX - g.getTranslateX(), oldY - g.getTranslateY());
		g.setClip(oldClipX, oldClipY, oldClipW, oldClipH);
	}

	/**
	 * Removes <code>layer</code> from the receiver's list of existing layers
	 * at position <code>index</code>. If <code>layer</code>
	 * does not exist in the list, the method does nothing. The indices of any
	 * layers following this layer are adjusted accordingly.
	 * @param layer the layer to remove
	 * @throws NullPointerException if <code>layer</code> is null
	 */
	public void remove(Layer layer) {
		if (layer == null)
			throw new NullPointerException(MidpMsg.getString("LayerManager.NullLayer")); //$NON-NLS-1$

		// remove the layer if it already exists then
		// add the layer to the end
		synchronized (fLock) {
			fLayers.removeElement(layer);
		}
	}

	/**
	 * Defines the location and size of an area to paint. The default values
	 * are (0, 0, Integer.MAX_VALUE, Integer.MAX_VALUE).
	 * @param x the position of the view window in the horizontal direction
	 * @param y the position of the view window in the vertical direction
	 * @param width the number of pixels for the width of the view window
	 * @param height the number of pixels for the height of the view window
	 * @throws IllegalArgumentException  if
	 * <code>width</code> or <code>height</code> is not greater than 0
	 * @see #paint(Graphics, int, int)
	 */
	public void setViewWindow(int x, int y, int width, int height) {
		if (width < 0 || height < 0)
			throw new IllegalArgumentException(MidpMsg.getString("LayerManager.setViewWindow.WidthHeight")); //$NON-NLS-1$

		synchronized (fLock) {
			fViewWindowX = x;
			fViewWindowY = y;
			fViewWindowW = width;
			fViewWindowH = height;
		}
	}

}
