package javax.microedition.lcdui.game;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import javax.microedition.lcdui.*;

import com.ibm.ive.midp.*;

public class Sprite extends Layer {

	/**
	 * image with no tranformational rotations applied.
	 * Value is 0.
	 */
	public static final int TRANS_NONE = 0;

	/**
	 * transform the image into a mirror image and
	 * then rotate it 180 degrees clockwise (the
	 * top is now the bottom). Value is 1.
	 */
	public static final int TRANS_MIRROR_ROT180 = 1;

	/**
	 * transform the image into a mirror image
	 * of itself (rotated 180 degress around
	 * the vertical axis - the right side is now
	 * the left side). Value is 2.
	 */
	public static final int TRANS_MIRROR = 2;

	/**
	 * transform the image by rotating it 180 degrees
	 * clockwise (the top is now on the bottom). Value
	 * is 3.
	 */
	public static final int TRANS_ROT180 = 3;

	/**
	 * transform the image into a mirror image and
	 * then rotate it 270 degrees clockwise (the top
	 * is now on the left). Value is 4.
	 */
	public static final int TRANS_MIRROR_ROT270 = 4;

	/**
	 * transform the image by rotating it 90 degrees
	 * clockwise (the top is now on the right). Value
	 * is 5.
	 */
	public static final int TRANS_ROT90 = 5;

	/**
	 * transform the image by rotating it 270 degrees
	 * clockwise (the top is now on the left). Value
	 * is 6.
	 */
	public static final int TRANS_ROT270 = 6;

	/**
	 * transform the image into a mirror image and
	 * then rotate it 90 degrees clockwise (the top
	 * is now on the right). Value is 7.
	 */
	public static final int TRANS_MIRROR_ROT90 = 7;

	// the image defining this sprite
	private Image fImage;
	// the sequence of frames within this image
	private int[] fFrameSequence;
	private boolean fIsDefaultSequence;
	// index into the sequence which defines the frame
	// currently showing
	private int fSequenceIndex;
	// the image is made up of frames in any
	// arrangement
	private int fFrameCount;
	// reference pixel
	private int fRefPixelX;
	private int fRefPixelY;
	// requested rotation
	private int fTransform;
	// collision rectangle
	private int fCollisionX;
	private int fCollisionY;
	private int fCollisionWidth;
	private int fCollisionHeight;

	private Image[] fImageCache;
	private int fFrameWidth;
	private int fFrameHeight;

	/**
	 * Creates a Sprite from the <code>image</code>.
	 * The frame width and height is the image width and height when the
	 * Sprite is created by this constructor.
	 * @param image the image which defines the sprite
	 * @throws NullPointerException if <code>image</code> is <code>null</code>
	 */
	public Sprite(Image image) {
		super();
		if (image == null) throw new NullPointerException();
		createFrom(image, image.getWidth(), image.getHeight(), true);
	}

	/**
	 * Creates a Sprite from the <code>image</code> with equally sized frames.
	 * The dimensions of the frames are specified by <code>frameWdith</code>
	 * and <code>frameHeight</code>.
	 * @param image the image which defines the sprite
	 * @param frameWidth the width of each frame
	 * @param frameHeight the height of each frame
	 * @throws NullPointerException if <code>image</code> is <code>null</code>
	 * @throws IllegalArgumentException if
	 * <code>frameWidth</code> &lt. 1, or
	 * <code>frameHeight</code> &lt. 1, or
	 * <code>image</code> width is not a multiple of <code>frameWidth</code> or
	 * <code>image</code> height is not a multiple of <code>frameHeight</code>
	 */
	public Sprite(Image image, int frameWidth, int frameHeight) {
		super();
		createFrom(image, frameWidth, frameHeight, true);
	}

	private void createFrom(Image image, int frameWidth, int frameHeight, boolean isNew) {
		if (image == null)
			throw new NullPointerException();
		if (frameWidth < 1)
			throw new IllegalArgumentException();
		if (frameHeight < 1)
			throw new IllegalArgumentException();
		if (image.getWidth() % frameWidth != 0)
			throw new IllegalArgumentException();
		if (image.getHeight() % frameHeight != 0)
			throw new IllegalArgumentException();

		// save the image
		synchronized (fLock) {
			fImage = image;
			fImageCache = null;

			// when creating a new sprite
			if (isNew) {
				// set the default reference pixel
				fRefPixelX = 0;
				fRefPixelY = 0;

				// set the default transform
				fTransform = TRANS_NONE;
			}

			// save the current (may be swapped) size for later comparisons
			// (note they are not set when creating new sprites)
			int savedWidth = getWidth();
			int savedHeight = getHeight();

			// get the x,y location of a sprite rotating from un-tranformed to
			// whatever is the current transform
			int x = getLeftEdge();
			int y = getTopEdge();

			// when the image is rotated 90 or 270 degrees (regardless
			// of being a mirror image), the width and height are swapped
			setWidth((fTransform < TRANS_MIRROR_ROT270) ? frameWidth : frameHeight);
			setHeight((fTransform < TRANS_MIRROR_ROT270) ? frameHeight : frameWidth);

			// when creating a new sprite.
			// Or, when the frame size changes ...
			if (isNew || getWidth() != savedWidth || getHeight() != savedHeight) {
				// set the default collision rectangle (to match the size
				// of the untransformed sprite)
				defineCollisionRectangle(0, 0, frameWidth, frameHeight);
				// when the sprite is transformed
				if (fTransform != TRANS_NONE) {
					// When the frame size changes for a transformed sprite,
					// move the upper-left corner so that the reference point
					// remains stationary. This means removing the previous
					// transform location and moving to the transform
					// location based on the new width and height
					move(-x + getLeftEdge(), -y + getTopEdge());
				}
			}

			// save current frame count for later comparisons
			// (note it is not set when creating new sprites)
			int savedFrameCount = fFrameCount;
			// determine number of raw frames
			fFrameCount = (image.getWidth() / frameWidth) * (image.getHeight() / frameHeight);

			// create the default frame sequences when creating a new sprite
			// or, when there are fewer frames than previously,
			// use the default frame sequence and set the index
			// so that it starts at the beginning.
			if (isNew || fFrameCount < savedFrameCount) {
				createDefaultFrameSequence();
				fSequenceIndex = 0;
			} else {
				// when a custom sequence is not is use,
				// then use the default sequence
				if (fIsDefaultSequence) createDefaultFrameSequence();
			}

			fFrameWidth = frameWidth;
			fFrameHeight = frameHeight;
			if (!fImage.isMutable()) createImageCache();
		}
	}

	/**
	 * Creates a new instance of a Sprite from an existing Sprite.
	 * @param aSprite The Sprite used to create this instance
	 * @throws NullPointerException if <code>aSprite</code> is <code>null</code>
	 */
	public Sprite(Sprite aSprite) {
		if (aSprite == null) throw new NullPointerException();

		synchronized (fLock) {
			// create this sprite based on the one passed
			fImage = Image.createImage(aSprite.fImage);  // take a snapshot of the image
			fFrameCount = aSprite.fFrameCount;
			fIsDefaultSequence = aSprite.fIsDefaultSequence;
			fFrameSequence = new int[aSprite.fFrameSequence.length];
			System.arraycopy(aSprite.fFrameSequence, 0, fFrameSequence, 0, fFrameSequence.length);
			fSequenceIndex = aSprite.fSequenceIndex;
			fRefPixelX = aSprite.fRefPixelX;
			fRefPixelY = aSprite.fRefPixelY;
			setPosition(aSprite.getX(), aSprite.getY());
			setWidth(aSprite.getWidth());
			setHeight(aSprite.getHeight());
			setVisible(aSprite.isVisible());
			fTransform = aSprite.fTransform;
			defineCollisionRectangle(
				aSprite.fCollisionX,
				aSprite.fCollisionY,
				aSprite.fCollisionWidth,
				aSprite.fCollisionHeight);
		}
	}

	public final boolean collidesWith(Image image, int x, int y, boolean pixelLevelDetection) {
		if (image == null) throw new NullPointerException();

		// invisible sprites can not collide with anything
		if (!isVisible()) return false;

		synchronized (fLock) {
			// get the collision rectangle based on the transform
			Rectangle collision = _getCollisionRectangle();
			// get the rectangle that represents the image
			Rectangle imageBounds = new Rectangle(x, y, image.getWidth(), image.getHeight());
			// first see if the image rectangle intersects with the collision rectangle
			boolean collides = collision.intersects(imageBounds);

			// for pixel-level detection we must check the pixels, but only
			// if the collision rectangle and image intersected
			if (pixelLevelDetection && collides) {
				// be careful the collision rectangle may be larger than the sprite.
				// those pixels outside of the sprite are considered tranparent
				collision = collision.intersection(
					new Rectangle(getX(), getY(), getWidth(), getHeight()));
				// now intersect the above smallest possible rectangle with the
				// image for finding opaque pixels
				Rectangle intersection = collision.intersection(imageBounds);

				collides = false;
				// only need to check pixels if there are common points
				if (!intersection.isEmpty()) {
					// get the pixels of the image that was passed to us
					int[] argb = new int[intersection.width * intersection.height];
					image.getRGB(argb,
						0,
						intersection.width,
						intersection.x - x,
						intersection.y - y,
						intersection.width,
						intersection.height);

					collides = _collides(argb, intersection);
				}
			}

			return collides;
		}
	}

	public final boolean collidesWith(Sprite aSprite, boolean pixelLevelDetection) {
		if (aSprite == null) throw new NullPointerException();

		synchronized (fLock) {
			// invisible sprites can not collide with anything
			if (!isVisible() || !aSprite.isVisible()) return false;

			// when pixel-level detection is not used, just check to see if the two collision
			// rectangles intersect
			if (!pixelLevelDetection) return _getCollisionRectangle().intersects(aSprite._getCollisionRectangle());

			// create a rectangle that is the intersection of the transformed sprite
			// and it's transformed collision rectangle
			Rectangle collision = _getCollisionRectangle().intersection(
				new Rectangle(getX(), getY(), getWidth(), getHeight()));
			// the collision rectangle could be totally outside of the bounds of the
			// sprite for some reason
			if (collision.isEmpty()) return false;

			// otherwise, using the current frame, check to see if the opaque
			// pixels of the images within the two sprite's collision rectangles
			// collide
			return aSprite._collidesWith(this, collision);
		}
	}

	public final boolean collidesWith(TiledLayer tiledLayer, boolean pixelLevelDetection) {
		if (tiledLayer == null) throw new NullPointerException();

		// invisible layers can not collide with anything
		if (!isVisible() || !tiledLayer.isVisible()) return false;

		synchronized (fLock) {
			// get the rectangle that overlays the background (and collision rectangle)
			int x = tiledLayer.getX();
			int y = tiledLayer.getY();
			Rectangle intersection = _getCollisionRectangle().intersection(new Rectangle(
				x, y, tiledLayer.getWidth(), tiledLayer.getHeight()));
			// if this much doesn't intersect then there will never be a collision
			if (intersection.isEmpty()) return false;

			boolean collides = false;
			// when not pixel level detection,  see if Sprite's collision
			// rectangle intersects with with an non-empty cell in the background.
			if (!pixelLevelDetection) {
				// determine the cells in the background to check
				int cellWidth = tiledLayer.getCellWidth();
				int cellHeight = tiledLayer.getCellHeight();
				int startCol = (intersection.x - x) / cellWidth;
				int endCol = (intersection.x + intersection.width - 1 - x) / cellWidth;
				int startRow = (intersection.y - y) / cellHeight;
				int endRow = (intersection.y + intersection.height -1 - y) / cellHeight;
				for (int rowIndex = startRow; (rowIndex <= endRow) && !collides; rowIndex++) {
					for (int colIndex = startCol; (colIndex <= endCol) && !collides; colIndex++) {
						collides = tiledLayer.getCell(colIndex, rowIndex) != 0;
					}
				}
			} else {
				// be careful the collision rectangle may be larger than the sprite.
				// those pixels outside of the sprite are considered tranparent.
				// now find only the pixels within the sprite's image for the smallest
				// possible rectangle.
				intersection = intersection.intersection(
					new Rectangle(getX(), getY(), getWidth(), getHeight()));
				if (!intersection.isEmpty()) {
					int[] argb = new int[intersection.width * intersection.height];
					tiledLayer.getRGB(argb,
						intersection.x - x,
						intersection.y - y,
						intersection.width,
						intersection.height);

					collides = _collides(argb, intersection);
				}
			}

			return collides;
		}
	}

	/**
	 * Defines the location and size of rectangle that is
	 * used for determining intersections with other
	 * rectangles.
	 * @param x the horizontal location for determining
	 * a colliion
	 * @param y the vertical location for determining
	 * a collision
	 * @param width the number of horizontal pixels for
	 * determining a collision
	 * @param height the number of vertical pixels for
	 * determining a collison
	 * @throws IllegalArgumentException if
	 * <code>width</code> or <code>height</code> is less then 0
	 */
	public void defineCollisionRectangle(int x, int y, int width, int height) {
		if (width < 0 || height < 0) throw new IllegalArgumentException();

		synchronized (fLock) {
			fCollisionX = x;
			fCollisionY = y;
			fCollisionWidth = width;
			fCollisionHeight = height;
		}
	}

	/**
	 * Defines a point relative to (0,0) of the un-transformed
	 * receiver. This point is stationary with regards to the
	 * upper left corner of the image and transformations
	 * are performed around it. The reference pixel may lie outside
	 * of the dimensions of the frames. Changing the reference pixel
	 * does not move a transformed sprite, rather the new definition
	 * is used with future transforms.
	 * @param x the location in the horizontal direction (relative to 0)
	 * @param y the location in the vertical direction (relative to 0)
	 */
	public void defineReferencePixel(int x, int y) {
		synchronized (fLock) {
			fRefPixelX = x;
			fRefPixelY = y;
		}
	}

	/**
	 * Answer the index into the sequence which defines
	 *  the frame that is currently showing. This is NOT
	 * the number of the frame that is showing.
	 * @return int the sequence index for the displayed frame
	 */
	public final int getFrame() {
		synchronized (fLock) {
			return fSequenceIndex;
		}
	}

	/**
	 * Answer the number of frame sequences which may or
	 * may not match the number of frames
	 * @return int the number of frame sequences
	 */
	public int getFrameSequenceLength() {
		synchronized (fLock) {
			return fFrameSequence.length;
		}
	}

	/**
	 * Answer the number of frames in this Sprite's image which
	 * may or may not match the number of frame sequences. As
	 * arrangement of frames within the image is arbitrary, the
	 * number of frames is determined by the product of the
	 * number of frames wide by the number of frame high.
	 * @return int the number of frames in the image
	 */
	public int getRawFrameCount() {
		synchronized (fLock) {
			return fFrameCount;
		}
	}

	/**
	 * Answer the value in the horizontal direction of the
	 * reference pixel in the painter's coordinate system.
	 * Since the reference pixel is stationary and the image
	 * moves, the value returned is the same regardless
	 * of the location or transform of the sprite.
	 * @return int the x pixel location relative to (0,0) of
	 * the painter's coordinate system
	 */
	public int getRefPixelX() {
		// get the x location of the untransformed sprite. The reference
		// pixel in the painter's coordinate system is relative to that.
		synchronized (fLock) {
			return getX() - getLeftEdge() + fRefPixelX;
		}
	}

	/**
	 * Answer the value in the vertical direction of the
	 * reference pixel in the painter's coordinate system.
	 * Since the reference pixel is stationary and the image
	 * moves, the value returned is the same regardless
	 * of the location or transform of the sprite.
	 * @return int the y pixel location relative to (0,0) of
	 * the painter's coordinate system
	 */
	public int getRefPixelY() {
		// get they location of the untransformed sprite. The reference
		// pixel in the painter's coordinate system is relative to that.
		synchronized (fLock) {
			return getY() - getTopEdge() + fRefPixelY;
		}
	}

	/**
	 * Increment the current sequence index so that
	 * the next frame in the sequence will show when
	 * displayed. If the new index is not less than the
	 * size of the sequence array, wrap around to 0.
	 */
	public void nextFrame() {
		// increment the frame index and check
		// that it is still within the bounds of the
		// sequence array, if not, wrap to the beginning.
		synchronized (fLock) {
			fSequenceIndex++;
			if (fSequenceIndex >= fFrameSequence.length) fSequenceIndex = 0;
		}
	}

	/**
	* @see Layer#paint(Graphics)
	*/
	public final void paint(Graphics g) {
		if (g == null) throw new NullPointerException();

		synchronized (fLock) {
			if (!isVisible()) return;
			// destX and destY have to be shifted to account for the fact
			// that the upper left of the TRANSFORMED sprite is actually
			// the point to start drawing
			int destX = getX();
			int destY = getY();

			if (fImageCache != null) {
				int frame = fFrameSequence[fSequenceIndex];
				g.drawImage(fImageCache[frame], destX, destY, 0);
			} else {
				// get a rectangle that defines the current frame of the untransformed
				// image
				Rectangle src = getCurrentRectangle();

				// copy the frame located at x,y into the graphic's image
				// at it's specified position
				g.drawRegion(fImage, src.x, src.y, src.width, src.height, fTransform, destX, destY, 0);
			}
		}
	}

	/**
	 * Decrment the current sequence index so that
	 * the previous frame in the sequence will show when
	 * displayed. If the new index is not greater than 0, wrap
	 * around to 1 less than the size of the sequence array.
	 */
	public void prevFrame() {
		// decrement the frame index and check
		// that it is still within the bounds of the
		// sequence array, if not, wrap to the end.
		synchronized (fLock) {
			fSequenceIndex--;
			if (fSequenceIndex < 0) fSequenceIndex = fFrameSequence.length - 1;
		}
	}

	/**
	 * Set the index into the array of frame sequences
	 * that defines the frame currently showing. This is
	 * NOT the number of the frame.
	 * @param sequenceIndex the index of the frame
	 * sequences that defines the frame to dispaly.
	 * @throws IndexOutOfBoundsException if
	 * <code>sequenceIndex</code> is less than 0 or
	 * <code>sequenceIndex</code> is greater than or equal to the
	 * number of frame sequences
	 */
	public void setFrame(int sequenceIndex) {
		if (sequenceIndex < 0) throw new IndexOutOfBoundsException();
		synchronized (fLock) {
			if (sequenceIndex >= fFrameSequence.length) throw new IndexOutOfBoundsException();
			fSequenceIndex = sequenceIndex;
		}
	}

	/**
	 * Customizes the order in which frames are shown. This
	 * an array of numbers corresponding to a frame to display.
	 * This method has the side effect of setting the sequence
	 * index to 0.
	 * @param sequences an array defining the sequence of
	 * the frames or null to indicate that the sequence should
	 * be reset to the default
	 * @throws ArrayIndexOutOfBoundsException if
	 * an element of sequence is not between 0 and the number
	 * of raw frames inclusively
	 * @throws IllegalArgumentException if
	 * the sequences array does not have one or more elements
	 */
	public void setFrameSequence(int[] sequences) {
		synchronized (fLock) {
			// null indicates frame sequences should become
			// the default
			if (sequences == null) {
				createDefaultFrameSequence();
			} else {
				if (sequences.length < 1) throw new IllegalArgumentException();

				// indicate the default sequence is no longer is use
				fIsDefaultSequence = false;
				// copy the sequences into our array
				// so that they can't be changed.
				fFrameSequence = new int[sequences.length];
				for (int i = 0; i < sequences.length; i++) {
					int frameNum = sequences[i];
					if (0 > frameNum || fFrameCount <= frameNum)
						throw new ArrayIndexOutOfBoundsException();
					fFrameSequence[i] = sequences[i];
				}
			}

			// reset current frame
			fSequenceIndex = 0;
		}
	}

	public void setImage(Image image, int frameWidth, int frameHeight) {
		synchronized (fLock) {
			createFrom(image, frameWidth, frameHeight, false);
		}
	}

	/**
	 * Moves the receiver such that the reference pixel
	 * becomes the location specified by <code>x</code>
	 * and <code>y</code> in the painter's coordinate system.
	 * By default, the reference pixel is defined as location (0,0).
	 * @param x the horizontal location of reference pixel
	 * after the reciever is moved
	 * @param y the vertical location of the reference pixel
	 * after the receiver is moved
	 */
	public void setRefPixelPosition(int x, int y) {
		synchronized (fLock) {
			// determine the distance to move based
			// on the current location of the reference pixel
			int distanceX = x - getRefPixelX();
			int distanceY = y - getRefPixelY();
			// move the required distance
			move(distanceX, distanceY);
		}
	}

	/**
	 * Defines a rotation around the reference pixel
	 * which is applied when the receiver is painted. The
	 * rotation is always applied to an un-transformed frame.
	 * @param transform rotation constant applied during paint
	 * @throws IllegalArgumentException if
	 * <code>transform</code> is not one of the valid
	 * transform constants
	 */
	public void setTransform(int transform) {
		// the valid transforms range from
		// 0 through 7 inclusive
		if (transform < 0  || transform > 7) throw new IllegalArgumentException();

		synchronized (fLock) {
			// don't do any work if the transform didn't change
			if (transform == fTransform) return;

			// get x,y location if current transform
			// were applied to an untransformed sprite
			int x = getLeftEdge();
			int y = getTopEdge();

			// when the image rotates 90 or 270 degrees
			// (regardless of being a mirror image), the width
			// and height are now swapped. If already swapped
			// don't do it again.
			if ((transform >= 4 && fTransform < 4) ||
				(transform < 4 && fTransform >=4)) {
				int temp = getWidth();
				setWidth(getHeight());
				setHeight(temp);
				temp = fCollisionWidth;
				fCollisionWidth = fCollisionHeight;
				fCollisionHeight = temp;
			}

			// now "apply" the transform
			fTransform = transform;

			// reposition the Sprite so that the location of the reference pixel in
			// the painter's coordinate system does not change as a result of changing
			// the transform. This means removing the previous transform location and
			// while moving to the new transform location.
			move(-x + getLeftEdge(), -y + getTopEdge());

			// Modify the collision rectangle so that it remains static relative
			// to the pixel data of the Sprite.  But don't do this now - delay
			// until collision detection since the collision rectangle must
			// move with the sprite.

			fImageCache = null;
			if (!fImage.isMutable()) createImageCache();
		}
	}

	private void createImageCache() {
		fImageCache = new Image[fFrameCount];
		int numHorzTiles = fImage.getWidth() / fFrameWidth;
		int numVertTiles = fImage.getHeight() / fFrameHeight;
		int index = 0;
		int y = 0;
		for (int i = 0; i < numVertTiles; i++) {
			int x = 0;
			for (int j = 0; j < numHorzTiles; j++) {
				fImageCache[index++] = Image.createImage(fImage, x, y, fFrameWidth, fFrameHeight, fTransform);
				x += fFrameWidth;
			}
			y += fFrameHeight;
		}
	}

	private int getLeftEdge() {
		//  optimize for x not changing -
		//no transform or mirror 180
		if (fTransform <= TRANS_MIRROR_ROT180) return 0;

		int x = 0;
		switch (fTransform) {
			case TRANS_MIRROR_ROT90:
			case TRANS_ROT90 :
				x = fRefPixelX + fRefPixelY - getWidth() + 1;
				break;

			case TRANS_MIRROR:
			case TRANS_ROT180:
				x = 2 * fRefPixelX - getWidth() + 1;
				break;

			case TRANS_MIRROR_ROT270:
			case TRANS_ROT270:
				x = fRefPixelX - fRefPixelY;
				break;

			default :
				break;
		}

		return x;
	}

	private int getTopEdge() {
		// optimize for y not changing -
		// no transform or mirror
		if (TRANS_NONE == fTransform || TRANS_MIRROR == fTransform) return 0;

		int y = 0;
		switch (fTransform) {
			case TRANS_MIRROR_ROT270:
			case TRANS_ROT90 :
				y = fRefPixelY - fRefPixelX;
				break;

			case TRANS_MIRROR_ROT180:
			case TRANS_ROT180:
				y = 2*fRefPixelY - getHeight() + 1;
				break;

			case TRANS_MIRROR_ROT90:
			case TRANS_ROT270:
				y = fRefPixelY + fRefPixelX - getHeight() + 1;
				break;

			default :
				break;
		}

		return y;
	}

	/*
	 * Answer a rectangle that represents the current frame
	 * of the image
	 */
	private Rectangle getCurrentRectangle() {
		// width and height are swapped for 90 and 270 transforms
		int width = (fTransform < TRANS_MIRROR_ROT270) ? getWidth() : getHeight();
		int height = (fTransform < TRANS_MIRROR_ROT270) ? getHeight() : getWidth();

		// determine the x,y coordinate of the current frame
		// in the image
		int numHorzFrames = fImage.getWidth() / width;
		int frame = fFrameSequence[fSequenceIndex];
		int srcX = (frame % numHorzFrames) * width;
		int srcY = (frame / numHorzFrames) * height;

		return new Rectangle(srcX, srcY, width, height);
	}

	/*
	 * reset the frame sequences to the default -
	 * {0,1,2,...,n} where n equals the number of frames
	 */
	private void createDefaultFrameSequence() {
		fIsDefaultSequence = true;
		fFrameSequence = new int[fFrameCount];
		for (int i = 0; i < fFrameSequence.length; i++) {
			fFrameSequence[i] = i;
		}
	}

	/*
	 * Answer the collision rectangle based on the current location
	 * and transform of the sprite
	 * @returns Rectangle - the current collision rectangle
	 */
	Rectangle _getCollisionRectangle() {
		// collision rectangle is always relative to the
		// sprite's current location
		int x = fCollisionX + getX();
		int y = fCollisionY + getY();
		// however, if the sprite is a mirror image then the
		// collision rectangle's left edge must also be a mirror image
		switch (fTransform) {
				case TRANS_MIRROR:
				case TRANS_MIRROR_ROT180:
					x = getX() + getWidth() - fCollisionX - fCollisionWidth;
					break;

				case TRANS_MIRROR_ROT270:
				case TRANS_MIRROR_ROT90:
					x = getX() + getHeight() - fCollisionX - fCollisionHeight;
					break;

				default :
					break;
			}

		return new Rectangle(x, y, fCollisionWidth, fCollisionHeight);
	}

	/*
	 * Answer whether or not the opaque pixels within this sprites collision
	 * rectangle collides with the opaque pixels withn the collision rectangle
	 * of another sprite.  In both cases, the current frame is used for the
	 * images.
	 */
	private boolean _collidesWith(Sprite aSprite, Rectangle collision) {
		// create a rectangle that is the intersection of the transformed sprite
		// and it's transformed collision rectangle
		Rectangle intersection = _getCollisionRectangle().intersection(
			new Rectangle(getX(), getY(), getWidth(), getHeight()));
		// the collision rectangle could be totally outside of the bounds of the
		// sprite for some reason
		if (intersection.isEmpty())
			return false;

		// now intersect the two smaller collision rectangles
		intersection = intersection.intersection(collision);
		if (intersection.isEmpty())
			return false;

		int[] argb = new int[intersection.width * intersection.height];
		getRGB(argb, intersection);

		// now that we have an image which matches the intersection of the collision
		// rectangle and the sprite, see if the sprite collides with this new image
		return aSprite._collides(argb, intersection);
	}

	private boolean _collides(int[] argb_i, Rectangle intersection) {
		boolean collides = false;

		// get the pixels for the current frame
		int[] argb = new int[argb_i.length];
		getRGB(argb, intersection);

		// get the pixels from the image using it's untransformed width and height
		int width = ((fTransform >= 4) ? intersection.height : intersection.width);
		int height = ((fTransform >= 4) ? intersection.width : intersection.height);
		// we have to compare the correct pixels of the transformed
		// sprite with the pixels of the image. The pixels from the
		// sprite are always placed into the array in row major order.
		// For example, a sprite with 4 columns and 3 rows:
		// compare pixels 0-11
		// left to right starting with the upper left corner
		int start = 0;
		int incr = 1;
		int next = width;
		switch (fTransform) {
			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 8-11, 4-7, 0-3
			// left to right starting with the bottom left corner
			case TRANS_MIRROR_ROT180 :
				start = width * (height-1);
				incr = 1;
				next = -width;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 3-0, 7-4, 11-8
			// right to left starting with the upper right corner
			case TRANS_MIRROR:
				start = width -1;
				incr = -1;
				next = width;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 11-0
			// right to left starting with the bottom right corner
			case TRANS_ROT180:
				start = (width * height) - 1;
				incr = -1;
				next = -width;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 0,4,8, 1,5,9, 2,6,10, 3,7,11
			// top to bottom starting with the upper left corner
			case TRANS_MIRROR_ROT270:
				start = 0;
				incr = width;
				next = 1;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 8,4,0, 9,5,1, 10,6,2, 11,7,3
			// bottom to top starting with the bottom left corner
			case TRANS_ROT90:
				start = width * (height -1);
				incr = -width;
				next = 1;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 3,7,11, 2,6,10, 1,5,9, 0,4,8
			// top to bottom starting with the top right corner
			case TRANS_ROT270:
				start = width - 1;
				incr = width;
				next = -1;
				break;

			// For example, a sprite with 4 colums and 3 rows:
			// compare pixels 11,7,3, 10,6,2, 9,5,1, 8,4,0
			// bottom to top starting with the bottom right corner
			case TRANS_MIRROR_ROT90:
				start = (width * height) -1;
				incr = -width;
				next = -1;
				break;

			default:
				break;
		}

		int index_i = 0;
		for (int i = start; index_i < argb_i.length && !collides; i = i + next) {
			// the end is at the top or bottom for 90 or 270 transforms, but
			// since width and height are swapped, we can always use the
			// width in this calculation
			int end = i + (incr * intersection.width);
			for (int index = i; index != end && !collides; index = index + incr) {
				collides = (((argb[index] & 0xff000000) & (argb_i[index_i] & 0xff000000)) != 0);
				index_i++;
			}
		}

		return collides;
	}

	void getRGB(int[] argb, Rectangle intersection) {
		// get the pixels from the image using it's untransformed width and height
		int width;
		int height;
		int offsetX;
		int offsetY;
		if (fTransform < 4) {
			width = intersection.width;
			height = intersection.height;
			offsetX = intersection.x - getX();
			offsetY = intersection.y - getY();
		} else {
			width = intersection.height;
			height = intersection.width;
			offsetX = intersection.y - getY();
			offsetY = intersection.x - getX();
		}

		// get the region that describes the current frame within the image
		Rectangle src = getCurrentRectangle();

		switch (fTransform) {
			case TRANS_ROT90 :
			case TRANS_MIRROR_ROT180 :
				offsetY = src.height -offsetY - height;
				break;

			case TRANS_ROT180 :
			case TRANS_MIRROR_ROT90 :
				offsetX = src.width - offsetX - width;
				offsetY = src.height - offsetY - height;
				break;

			case TRANS_ROT270 :
			case TRANS_MIRROR :
				offsetX = src.width - offsetX - width;
				break;

			default :
				break;
		}

		// get the pixels for the current frame
		fImage.getRGB(argb,
			0,
			width,
			src.x + offsetX,
			src.y + offsetY,
			width,
			height);
	}
}
