package javax.microedition.lcdui.game;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import javax.microedition.lcdui.*;
import com.ibm.ive.midp.*;

public class TiledLayer extends Layer {

	// grid of tiles laying out the background
	private int[][] fGrid;
	// the dimension of the background grid
	private int fColumns;
	private int fRows;
	// the image of tiles
	private Image fImage;
	// dimensions of a tile
	private int fTileWidth;
	private int fTileHeight;
	// number of static tiles
	private int fStaticCount;
	// index of last animated tile
	private int fLastAnimated;
	// the mapping of animated to static tiles
	private FastVector fAnimatedSet;

	private Image[] fImageCache;

	public TiledLayer(int columns, int rows, Image image, int tileWidth, int tileHeight) {
		if (rows < 1 || columns < 1) throw new IllegalArgumentException();

		// validate the image and save everything
		// about it
		createFrom(image, tileWidth, tileHeight, rows, columns, true);

		// save the grid dimensions
		fColumns = columns;
		fRows = rows;
	}

	/**
	 * Define an animated tile by associating it with a
	 * static tile (or 0 indicating no association).
	 * @param staticTileIndex the static tile this animated
	 * tile is associated with
	 * @return int the index (always negative) of the animated tile
	 * @throws IndexOutOfBoundsException if
	 * <code>staticTileIndex</code> is less than 0 or a static tile index
	 */
	public int createAnimatedTile(int staticTileIndex) {
		synchronized (fLock) {
			if (staticTileIndex < 0 || staticTileIndex > fStaticCount) throw new IndexOutOfBoundsException();
			fLastAnimated--;
			fAnimatedSet.addElement(new Integer(staticTileIndex));
			return fLastAnimated;
		}
	}

	/**
	 * Places the tile with the specified <code>tileIndex</code>
	 * in the background layout grid for the range specified by
	 * <code>row</code>, <code>colmun</code>, <code>numCols</code>,
	 * and <code>numRows</code>.
	 * @param col the zero based column in the grid
	 * @param row the zero based row in the grid
	 * @param numCols the number of columns to fill
	 * @param numRows the number of rows to fill
	 * @param tileIndex the tile which goes into the grid
	 * @throws IndexOutOfBoundsException if
	 * <code>row</code> or <code>col</code> is less than 0 or
	 * the range of cells specified is greater than the grid dimensions.
	 * <code>tileIndex</code> does not exist
	 * @throws IllegalArgumentException if
	 * <code>numCols</code> or <code>numRows</code> is less than 0
	 */
	public void fillCells(int col, int row, int numCols, int numRows, int tileIndex) {
		if (numCols < 0 || numRows < 0) throw new IllegalArgumentException();
		synchronized (fLock) {
			if (row < 0 || numRows + row > fRows || numRows + row < 0) throw new IndexOutOfBoundsException();
			if (col < 0 || numCols + col > fColumns || numCols + col < 0) throw new IndexOutOfBoundsException();
			if (tileIndex < fLastAnimated || tileIndex > fStaticCount) throw new IndexOutOfBoundsException();

			for (int rowIndex = row; rowIndex - row < numRows; rowIndex++) {
				for (int colIndex = col; colIndex - col < numCols; colIndex++) {
					fGrid[rowIndex][colIndex] = tileIndex;
				}
			}
		}
	}

	/**
	 * Answer the index of the static tile associated
	 * with <code>animatedTileIndex</code>
	 * @param animatedTileIndex index of the animated tile
	 * @return int index of the associated static tile
	 * @throws IndexOutOfBoundsException if
	 * <code>animatedTileIndex</code> is does not exist
	 */
	public int getAnimatedTile(int animatedTileIndex) {
		synchronized (fLock) {
			if (animatedTileIndex < fLastAnimated || animatedTileIndex > 0) throw new IndexOutOfBoundsException();

			int index = (animatedTileIndex + 1) * -1;
			return ((Integer) fAnimatedSet.elementAt(index)).intValue();
		}
	}

	/**
	 * Answers the index of the tile stored in the backgroud
	 * grid at position <code>row</code> and <code>col</code>.
	 * @param col the zero based column in the grid
	 * @param row the zero based row in the grid
	 * @return int the tile stored in the grid
	 * @throws IndexOutOfBoundsException if
	 * <code>row</code> or <code>col</code> is less than 0 or
	 * greater than the grid dimensions
	 */
	public int getCell(int col, int row) {
		synchronized (fLock) {
			if (row < 0 || row >= fRows)throw new IndexOutOfBoundsException();
			if (col < 0 || col >= fColumns) throw new IndexOutOfBoundsException();
			return fGrid[row][col];
		}
	}

	/**
	 * Answer the height of a single cell (equivalent
	 * to the height of a single tile when the receiver
	 * was created)
	 * @return int pixel height of a cell/tile
	 */
	public final int getCellHeight() {
		synchronized (fLock) {
			return fTileHeight;
		}
	}

	/**
	 * Answer the width of a single cell (equivalent
	 * to the width of a single tile when the receiver
	 * was created)
	 * @return int pixel width of a cell/tile
	 */
	public final int getCellWidth() {
		synchronized (fLock) {
			return fTileWidth;
		}
	}

	/**
	 * Answer the number of columns in the grid
	 * of tiles laying out the background
	 * @return int number of columns
	 */
	public final int getColumns() {
		synchronized (fLock) {
			return fColumns;
		}
	}

	/**
	 * Answer the number of rows in the grid
	 * of tiles laying out the background
	 * @return int number of rows
	 */
	public final int getRows() {
		synchronized (fLock) {
			return fRows;
		}
	}

	/**
	 * @see Layer#paint(Graphics)
	 */
	public final void paint(Graphics g) {
		if (g == null) throw new NullPointerException();

		synchronized (fLock) {
			// determine the number of tile columns
			// in the image of tiles
			int numHorzTiles = fImage.getWidth() / fTileWidth;

			// iterate through the grid
			for (int row = 0; row < fRows; row++) {
				for (int col = 0; col <fColumns; col++) {
					// which the tile goes into the background
					int tile = fGrid[row][col];
					if (tile < 0) tile = getAnimatedTile(tile);

					// skip the empty tiles
					if (tile != 0) {
						// x,y coordinate of where the tile
						// goes into the background image
						int destX = (col * fTileWidth) + getX();
						int destY = (row * fTileHeight) + getY();

						if (fImageCache != null) {
							Image tileImage = fImageCache[tile];
							g.drawImage(tileImage, destX, destY, 0);
						} else {
							// x,y coordinate of where the tile is in
							// image of tiles (tile# is 1 based)
							// remainder gives the tile's column
							int srcX = ((tile - 1) % numHorzTiles) * fTileWidth;
							// quotient gives the tile's row
							int srcY = ((tile - 1) / numHorzTiles) * fTileHeight;

							// place the tile in the background
							g.drawRegion(fImage, srcX, srcY, fTileWidth, fTileHeight, Sprite.TRANS_NONE, destX, destY, 0);
						}
					}
				}
			}
		}
	}

	/**
	 * Associate the animated tile at index <code>animatedTileIndex</code>
	 * with a different static tile - the one at index
	 * <code>staticTileIndex</code>
	 * @param animatedTileIndex index of animated tile
	 * @param staticTileIndex new static tile associated with the animated tile
	 * @throws IndexOutOfBoundsException if
	 * <code>animatedTileIndex</code> does not exist
	 * <code>staticTileIndex</code> does not exist or is less thatn 0
	 */
	public void setAnimatedTile(int animatedTileIndex, int staticTileIndex) {
		synchronized (fLock) {
			if (animatedTileIndex < fLastAnimated || animatedTileIndex > 0) throw new IndexOutOfBoundsException();
			if (staticTileIndex < 0 || staticTileIndex > fStaticCount) throw new IndexOutOfBoundsException();

			int index = (animatedTileIndex + 1) * -1;
			fAnimatedSet.setElementAt(new Integer(staticTileIndex), index);
		}
	}

	/**
	 * Places the tile with the specified <code>tileIndex</code>
	 * in the background layout grid at the specified <code>row</code>
	 * and <code>colmun</code>.
	 * @param col the zero based column in the grid
	 * @param row the zero based row in the grid
	 * @param tileIndex the tile which goes into the grid
	 * @throws IndexOutOfBoundsException if
	 * <code>row</code> or <code>col</code> is less than 0 or
	 * greater than the grid dimensions
	 * <code>tileIndex</code> does not exist
	 */
	public void setCell(int col, int row, int tileIndex) {
		synchronized (fLock) {
			if (row < 0 || row >= fRows) throw new IndexOutOfBoundsException();
			if (col < 0 || col >= fColumns) throw new IndexOutOfBoundsException();
			if (tileIndex < fLastAnimated || tileIndex > fStaticCount) throw new IndexOutOfBoundsException();
			fGrid[row][col] = tileIndex;
		}
	}

	/**
	 * Change the background by setting new static tiles. Clear the
	 * background grid and animated tiles when the new
	 * contains fewer tiles than the previous image.
	 * @param image the image which contains the new tiles
	 * @param tileWidth the width in pixels of each tile
	 * @param tileHeight the height in pixels of each tile
	 * @throws NullPointerException if <code>image</code> is <code>null</code>
	 * @throws IllegalArgumentException if
	 * <code>tileWidth</code> &lt. 1, or
	 * <code>tileHeight</code> &lt. 1, or
	 * <code>image</code> width is not a multiple of <code>tileWidth</code> or
	 * <code>image</code> height is not a multiple of <code>tileHeight</code>
	 */
	public void setStaticTileSet(Image image, int tileWidth, int tileHeight) {
		// validate the image and save everything
		// about it
		synchronized (fLock) {
			createFrom(image, tileWidth, tileHeight, fRows, fColumns, false);
		}
	}

	/*
	 * Create a background grid layout initially
	 * containing no tiles which also implies that
	 * there are no animated tiles yet
	 * @param rows - number of rows in the grid
	 * @param columns - number of columns in the grid
	 */
	private void initGrid(int rows, int columns) {
		// reset index of last animated tile
		fLastAnimated = 0;
		fAnimatedSet = new FastVector();

		// create empty background grid. It
		// int arrays are  filled with 0's when
		// created with new
		fGrid = new int[rows][columns];
	}

	/*
	 * Verify that the image and the tile size are valid to use.
	 * If it is okay then save the values.
	 * @param image - image consisting of tiles
	 * @param tileWidth - the width of each tile
	 * @param tileHeight - the height of each tile
	 * @param rows - the number of rows in the background image
	 * @param columns - the number of columns in the background image
	 */
	private void createFrom(Image image, int tileWidth, int tileHeight, int rows, int columns, boolean isNew) {
		if (image == null) throw new NullPointerException();

		int width = image.getWidth();
		int height = image.getHeight();
		if (tileWidth < 1) throw new IllegalArgumentException();
		if (tileHeight < 1) throw new IllegalArgumentException();
		if (width % tileWidth != 0) throw new IllegalArgumentException();
		if (height % tileHeight != 0) throw new IllegalArgumentException();

		// save the image
		synchronized (fLock) {
			fImage = image;
			fImageCache = null;
			// save tile dimensions
			fTileWidth = tileWidth;
			fTileHeight = tileHeight;
			// set the background image (grid) dimensions
			setWidth(tileWidth * columns);
			setHeight(tileHeight * rows);

			// save the static tile count for later comparisons.
			// (note this may not be set when creating a new TiledLayer)
			int savedTileCount = fStaticCount;
			// determine number of static tiles
			fStaticCount = (width / tileWidth) * (height / tileHeight);
			// create the background grid when creating new TiledLayer.
			// or, when we have fewer tiles than before clear out the
			// grid and animated tiles
			if (isNew || fStaticCount < savedTileCount) initGrid(rows, columns);
			if (!fImage.isMutable()) createImageCache();
		}
	}

	private void createImageCache() {
		/* Remember to leave space for the 0 tile (empty tile) */
		fImageCache = new Image[fStaticCount + 1];
		int numHorzTiles = fImage.getWidth() / fTileWidth;
		int numVertTiles = fImage.getHeight() / fTileHeight;
		int index = 1;
		int y = 0;
		for (int i = 0; i < numVertTiles; i++) {
			int x = 0;
			for (int j = 0; j < numHorzTiles; j++) {
				fImageCache[index++] = Image.createImage(fImage, x, y, fTileWidth, fTileHeight, Sprite.TRANS_NONE);
				x += fTileWidth;
			}
			y += fTileHeight;
		}
	}

	void getRGB(int[] argb, int x, int y, int width, int height) {
		synchronized (fLock) {
			int startCol = (x) / fTileWidth;
			int endCol = (x + width - 1) / fTileWidth;
			int startRow = (y) / fTileHeight;
			int endRow = (y + height -1) / fTileHeight;
			int diffColStart = x - (startCol * fTileWidth);
			int diffRowStart = y - (startRow * fTileHeight);

			// determine the number of tile columns
			// in the image of tiles
			int numHorzTiles = fImage.getWidth() / fTileWidth;

			int offset = 0;
			// iterate through the grid
			for (int row = startRow; row <= endRow; row++) {
				for (int col = startCol; col <= endCol; col++) {

					// which the tile goes into the background
					int tile = fGrid[row][col];
					if (tile < 0) tile = getAnimatedTile(tile);

					// skip the empty tiles
					if (tile != 0) {
						// x,y coordinate of where the tile is in
						// image of tiles (tile# is 1 based)
						// remainder gives the tile's column
						int srcX = ((tile - 1) % numHorzTiles) * fTileWidth;
						// quotient gives the tile's row
						int srcY = ((tile - 1) / numHorzTiles) * fTileHeight;

						int w = fTileWidth;
						if (col == startCol) {
							w = fTileWidth - diffColStart;
							srcX = srcX + diffColStart;
						}
						if (col == endCol) {
							w = w - (((endCol + 1) * fTileWidth) - (x + width));
						}

						int h = fTileHeight;
						if (row == startRow) {
							h = fTileHeight - diffRowStart;
							srcY = srcY + diffRowStart;
						}
						if (row == endRow) {
							h = h - (((endRow + 1) * fTileHeight) - (y + height));
						}

						fImage.getRGB(argb, offset, w, srcX, srcY, w, h);
						offset = offset + w;
					} else {
						offset = offset + fTileWidth;
					}
				}
			}
		}
	}
}
