package javax.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import java.io.*;
import java.util.*;

import javax.microedition.io.*;
import com.ibm.microedition.media.*;
import com.ibm.microedition.media.typechecker.*;
import com.ibm.oti.connection.file.*;

/**
 * Platform independent version of Manager class. All the platform dependent code is in SupportDatabase and PlayToneImpl.
 */
//TODO: Externalize the strings
public final class Manager {

	final public static String TONE_DEVICE_LOCATOR = "device://tone";
	final static String MIDI_DEVICE_LOCATOR = "device://midi";

	/**
	 * The class should not be allowed to be created.
	 */
	private Manager() {
	}

	/**
	 * @param note
	 *            0-127
	 * @param duration
	 *            Duration in milliseconds.
	 * @param volume
	 *            0-100
	 * @throws MediaException
	 */
	static public void playTone(int note, int duration, int volume) throws MediaException {
		if (note < 0 || note > 127 || duration <= 0)
			throw new IllegalArgumentException("Note out of range");

		if (duration <= 0) {
			throw new IllegalArgumentException("Duration should be a positive value");
		}
		if (volume < 0) {
			volume = 0;
		} else if (volume > 100) {
			volume = 100;
		}
		PlayToneImpl.playToneImpl(note, duration, volume);
	}

	/**
	 * Returns the supported content types for the specified protocol. If null is passed in, all content types supported by the implementation will be returned.
	 *
	 * @param protocol
	 *            the protocol for which one want to determine available content types.
	 *
	 * @return an array of Strings, which are the content types available for the protocol.
	 */
	public static String[] getSupportedContentTypes(String protocol) {
		Hashtable table = SupportDatabase.getSupportTable();

		if (protocol != null) {
			String[] types = (String[]) table.get(protocol);
			return types == null ? new String[0] : types;
		}

		// return all the supported content types
		Vector all = new Vector();
		Enumeration protocolArrays = table.elements();
		while (protocolArrays.hasMoreElements()) {
			String[] protocolArray = (String[]) protocolArrays.nextElement();
			for (int i = 0; i < protocolArray.length; i++) {
				if (all.contains(protocolArray[i])) continue; // ignore duplicates
				all.addElement(protocolArray[i]);
			}
		}
		String[] results = new String[all.size()];
		all.copyInto(results);
		return results;
	}

	/**
	 * Returns the supported protocols for the specified content type. If null is passed in, all protocols supported by the implementation will be returned.
	 *
	 * @param contentType
	 *            the content type for which one want to determine available protocls.
	 *
	 * @return an array of Strings, which are the protocols available for the content type.
	 */
	static public String[] getSupportedProtocols(String contentType) {
		Hashtable table = SupportDatabase.getSupportTable();
		String[] results;
		Enumeration supProt = table.keys();
		if (contentType == null) {
			results = new String[table.size()];
			int k = 0;
			while (supProt.hasMoreElements()) {
				results[k] = (String) supProt.nextElement();
				k++;
			}
			return results;
		}
		Vector protocols = new Vector(table.size());
		while (supProt.hasMoreElements()) {
			String supportedProtocol = (String) supProt.nextElement();
			if (findProtocolByContentType(contentType, table.get(supportedProtocol))) protocols.addElement(supportedProtocol);
		}
		results = new String[protocols.size()];
		protocols.copyInto(results);
		return results;
	}

	public static Player createPlayer(String locator) throws IOException, MediaException {

		if (locator == null) throw new IllegalArgumentException("Try with a valid locator");

		String locatorString = locator.toLowerCase();

		// return the basic player first
		if (locatorString.equals(TONE_DEVICE_LOCATOR)) return createClass(MediaConstants.MIME_TYPE_TONE, null);
		if (locatorString.equals(MIDI_DEVICE_LOCATOR)) return createClass(MediaConstants.MIME_TYPE_MIDI, null);

		// check if the protocol is supported
		String[] supportedProtocols = getSupportedProtocols(null);
		Vector protocols = new Vector();
		for (int t = 0; t < supportedProtocols.length; t++) {
			protocols.addElement(supportedProtocols[t]);
		}
		if (!protocols.contains((Object) getScheme(locatorString))) throw new MediaException("Unable to create player for the locator" + locator); //$NON-NLS-1$

		MediaDescriptor mediaDescriptor = createMediaDescriptor(locator);

		String mimeType = AbstractTypeChecker.createContentTypeCheckerChain().checkContentType(mediaDescriptor);
		if (mimeType != null) {
			// create the player for the media
			return createClass(mimeType, mediaDescriptor);
		} else {
			throw new IOException(MediaPlayer.getTranslatedString("Manager.msg5") + locator);
		}
	}

	public static Player createPlayer(InputStream stream, String type) throws IOException, MediaException {
		String mimeType = type;

		if (stream == null) throw new IllegalArgumentException();

		// set the media descriptor
		MediaDescriptor mediaDescriptor = new MediaDescriptor();
		mediaDescriptor.setMediaIS(stream);
		mediaDescriptor.setProtocol(MediaConstants.PROTOCOL_FILE);

		if (null == type) {// Try to determine the content type
			mimeType = AbstractTypeChecker.createContentTypeCheckerChain().checkContentType(mediaDescriptor);
		}
		if (mimeType != null) {
			// create the player for the media
			return createClass(mimeType, mediaDescriptor);
		} else {
			throw new MediaException("No player found");
		}
	}

	/*
	 * Convenience method to create a registered Player implementation for a specific media.
	 */
	private static Player createClass(String mimeType, MediaDescriptor mediaDescriptor) throws MediaException {
		String className = (String) SupportDatabase.getPlayers().get(mimeType);
		if (className == null) throw new MediaException("No player class registered for " + mimeType);

		try {
			Player player = (Player) Class.forName(className.trim()).newInstance();
			((MediaPlayer) player).setMediaDescriptor(mediaDescriptor);
			return player;
		} catch (Exception ex) {
			throw new MediaException("Unable to create class");
		}
	}

	private static boolean findProtocolByContentType(String contentType, Object types) {
		for (int i = 0; i < ((String[]) types).length; i++) {
			if (((String[]) types)[i].equals(contentType)) {
				return true;
			}
		}
		return false;
	}

	private static String getScheme(String locator) {
		/* Calling function should ensure locator is not null */
		int index = locator.indexOf(':');
		if (-1 == index) return null;
		return (locator.substring(0, index));
	}

	/*
	 * Creates a descriptor for a specific media.
	 */
	private static MediaDescriptor createMediaDescriptor(String locator) throws IOException, MediaException {
		String locatorString = locator.toLowerCase();
		InputStream is = null;
		MediaDescriptor mediaDescriptor = new MediaDescriptor();
		mediaDescriptor.setProtocol(getScheme(locatorString));

		if (getScheme(locatorString).equals(MediaConstants.PROTOCOL_FILE)) {
			String fullFileName = locator.substring(locator.indexOf(":") + 4, locator.length());
			mediaDescriptor.setFullFileName(fullFileName);
			is = new FileInputStream(fullFileName);

			if (is == null) throw new IOException(MediaPlayer.getTranslatedString("Manager.msg1"));

			mediaDescriptor.setMediaIS(is);
		} else if (getScheme(locatorString).equals(MediaConstants.PROTOCOL_HTTP) || getScheme(locatorString).equals(MediaConstants.PROTOCOL_HTTPS)) {
			ContentConnection connection = null;
			try {
				connection = (ContentConnection)Connector.open(locator, Connector.READ);
			} catch (Exception e) {
				throw new IOException(MediaPlayer.getTranslatedString("Manager.msg5") + locator);
			}
			mediaDescriptor.setHttpContentType(connection.getType());
			is = connection.openInputStream();
			if (is == null) throw new IOException(MediaPlayer.getTranslatedString("Manager.msg1"));

			mediaDescriptor.setMediaIS(is);
		} else if (getScheme(locatorString).equals(MediaConstants.PROTOCOL_DEVICE)) {
			// nothing to do, special cases are handling this
		}
		return mediaDescriptor;
	}
}
