package javax.microedition.midlet;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;

import com.ibm.oti.security.midp.*;

/**
 * <code>MIDlet</code> is the common superclass for the MIDP applications.<p>
 * All applications must subclass <code>MIDlet</code> and implement the
 * following methods: <code>startApp(), pauseApp(), destroyApp()</code>.
 *
 * @see MIDlet#startApp
 * @see MIDlet#pauseApp
 * @see MIDlet#destroyApp
 */
public abstract class MIDlet {

	int fState;
	int fId;  // A unique id for this midlet in this midlet suite

	IAppManager fAppManager;

	/**
	 * @throws SecurityException if the midlet cannot create other MIDlets
	 */
	protected MIDlet() {
		fState = MIDletManager.MIDLET_PAUSED;
		new MIDletAccessor(this); // this call sets up fAppManager

		fAppManager.incrementInstanceCount();
		if (!fAppManager.instantiationAllowed()) throw new SecurityException();
	}

	/**
	 * <code>destroyApp</code> is invoked by the Java Application Manager
	 * to notify the receiver of its termination. The receiver must
	 * perform the required clean-up. Throwing <code>MIDletStateChangeException</code>
	 * when <code>unconditional</code> is <code>true</code> is not an
	 * acceptable behaviour.
	 *
	 * @throws MIDletStateChangeException
	 */
	protected abstract void destroyApp(boolean unconditional) throws MIDletStateChangeException;

	/**
	 * <code>pauseApp</code> is invoked by the Java Application Manager
	 * to notify the receiver that it is being suspended.
	 */
	protected abstract void pauseApp();

	/**
	 * <code>startApp</code> is invoked by the Java Application Manager
	 * to notify the receiver that it is being started.
	 */
	protected abstract void startApp() throws MIDletStateChangeException;

	/**
	 * Retrieves the property <code>propertyName</code>
	 * from the Java Application Descriptor (JAD) or
	 * from the META-INF file. <p>
	 * <code>propertyName</code> cannot be null.
	 * @return String the value found for <code>propertyName</code>, null if
	 *                 none was found.
	 * @throws NullPointerException if <code>propertyName</code> is null.
	 */
	public final String getAppProperty(String propertyName){
		if (propertyName == null) throw new NullPointerException();
		return (String) MIDletManager.gProperties.get(propertyName);
	}

	/**
	 * Notifies the Java Application Manager the receiver
	 * has performed its clean up and is ready to be disposed of.<p>
	 * The JAM will not call <code>destroyApp</code>.
	 */
	public final void notifyDestroyed() {
		fAppManager.notifyDestroyed(this);
	}

	/**
	 * Notifies the Java Application Manager the receiver
	 * has entered its paused state.
	 */
	public final void notifyPaused(){
		if (fState == MIDletManager.MIDLET_ACTIVE) fAppManager.notifyPaused(this);
	}

	/**
	 * Requests the Java Application Manager to resume
	 * the receiver.
	 */
	public final void resumeRequest(){
		if (fState == MIDletManager.MIDLET_PAUSED) fAppManager.resumeRequest(this);
	}

	/**
	 * Return the status of permission.  The return value will
	 * be one of the following three values:
	 *
	 *  1 (allowed),
	 *  0 (denied),
	 * -1 (unknown)
	 *
	 * @since 2.0
	 */
	public final int checkPermission(String permission) {
		// get the midlet's domain
		String protectionDomainId = MIDletManager.getSecurityDomain();
		ProtectionDomain protectionDomain = SecurityPolicy.getInstance().getProtectionDomain(protectionDomainId);

		// get the permission from the security policy
		int domainPermission = protectionDomain.getHighestPermissionLevel(permission);

		// if permission doesn't exist or is denied in the security policy
		if (domainPermission == ProtectionDomain.PERMISSION_LEVEL_DENY) return 0;

		// if permission is allowed in the security policy
		if (domainPermission == ProtectionDomain.PERMISSION_LEVEL_ALLOW) return 1;

		// otherwise, this is a permission that the user must grant/deny so it is unknown
		return -1;
	}

	/**
	 * Request that the url be handled.  There are only two types
	 * of URLs which must be handled by the implementation.
	 *
	 * (1) MIDlet Suite - (JAD file or Jar file),
	 * (2) Request for phone dialing (as specified in RFC2806)
	 *
	 * @throws java.microedition.io.ConnectionNotFoundException if url cannot be handled
	 * @return true if the MIDlet suite must exit before the request can be processed
	 * @since 2.0
	 */
	public final boolean platformRequest(String url) throws javax.microedition.io.ConnectionNotFoundException {
		if (url == null) return false;

		if (url.equals("")) { //$NON-NLS-1$
			// cancel any pending requests
			PlatformRequest.cancelPendingRequests();
			return false;
		}

		return PlatformRequest.newRequest(this,url);
	}

	public boolean equals(Object obj) {
		if (obj instanceof MIDlet) return ((MIDlet)obj).fId == fId;

		return false;
	}
}
