package javax.microedition.rms;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2004  All Rights Reserved
 */

/**
 * This is an implementation of the recordEnumeration.
 * <p>
 * N.B.  Several methods return void and throw no exceptions, even though they must
 * call methods which can fail.  In such methods one might have an invalid bit which
 * is set, such that if a method which can throw exceptions is called, one can be,
 * indicating that the enumeration is in an invalid state.
 */
class RecordEnumerationImpl implements RecordEnumeration {

/**
 * RecordEnumeration itself must not implement RecordListener for then a user
 * could add or remove an enumeration object from the record store queue of
 * listeners without ever calling keepUpdated(true|false).
 * <p>
 * i.e. given a record store object rs, a user could otherwise call:
 * 	rs.addRecordListener((RecordListener)rs.getRecordEnumeration(null, null, false));
 * OR:
 *  ((RecordListener)rs.getRecordEnumeration(null, null, false)).recordAdded(...);
 *  <p>
 * By not having this class implement RecordListener directly these problems are avoided.
 */
class RecordListenerImpl implements RecordListener {
	RecordListenerImpl() {}

	/**
	 * @see javax.microedition.rms.RecordListener#recordAdded(javax.microedition.rms.RecordStore, int)
	 */
	public void recordAdded(RecordStore store, int recordId) {
		recordAddedRS(recordId);
	}
	/**
	 * @see javax.microedition.rms.RecordListener#recordChanged(javax.microedition.rms.RecordStore, int)
	 */
	public void recordChanged(RecordStore store, int recordId) {
		recordChangedRS(recordId);
	}
	/**
	 * @see javax.microedition.rms.RecordListener#recordDeleted(javax.microedition.rms.RecordStore, int)
	 */
	public void recordDeleted(RecordStore store, int recordId) {
		recordDeletedRS(recordId);
	}
}

/**
 *
 */
RecordStore store;
/**
 *
 */
RecordFilter filter;
/**
 *
 */
RecordComparator comparator;
/**
 *
 */
boolean keepUpdated;

/**
 *
 */
boolean destroyed = false;
/**
 *
 */
RecordListener listener = null;

/**
 * Constructs and answers a new RecordEnumeration object.  Also attempts to
 * construct the index.
 */
RecordEnumerationImpl(RecordStore store, RecordFilter filter, RecordComparator comparator, boolean keepUpdated)
{
	this.store = store;
	this.filter = filter;
	this.comparator = comparator;
	this.keepUpdated = keepUpdated;

	if (this.keepUpdated) {
		this.listener = new RecordListenerImpl();
		this.store.addRecordListener(listener);
	}
	build();
}

/**
 * Inner class vector.
 * java.util.vector was not used for it is a collection of objects,
 * and constantly wrapping and unwrapping ints with Integer objects did
 * not seem very desirable.
 * Also the order of elements in a vector can not be guaranteed.
 */
/**
 * index is an array of record indeces, and is the principal structure of
 * RecordEnumeration.
 */
int[] index = new int[0];
int size = 0; 	   //The size of the index.
int position = -1; //The current position within the index, used for next|previous methods.

/**
 * Deletes the value at the given location.  Shifts all subsequent values over one to
 * the left.
 */
void delete(int location)
{
	if (location < --size) {
		System.arraycopy(index, location + 1, index, location, size - location);
	};

	if (location <= position) {
		position--;
	};
}

/**
 * Performs a simple binary search to determine where to insert a new record into
 * the index.
 */
int findInsertPosition(byte[] record)
{
	if (comparator == null) return size;

	int start = 0;
	int end = size;
	int mid = (start + end)/2;

	try {
		synchronized(RecordStore.lock) {
			while(start != end){
				byte[] compareRecord = store.getRecord(index[mid]);
				int res = comparator.compare(record, compareRecord);
				switch (res) {
					case RecordComparator.EQUIVALENT :
						return mid;
					case RecordComparator.PRECEDES :
						end = mid;
						mid = (start + end)/2;
						break;
					case RecordComparator.FOLLOWS :
						start = mid + 1;
						mid = (start + end)/2;
						break;
					default :
						return size;
				}
			}
		}

	} catch (Exception e) {
		return -1;
	}
	return mid;
}

/**
 * Answers the location of the given value in the index.  This takes O(numRecord()) time, and
 * answers the size of the index if the value is not found.
 */
int findLocation(int recordId)
{
	int i;
	for (i = 0; i < size; i++) {
		if (index[i] == recordId){
			return i;
		}
	}
	return i;
}

/**
 * Inserts the given value into the index at the given location.  If a value exists
 * at the insert location first shift it and all following values to the right one.
 * If the index is not large enough, first increase it's size by 50% (Not a linear
 * amount?)
 */
void insert(int recordId, int insertLocation)
{
	if (++size > index.length) {
		int[] newIndex = new int[size * 2];
		if (insertLocation < size - 1){
			System.arraycopy(index, 0, newIndex, 0, insertLocation);
			System.arraycopy(index, insertLocation, newIndex, insertLocation + 1, size - insertLocation - 1);
		}
		else {
			System.arraycopy(index, 0, newIndex, 0, size-1);
		}
		index = newIndex;
	} else if (insertLocation < size - 1){
		System.arraycopy(index, insertLocation, index, insertLocation + 1, size - insertLocation - 1);
	}

	if (insertLocation <= position) {
		position++;
	}

	index[insertLocation] = recordId;
}

/**
 * Shrinks the index so that its length matches the number of records in the receiver.
 */
void trim()
{
	if (index.length > size) {
		int[] newIndex = new int[size];
		System.arraycopy(index, 0, newIndex, 0, size);
		index = newIndex;
	}
}

/**
 * Called when a record has been added to a record store.
 */
void recordAddedRS(int recordId)
{
	if ((filter != null) || (comparator != null)){
		byte[] data = null;
		try {
			data = store.getRecord(recordId);
		} catch (RecordStoreException e) {
		}

		boolean matches = false;
		if (filter == null) {
			matches = true;
		} else {
			matches = filter.matches(data);
		}

		if (matches) {
			insert(recordId, findInsertPosition(data));
		};
	} else {
		insert(recordId, size);
	}
}

/**
 * Called after a record in a record store has been changed.
 * The cases for this function are very complex, as the following
 * factors are relevant:
 *	-If a record did pass the filter but now does not it must be removed,
 * and vice versa.
 *  -If there is a comparator assume the sort order of the record has changed.
 */
void recordChangedRS(int recordId)
{
	if ((filter != null) || (comparator != null)) {
		int oldLocation = findLocation(recordId);
		byte[] data = null;
		try {
			data = store.getRecord(recordId);
		} catch (RecordStoreException e) {
		}

		boolean matches = false;
		if (filter == null) {
			matches = true;
		} else {
			matches = filter.matches(data);
		}

		boolean exists = false;
		if (oldLocation < size) {
			exists = true;
		} else {
			exists = false;
		}
		int oldPosition = position;
		if (exists && (!matches || (comparator != null))) {
			delete(oldLocation);
			exists = false;
		}

		if (!exists && matches) {
			int pos = findInsertPosition(data);
			insert(recordId, pos);
			if (oldLocation == pos) {
				position = oldPosition;
			}
		}
	}
}

/**
 * Called after a record has been deleted from a record store.
 */
void recordDeletedRS(int recordId)
{
	int location = findLocation(recordId);
	if (location != size) {
		delete(location);
	}

}

/**
 * Frees internal resources used by this RecordEnumeration.
 * What if you call destroy() and then rebuild()?
 */
public void destroy()
{
	store.removeRecordListener(listener);
	listener = null;
	store = null;
	filter = null;
	comparator = null;
	index = new int[0];
	destroyed = true;
}

/**
 * N.B.  A call to next element immediately followed by a call to previous element
 * will return the same result each time, i.e. there is not a previous, current and
 * next element, only a previous and next element.
 * A call to next or previous element will always return a value, for the next element
 * after the last is the first element, and vice versa.
 * A call to has(Next|Previous)Element will return false once one of the endpoints is
 * passed.
 */
/**
 * Answers true if more elements exist in the next direction.
 *
 * @return		<code>true</code> if there is a next element, <code>false</code> otherwise
 */
public boolean hasNextElement()
{
	if (destroyed) {
		 throw new IllegalStateException();
	}
	return (position < size - 1);
}

/**
 * Answers true if more elements exist in the previous direction.
 *
 * @return		<code>true</code> if there is a previous element, <code>false</code> otherwise
 */
public boolean hasPreviousElement()
{
	if (destroyed) {
		 throw new IllegalStateException();
	}
	if (position == -1) {
		 return size != 0;
	};
	return (position > 0);
}

/**
 * Answers true if the enumeration keeps its enumeration current with any changes in
 * the records.
 *
 * @return		<code>true</code> if the enumeration is updated, <code>false</code> otherwise
 */
public boolean isKeptUpdated()
{
	if (destroyed) {
		throw new IllegalStateException();
	}
	return keepUpdated;
}

/**
 * Used to set whether the enumeration will be keep its internal index up to date with the record store record
 * additions/deletions/changes.
 *
 * @param 		keepUpdated		keep the enumaration up-to-date
 */
public void keepUpdated(boolean keepUpdated)
{
	if (destroyed) {
		throw new IllegalStateException();
	}
	if (keepUpdated == this.keepUpdated) {
		return;
	}
	if (keepUpdated) {
		rebuild();
		listener = new RecordListenerImpl();
		store.addRecordListener(listener);
	} else {
		trim();
		store.removeRecordListener(listener);
		listener = null;
	}
	this.keepUpdated = keepUpdated;
}

/**
 * Answers a copy of the next record in this enumeration, where next is defined by the comparator and/or filter
 * supplied in the constructor of this enumerator.
 *
 * @return		the next record
 *
 * @throws		InvalidRecordIDException		If the next record is invalid
 * @throws		RecordStoreNotOpenException		If the record store is not open
 * @throws		RecordStoreException			If there is another record store exception
 */
public byte[] nextRecord()
	throws InvalidRecordIDException, RecordStoreNotOpenException, RecordStoreException
{
	if (destroyed) {
		throw new IllegalStateException();
	};
	if (position < size - 1) {
		position++;
	} else {
		throw new InvalidRecordIDException();
	}

	return store.getRecord(index[position]);
}

/**
 * Answers the recordId of the next record in this enumeration, where next is defined by the comparator and/or
 * filter supplied in the constructor of this enumerator.
 *
 * @return		the ID of the next record
 *
 * @throws		InvalidRecordIDException		If the record ID is invalid
 */
public int nextRecordId() throws InvalidRecordIDException
{
	if (destroyed) {
		throw new IllegalStateException();
	};

	if (position < size - 1) {
		position++;
	} else {
		throw new InvalidRecordIDException();
	}

	return index[position];
}

/**
 * Answers the number of records available in this enumeration's set.
 *
 * @return		the number of available records
 */
public int numRecords()
{
	if (destroyed) {
		throw new IllegalStateException();
	};

	return size;
}

/**
 * Answers a copy of the previous record in this enumeration, where previous is defined by the comparator and/or
 * filter supplied in the constructor of this enumerator.
 *
 * @return		the previous record
 *
 * @throws		InvalidRecordIDException		If the previous record is invalid
 * @throws		RecordStoreNotOpenException		If the record store is not open
 * @throws		RecordStoreException			If there is another record store exception
 */
public byte[] previousRecord()
	throws InvalidRecordIDException, RecordStoreNotOpenException, RecordStoreException
{
	if (destroyed) {
		throw new IllegalStateException();
	};
	if (position == -1) {
		 position = size;
	};
	if (position > 0) {
		position--;
	} else {
		throw new InvalidRecordIDException();
	}

	return store.getRecord(index[position]);
}

/**
 * Answers the recordId of the previous record in this enumeration, where previous is defined by the comparator
 * and/or filter supplied in the constructor of this enumerator.
 *
 * @return		the ID of the previous record
 *
 * @throws		InvalidRecordIDException		If the record ID is invalid
 */
public int previousRecordId() throws InvalidRecordIDException
{
	if (destroyed) {
		 throw new IllegalStateException();
	};
	if (position == -1) {
		 position = size;
	};

	if (position > 0) {
		position--;
	} else {
		throw new InvalidRecordIDException();
	}

	return index[position];
}

/**
 * Requests that the enumeration be updated to reflect the current record set.
 */
public void rebuild()
{
	if (destroyed) {
		throw new IllegalStateException();
	}
	if (!keepUpdated) {
		 build();
	}
}

/**
 * This method will directly manipulate the record store, just to save time.
 * Algorithm: Read through the source file one record at a time and, if they
 * pass the filter, insert them into the (possibly) sorted index.
 */
void build()
{
	try {
		synchronized(RecordStore.lock) {
			position = -1;
			int numRecords = store.getNumRecords();
			if (numRecords == 0) {
				size = 0;
				index = new int[0];
			} else {
				buildImplementation(numRecords);
			}
		}
	} catch (RecordStoreException e) {
	}

	if (!this.keepUpdated) {
		trim();
	}
}

/**
 * Returns the index point of the enumeration to the beginning.
 */
public void reset()
{
	if (destroyed) {
		throw new IllegalStateException();
	}
	position = -1;
}

private void buildImplementation(int numRecords)
	throws RecordStoreException
{
	int [] baseIndex = store.getBaseRecordIndexImplement();
	int baseIndexSize = baseIndex.length;
	if ((filter == null) && (comparator == null)) {
		index = baseIndex;
		size = baseIndexSize;
		return;
	}

	size = 0;
	byte[] data = null;
    for (int i = 0; i < baseIndexSize; i++) {
		data = store.getRecord(baseIndex[i]);
		if ((filter == null) || (filter.matches(data))){
			insert(baseIndex[i], findInsertPosition(data));
		}
	}
}

}
