package javax.microedition.rms;

import java.util.*;
import com.ibm.oti.midlet.help.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

/**
 * This class represents a record store.
 *
 * @author		OTI
 * @version		initial
 */
public class RecordStore {

static {
	com.ibm.oti.vm.VM.addShutdownClass(shutdownClass());
	new RecordStoreFullException();
}

private static Runnable shutdownClass() {
	return new Runnable() {
		public void run() {
			Enumeration values = openStores.elements();
			while (values.hasMoreElements()) {
				RecordStore store = (RecordStore)values.nextElement();
				try {
					if (store.openCount > 0) store.closeRecordStore();
				} catch (RecordStoreException e) {}
			}
		}};
}

/**
 * Allows authorization only to the current Midlet suite
 */
public static final int AUTHMODE_PRIVATE=0;

/**
 * Allows authorization to any Midlet suite
 */
public static final int AUTHMODE_ANY=1;

private boolean writable=true;
private boolean owner = true;

/**
 * Maintains a collection of all opened record stores.
 */
static Hashtable openStores = new Hashtable();
/**
 * An object used for synchronizing methods of the receiver.
 */
static Object lock = new Object();
/**
 * Maintains a collection of all record Listeners of the receiver.
 */
Vector listeners = new Vector();
/**
 * The number of times the receiver has been opened minus the number
 * of times it has been closed.
 */
short openCount = 0;
/**
 * The name of the record store.
 */
String name;
/**
 * The record Id of the next record added to the receiver.
 */
int nextRecordID = 1;
/**
 * The midlet suite name and vendor name.
 */
private MidletSuiteId midletSuiteId;

/**
 *  Constructs a new instance of the record store.
 * The constructor does not allocate any storage memory for the RS.
 *
 * @author		OTI
 * @version		initial
 */
private RecordStore(MidletSuiteId midletSuiteId) {
	this.midletSuiteId = midletSuiteId;
}

/**
 *  Adds a new record to the RecordStore and returns a new recordID.
 *
 *
 * @author		OTI
 * @version		initial
 *
 * @param		data 		the record buffer
 * @param 		offset 		buffer offset
 * @param		numBytes 	the number of bytes to write
 * @return		new record ID
 *
 * @throws 		RecordStoreNotOpenException 	If the RecordStore is closed
 * @throws 		RecordStoreException 			If the transaction fails for any other reasons
 * @throws 		RecordStoreFullException 		If the transaction fails due to insufficient memory
 */
public int addRecord(byte[] data, int offset, int numBytes)
	throws RecordStoreException, RecordStoreNotOpenException, RecordStoreFullException
{
	synchronized(lock){
		if (openCount == 0)
			throw new RecordStoreNotOpenException();
		if (!writable)
			throw new SecurityException();
		// add record
		int addRecordID = addRecordImplement(data, offset, numBytes);
		nextRecordID = addRecordID + 1;
		// notify listeners of adding record
		if (listeners.size() > 0) {
			Enumeration elemEnum = listeners.elements();
			while (elemEnum.hasMoreElements()) {
				((RecordListener)elemEnum.nextElement()).recordAdded(this, addRecordID);
			}
		}
		return addRecordID;
	}
}

/**
 *  Registers the specified RecordListener with this RecordStore.
 * If the RecordListener has already been, it will not be added
 * a second time.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		listener 	RecordListener
 */
public void addRecordListener(RecordListener listener)
{
	synchronized(listeners) {
		if (!listeners.contains(listener))
			listeners.addElement(listener);
	}
}

/**
 *  Closes the RecordStore and removes all the registered listeners.
 * The RecordStore will not be closed until closeRecordStore is called
 * as many time as openRecordStore was called.
 *
 * @author		OTI
 * @version		initial
 *
 * @throws 		RecordStoreException 			if the transaction fails
 * @throws 		RecordStoreNotOpenException		if the RecordStore is closed
 */
public void closeRecordStore()
	throws RecordStoreNotOpenException, RecordStoreException
{
	synchronized(lock) {
		if (openCount == 0) {
			throw new RecordStoreNotOpenException();
		}
		// the last reference to the RecordStore - close the RecordStore
		try {
			if (openCount == 1) {
				OpenStoresKey tempKey = new OpenStoresKey(midletSuiteId, name);
				openStores.remove(tempKey);
				closeRecordStoreImplement();
				listeners.removeAllElements();
			}
		} finally {
			openCount--;
		}
	}
}

/**
 *  Deletes the specified record from the record store
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordId	the record ID
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 * @throws 		RecordStoreException 			if the transaction fails for any other reasons
 * @throws 		RecordStoreFullException 		if the transaction fails due to unsufficient memory
 * @throws		InvalidRecordIDException		if the transaction id is invalid
 */
public void deleteRecord(int recordId)
	throws RecordStoreNotOpenException, InvalidRecordIDException, RecordStoreException
{
	synchronized(lock){
		if (openCount == 0)
			throw new RecordStoreNotOpenException();
		if (!writable)
			throw new SecurityException();
		// delete record
		deleteRecordImplement(recordId);
		// notify listeners of deleting record.
		if (listeners.size() > 0) {
			Enumeration elemEnum = listeners.elements();
			while (elemEnum.hasMoreElements()) {
				((RecordListener)elemEnum.nextElement()).recordDeleted(this, recordId);
			}
		}
	}
}

/**
 *  Deletes the specified RecordStore
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordStoreName 	the RecordStore name
 *
 * @throws 		RecordStoreException 			if a RecordStore-releated exception occured
 * @throws 		RecordStoreNotFoundException 	if the specified RecordStore is not found
 */
public static void deleteRecordStore(String recordStoreName)
	throws RecordStoreException, RecordStoreNotFoundException
{
	synchronized(lock){
		MidletSuiteId midletSuiteId = MidletLoader.getMidletSuiteId();
		RecordStore store =
			(RecordStore) openStores.get(new OpenStoresKey(midletSuiteId, recordStoreName));
		// the RecordStore is currently open by a MIDlet when this method is called
		if (store != null)
			throw new RecordStoreException();
		deleteRecordStoreImplement(midletSuiteId, recordStoreName);
	}
}

/**
 *  Returns a RecordEnumeration built with the specified RecordComparator and RecordFilter.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		filter		RecordFilter
 * @param 		comparator 	RecordComparator
 * @param		keepUpdated boolean
 * @return		RecordEnumeration
 *
 * @throws 		RecordStoreNotOpenException		if the RecordStore is closed
 */
public RecordEnumeration enumerateRecords(RecordFilter filter, RecordComparator comparator, boolean keepUpdated)
	throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();
		return new RecordEnumerationImpl(this, filter, comparator, keepUpdated);
	}
}

/**
 *  Returns the time in miliseconds(in the same format as the vm) when
 * the RecordStore was last modified.
 * Record added/modified/deleted as well as RecordStore open/close events
 * cause the last modified time update.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the last modified time
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public long getLastModified() throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getLastModifiedImplement();
	}
}

/**
 *  Returns the record store name
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the RecordStore name
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public String getName()
	throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return name;
	}
}

/**
 *  Returns the ID that will be given to the next record added to the store.
 * Records are numerated starting from 1. Record IDs are not reused
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next record ID
 *
 * @throws 		RecordStoreNotOpenException 	if the RecorStore is closed
 * @throws 		RecordStoreException 			if the transaction fails for any other reason
 */
public int getNextRecordID()
	throws RecordStoreNotOpenException, RecordStoreException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return nextRecordID;
	}
}

/**
 *  Returns the number of records in the RecordStore.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the RecordStore size
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public int getNumRecords()
	throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getNumRecordsImplement();
	}
}

/**
 *  Returns the specified record
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordId 	the record ID
 * @return		the record data
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 * @throws 		InvalidRecordIDException 		if record with the specified ID is not found
 * @throws 		RecordStoreException 			if the transaction fails for any other reason
 */
public byte[] getRecord(int recordId)
	throws RecordStoreNotOpenException,	InvalidRecordIDException, RecordStoreException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getRecordImplement(recordId);
	}
}

/**
 *  Copies the specified record to the buffer and returns the number of bytes copied.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordId	the record ID
 * @param		buffer		the buffer array
 * @param 		offset		buffer offset
 * @return		the number of bytes copied
 *
 * @throws 		RecordStoreNotOpenException		if the RecordStore is closed
 * @throws 		RecordStoreException 			if the transaction fails for any other reasons
 * @throws 		InvalidRecordIDException 		if record with the specified ID is not found
 * @throws 		ArrayIndexOutOfBoundsException 	if the buffer size is less than the record size
 */
public int getRecord(int recordId, byte[] buffer, int offset)
	throws RecordStoreNotOpenException, InvalidRecordIDException,
		   RecordStoreException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getRecordImplement(recordId, buffer, offset);
	}
}

/**
 *  Returns the size of the specified record.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordId	the record ID
 * @return		the record size
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordSized is closed
 * @throws 		InvalidRecordIDException 		if record with the specified ID is not found
 * @throws 		RecordStoreException 			if the transaction fails for any other reasons
 */
public int getRecordSize(int recordId)
	throws RecordStoreNotOpenException, InvalidRecordIDException, RecordStoreException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException(name);

		return getRecordSizeImplement(recordId);
	}
}

/**
 *  Returns the record store size in bytes.
 * (That includes the amount of space occupied by the RecordStore
 * implementation-specific structures)
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the RecordStore size
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public int getSize()
	throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getSizeImplement();
	}
}

/**
 *  Returns the number of bytes available for this RecordStore to grow.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the number of bytes available
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public int getSizeAvailable() throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getSizeAvailableImplement();
	}
}

/**
 *  Returns the RecordStore version number. Version number is incremented by 1 every
 * time a record is added/deleted/modified.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the RecordStore version
 *
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 */
public int getVersion() throws RecordStoreNotOpenException
{
	synchronized(lock) {
		if (openCount == 0)
			throw new RecordStoreNotOpenException();

		return getVersionImplement();
	}
}

/**
 *  Returns an array of RecordStore names belonging to this MIDlet suite.
 * (null is returned if there are none). The order of the RecordStore
 * names is implementation dependet.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		an array with RecordStore names
 */
public static String[] listRecordStores()
{
	synchronized(lock) {
		return listRecordStoresImplement();
	}
}

/**
 * Opens (and possibly creates) a record store with the specified name.
 * RecordStore names are unique across the MIDlets of the same suite.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordStoreName		the RecordStore name
 * @param 		createIfNecessary 	if the RecordStore does not exist and this argument is true, it will be created
 * @return		the record store
 *
 * @throws 		RecordStoreException 			if the transaction fails
 * @throws 		RecordStoreFullException 		if unsufficient memory
 * @throws 		RecordStoreNotFoundException 	if the RecordStore is not found and the create option is off
 */
public static RecordStore openRecordStore(String recordStoreName, boolean createIfNecessary)
	throws RecordStoreException, RecordStoreFullException, RecordStoreNotFoundException
{
	synchronized(lock) {
		if (recordStoreName.length() > 32)
			// K01a7 = Name too long: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01a7", recordStoreName));
		MidletSuiteId midletSuiteId = MidletLoader.getMidletSuiteId();
		OpenStoresKey key = new OpenStoresKey(midletSuiteId, recordStoreName);
		RecordStore store = (RecordStore) openStores.get(key);
		if (store != null) {
			store.openCount++;
			return store;
		}
			store = new RecordStore(midletSuiteId);
			store.openCount = 1;
			store.openRecordStoreImplement(recordStoreName, createIfNecessary, 0, false);
			openStores.put(key, store);
			return store;
	}
}

/**
 * Opens (and possibly creates) a record store with the specified name with the
 * possibility of allowing other MIDlet suites access.
 * RecordStore names are unique across the MIDlets of the same suite.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordStoreName		the RecordStore name
 * @param 		createIfNecessary 	if the RecordStore does not exist and this argument is true, it will be created
 * @param		authMode 			to use for created store, use either AUTHMODE_ANY or AUTHMODE_PRIVATE
 * @param		writable 			indicating whether the RecordStore is writable to any authorized MIDlet suites.
 * 									If the RecordStore exists, this parameter is ignored.
 * @return		the record store
 *
 * @throws 		RecordStoreException 			if the transaction fails
 * @throws 		RecordStoreFullException 		if unsufficient memory
 * @throws 		RecordStoreNotFoundException 	if the RecordStore is not found and the create option is off
 * @throws 		IllegalArgumentException 		if authMode or writable are invalid
 *
 * @since		MIDP 2.0
 */
public static RecordStore openRecordStore(String recordStoreName, boolean createIfNecessary, int authMode, boolean writable)
	throws RecordStoreException, RecordStoreFullException, RecordStoreNotFoundException
{
	synchronized(lock) {
		if (recordStoreName.length() > 32)
			// K01a7 = Name too long: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01a7", recordStoreName));
		if (authMode < 0 || authMode > 1)
			// K01ca = authMode is invalid
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01ca"));
		MidletSuiteId midletSuiteId = MidletLoader.getMidletSuiteId();
		OpenStoresKey key = new OpenStoresKey(midletSuiteId, recordStoreName);
		RecordStore store = (RecordStore) openStores.get(key);
		if (store != null) {
			store.openCount++;
			return store;
		}
			store = new RecordStore(midletSuiteId);
			store.openCount = 1;
			store.openRecordStoreImplement(recordStoreName, createIfNecessary, authMode, writable);
			openStores.put(key, store);
			return store;
	}
}

/**
 * Opens a named record store associated with a particular MIDlet suite. This will only
 * be successful if the MIDlet suite which originally created the RecordStore
 * granted access.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordStoreName		the RecordStore name
 * @param		vendorName			identifying the Vendor of the RecordStore to open
 * @param		suiteName			identifying the MIDlet suite name of the RecordStore to open
 * @return		the record store
 *
 * @throws 		RecordStoreException 			if the transaction fails
 * @throws 		RecordStoreNotFoundException 	if the RecordStore is not found
 * @throws		SecurityException 				if the MIDlet suite is not allowed access to the particular RecordStore
 * @throws 		IllegalArgumentException 		if the recordStoreName is not valid
 *
 * @since		MIDP 2.0
 */
public static RecordStore openRecordStore(String recordStoreName, String vendorName, String suiteName)
	throws RecordStoreException, RecordStoreNotFoundException
{
	synchronized(lock) {
		if (recordStoreName.length() > 32)
			// K01a7 = Name too long: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01a7", recordStoreName));
		MidletSuiteId midletSuiteId = new MidletSuiteId(suiteName, vendorName);
		RecordStore store =
			(RecordStore) openStores.get(new OpenStoresKey(midletSuiteId, recordStoreName));
		if (store != null) {
			store.openCount++;
			return store;
		}
			store = new RecordStore(midletSuiteId);
			store.openCount = 1;
			if (midletSuiteId.equals(MidletLoader.getMidletSuiteId()))
				store.openRecordStoreImplement(recordStoreName, false, AUTHMODE_PRIVATE, false);
			else
				store.openRecordStoreImplement(recordStoreName);
			openStores.put(new OpenStoresKey(midletSuiteId, recordStoreName), store);
			return store;
	}
}

/**
 * Changes which access mode is applicable to this RecordStore
 *
 * @author		OTI
 * @version		initial
 *
 * @param		authMode 	to use for created store, use either AUTHMODE_ANY or AUTHMODE_PRIVATE
 * @param		writable 	indicating whether the RecordStore is writable to any authorized MIDlet suites.
 *
 * @throws 		RecordStoreException 		if the transaction fails
 * @throws 		SecurityException 			if this MIDlet is unauthorized to make the change
 * @throws 		IllegalArgumentException 	if authMode is invalid
 *
 * @since		MIDP 2.0
 */
public void setMode(int authMode, boolean writable) throws RecordStoreException {
		if (authMode < AUTHMODE_PRIVATE || authMode > AUTHMODE_ANY)
			// K01ca = authMode is invalid
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01ca"));
		if (!owner)
			throw new SecurityException();
		if (openCount == 0)
			throw new RecordStoreNotOpenException();
		setModeImplement(authMode, writable);
}

/**
 *  Removes the specified RecordListener from the collection of listeners.
 * If the RecordLister doesn't exist, this method does nothing
 *
 * @author		OTI
 * @version		initial
 *
 * @param		listener 	RecordListener
 */
public void removeRecordListener(RecordListener listener)
{
	synchronized (listeners){
		listeners.removeElement(listener);
	}
}

/**
 *  Modifies the record with the specified ID
 *
 * @author		OTI
 * @version		initial
 *
 * @param		recordId int the record to modify
 * @param		newData byte[] the new record data
 * @param 		offset int buffer offset
 * @param		numBytes int the number of bytes to be written
 *
 * @throws 		InvalidRecordIDException 		if a record with the specified ID is not found
 * @throws 		RecordStoreNotOpenException 	if the RecordStore is closed
 * @throws 		RecordStoreException 			if the transaction fails for any other reasons
 * @throws 		RecordStoreFullException 		if the transaction fails due to unsufficient memory
 */
public void setRecord(int recordId, byte[] newData,int offset, int numBytes)
	throws RecordStoreNotOpenException, InvalidRecordIDException,
		   RecordStoreException, RecordStoreFullException
{
	synchronized(lock){
		if (openCount == 0)
			throw new RecordStoreNotOpenException();
		if (!writable)
			throw new SecurityException();
		// change record
		setRecordImplement(recordId, newData, offset, numBytes);
		// notify listeners of changing record
		Enumeration elemEnum = listeners.elements();
		while (elemEnum.hasMoreElements()) {
			((RecordListener)elemEnum.nextElement()).recordChanged(this, recordId);
		}
	}
}

////
// The current version of the receiver, incremented each time the
// reciever is modified.
////
int storeVersion;
////
// The current number of records in the receiver.
////
int storeNumRecords;
////
// A descriptor for the receiver's open source file.
////
private long sourceFD = -1;
////
//
////
private byte[] path;
////
// A pointer to dbHeaderBuffer that is used by the native to store the RecordStore database headers.
////
private long dbHeaderPtr = -1;

////
//	addRecordImpl
////
private native int addRecordImpl(byte[] data, int offset, int numBytes,
								 long sourceFD, long dbHeaderPtr)
	throws RecordStoreException, RecordStoreNotOpenException, RecordStoreFullException;

private int addRecordImplement(byte[] data, int offset, int numBytes)
	throws RecordStoreException, RecordStoreNotOpenException, RecordStoreFullException
{
	int addRecordID =0;
	try {
		addRecordID = addRecordImpl(data, offset, numBytes, sourceFD, dbHeaderPtr);
	} catch (RecordStoreException e) {
		if (e instanceof RecordStoreNotOpenException)
			throw e;
		throw new RecordStoreFullException();
	}

	storeVersion++;
	storeNumRecords++;

	return addRecordID;
}

////
//	closeRecordStore
////
private native void closeRecordStoreImpl(long sourceFD, long dbHeaderPtr)
	throws RecordStoreNotOpenException, RecordStoreException;

private void closeRecordStoreImplement()
	throws RecordStoreNotOpenException, RecordStoreException
{
	closeRecordStoreImpl(sourceFD, dbHeaderPtr);
	sourceFD = -1;
}

////
//	deleteRecord
////
private native void deleteRecordImpl(int recordID, long sourceFD, long dbHeaderPtr)
	throws RecordStoreException;

private void deleteRecordImplement(int recordID)
	throws RecordStoreNotOpenException, InvalidRecordIDException, RecordStoreException
{
	deleteRecordImpl(recordID, sourceFD, dbHeaderPtr);
	storeVersion++;
	storeNumRecords--;
}

////
//	deleteRecordStore
////
private static void deleteRecordStoreImplement(MidletSuiteId midletSuiteId, String recordStoreName)
	throws RecordStoreException, RecordStoreNotFoundException
{
	MidletLoader.acquireStore();
	try {
		if (!MidletLoader.storeExists(midletSuiteId, recordStoreName))
			throw new RecordStoreNotFoundException();
		MidletLoader.removeStore(midletSuiteId, recordStoreName);
	} finally {
		MidletLoader.releaseStore();
	}
}

////
//	getBaseRecordIndexImpl
////
//
// Answers an array of recordIds representing each record in the receiver, sorted
// by recordId.  This is the index which would be returned for an enumeration
// with no RecordComparator or RecordFilter.
////
private native int[] getBaseRecordIndexImpl(long sourceFD, long dbHeaderPtr)
	throws RecordStoreException;

int[] getBaseRecordIndexImplement()
	throws RecordStoreException
{
	return getBaseRecordIndexImpl(sourceFD, dbHeaderPtr);
}

////
//	getLastModified
////
private native long getLastModifiedImpl(long sourceFD, long dbHeaderPtr)
	throws RecordStoreNotOpenException;

private long getLastModifiedImplement()
	throws RecordStoreNotOpenException
{
	return getLastModifiedImpl(sourceFD, dbHeaderPtr);
}

////
//	getNumRecords
////
private int getNumRecordsImplement()
	throws RecordStoreNotOpenException
{
	return storeNumRecords;
}

////
//	getRecord
////
private native byte[] getRecordImpl(int recordID, long sourceFD, long dbHeaderPtr)
	throws RecordStoreNotOpenException,	InvalidRecordIDException, RecordStoreException;

private byte[] getRecordImplement(int recordID)
	throws RecordStoreNotOpenException,	InvalidRecordIDException, RecordStoreException
{
	return getRecordImpl(recordID, sourceFD, dbHeaderPtr);
}

////
//	getRecord
////
private native int getRecordImpl(int recordId, byte[] buffer, int offset, long sourceFD, long dbHeaderPtr)
	throws RecordStoreNotOpenException, InvalidRecordIDException,
		   RecordStoreException, ArrayIndexOutOfBoundsException;

private int getRecordImplement(int recordId, byte[] buffer, int offset)
	throws RecordStoreNotOpenException, InvalidRecordIDException,
		  RecordStoreException, ArrayIndexOutOfBoundsException
{
	return getRecordImpl(recordId, buffer, offset, sourceFD, dbHeaderPtr);
}

////
//	getRecordSize
////
private native int getRecordSizeImpl(int recordID, long sourceFD, long dbHeaderPtr)
	throws InvalidRecordIDException, RecordStoreException;

private int getRecordSizeImplement(int recordID)
	throws RecordStoreNotOpenException, InvalidRecordIDException, RecordStoreException
{
	return getRecordSizeImpl(recordID, sourceFD, dbHeaderPtr);
}

////
//	getSize
////
private native int getSizeImpl(long sourceFD)
	throws RecordStoreNotOpenException;

private int getSizeImplement() throws RecordStoreNotOpenException
{
	return getSizeImpl(sourceFD);
}

////
//	getSizeAvailable
////
private int getSizeAvailableImplement() throws RecordStoreNotOpenException
{
	return Integer.MAX_VALUE - getSizeImpl(sourceFD);
}

////
//	getVersion
////
private int getVersionImplement() throws RecordStoreNotOpenException
{
	return storeVersion;
}

////
//	listRecordStores
////
private synchronized static native byte[] [] listRecordStoresImpl(byte[] directory);

////
//	listRecordStores
////
private static String[] listRecordStoresImplement()
{
	return MidletLoader.listStores(MidletLoader.getMidletSuiteId());
}

////
//	openRecord
////
private native void openRecordStoreImpl(byte[] storePath, boolean createIfNecessary)
	throws RecordStoreException;

private void openRecordStoreImplement(String recordStoreName)
	throws RecordStoreException, RecordStoreFullException, RecordStoreNotFoundException
{
	MidletLoader.acquireStore();
	try {
		if (!MidletLoader.storeExists(midletSuiteId, recordStoreName))
			throw new RecordStoreNotFoundException();
		if (!MidletLoader.storeAccessible(midletSuiteId, recordStoreName))
			throw new SecurityException();
		owner = false;
		writable = MidletLoader.storeWritable(midletSuiteId, recordStoreName);
		this.name = recordStoreName;
		this.path = com.ibm.oti.util.Util.getBytes(MidletLoader.getStorePath(midletSuiteId, recordStoreName));
		try {
			this.openRecordStoreImpl(this.path, false);
		} catch (RecordStoreException e) {
			// Create failed, remove the entry
			MidletLoader.removeStore(midletSuiteId, recordStoreName);
			throw e;
		}
	} finally {
		MidletLoader.releaseStore();
	}
}

// Strictly called by current midlet suite API's only
private void openRecordStoreImplement(String recordStoreName, boolean createIfNecessary, int authMode, boolean writable)
	throws RecordStoreException, RecordStoreFullException, RecordStoreNotFoundException
{
	MidletLoader.acquireStore();
	try {
		boolean storeExisted = MidletLoader.storeExists(midletSuiteId, recordStoreName);
		if (!createIfNecessary && !storeExisted)
			throw new RecordStoreNotFoundException();
		this.name = recordStoreName;
		this.path = com.ibm.oti.util.Util.getBytes(MidletLoader.getStorePath(midletSuiteId, recordStoreName));
		try {
			this.openRecordStoreImpl(this.path, createIfNecessary);
		} catch (RecordStoreException e) {
			if (createIfNecessary) {
				// Create failed, remove the entry
				MidletLoader.removeStore(midletSuiteId, recordStoreName);
			}
			throw e;
		}
		if (!storeExisted)
			MidletLoader.setFlags(midletSuiteId, recordStoreName, authMode, writable);
	} finally {
		MidletLoader.releaseStore();
	}
}

////
//	setMode
////
private void setModeImplement(int authMode, boolean writable) throws RecordStoreException {
	MidletLoader.acquireStore();
	try {
		if (!MidletLoader.storeExists(midletSuiteId, this.name)) {
			// Assume that if it doesn't exist in the default view then throw the exeption
			throw new SecurityException();
		}
		MidletLoader.setFlags(midletSuiteId, name, authMode, writable);
		} finally {
			MidletLoader.releaseStore();
		}
}

////
//	setRecord
////
private native void setRecordImpl(int recordId, byte[] newData, int offset, int numBytes,
								  long sourceFD, long dbHeaderPtr)
	throws InvalidRecordIDException, RecordStoreException, RecordStoreFullException;

private void setRecordImplement(int recordId, byte[] newData,int offset, int numBytes)
	throws InvalidRecordIDException, RecordStoreException, RecordStoreFullException
{
	try {
		setRecordImpl(recordId, newData, offset, numBytes, sourceFD, dbHeaderPtr);
	} catch (RecordStoreException e) {
		if (e instanceof InvalidRecordIDException)
			throw e;
		throw new RecordStoreFullException();
	}
	storeVersion++;
}

private static final class OpenStoresKey {
	String suiteId;
	String recordStoreName;
	OpenStoresKey(MidletSuiteId midletSuiteId, String recordStoreName) {
		this.suiteId = midletSuiteId.getId() + "KEN";
		this.recordStoreName = recordStoreName;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return suiteId.hashCode() + ((recordStoreName !=  null) ? recordStoreName.hashCode() : 0);
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		if (obj == null) return false;
		if (this == obj) return true;
		if ( ! (obj instanceof OpenStoresKey) ) return false;

		OpenStoresKey other = (OpenStoresKey)obj;
		if (recordStoreName == null) {
			return ((suiteId.equals(other.suiteId)) && (other.recordStoreName == null));
		} else {
			return ((suiteId.equals(other.suiteId)) && recordStoreName.equals(other.recordStoreName));
		}
	}
}

}
