
/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

package com.ibm.oti.io;

import java.io.ByteArrayOutputStream;
import com.ibm.oti.util.BinarySearch;
import com.ibm.oti.io.CharacterConverter_EUC_JP;

class CharacterConverter_ISO2022JP extends CharacterConverter {
	private boolean isModal = true;
	private int countMode = ASCII;
	private int byteMode = ASCII;
	private int charMode = ASCII;

	private static final int UNKOWN = 0;
	private static final int ASCII = 1;
	private static final int ROMAN = 2;
	private static final int JIS6226 = 3;
	private static final int JIS208 = 4;
	private static final int JIS201 = 5;

	private static final byte[] EMPTY = new byte[0];
	private static final byte[] ESC_ASCII = new byte[] {0x1b, '(', 'B'};
	private static final byte[] ESC_ROMAN = new byte[] {0x1b, '(', 'J'};
	private static final byte[] ESC_JIS6226 = new byte[] {0x1b, '$', '@'};
	private static final byte[] ESC_JIS208 = new byte[] {0x1b, '$', 'B'};
	private static final byte[] ESC_JIS201 = new byte[] {0x1b, '(', 'I'};

public CharacterConverter getModeless() {
	isModal = false;
	return this;
}

public int countChars(byte[] bytes, int offset, int count) {
	if (count < 0) throw new StringIndexOutOfBoundsException();
	int mode = ASCII;
	if (isModal) {
		// assumes the bytes resume from the last call to countChars()
		mode = countMode;
	}
	int length = offset + count, total = 0;
	while (offset < length) {
		int b = bytes[offset++] & 0xff;
		if (b == 0x1b) {
			if (offset + 1 < length) {
				if (bytes[offset] == 0x28) {
					if (bytes[offset+1] == 0x42) {
						mode = ASCII;
						offset += 2;
						continue;
					} else if (bytes[offset+1] == 0x4a) {
						mode = ROMAN;
						offset += 2;
						continue;
					} else if (bytes[offset+1] == 0x49) {
						mode = JIS201;
						offset += 2;
						continue;
					}
				} else if (bytes[offset] == 0x24) {
					if (bytes[offset+1] == 0x40) {
						mode = JIS6226;
						offset += 2;
						continue;
					} else if (bytes[offset+1] == 0x42) {
						mode = JIS208;
						offset += 2;
						continue;
					}
				}
			} else {
				// not enough bytes
				break;
			}
		}
		if (mode == JIS6226 || mode == JIS208 && b >= 0x20) {
			if (offset >= length) break;
			offset++;
		}
		total++;
	}
	if (isModal) {
		countMode = mode;
	}
	return total;
}

public int convert(byte[] bytes, int offset, char[] chars, int charOffset, int total) {
	int mode = ASCII;
	if (isModal) {
		// assumes the bytes resume from the last call to convert(byte[])
		mode = byteMode;
	}
	total += charOffset;
	while (charOffset < total) {
		int b = bytes[offset++] & 0xff;
		if (b == 0x1b && offset + 1 < bytes.length) {
			if (bytes[offset] == 0x28) {
				if (bytes[offset+1] == 0x42) {
					mode = ASCII;
					offset += 2;
					continue;
				} else if (bytes[offset+1] == 0x4a) {
					mode = ROMAN;
					offset += 2;
					continue;
				} else if (bytes[offset+1] == 0x49) {
					mode = JIS201;
					offset += 2;
					continue;
				}
			} else if (bytes[offset] == 0x24) {
				if (bytes[offset+1] == 0x40) {
					mode = JIS6226;
					offset += 2;
					continue;
				} else if (bytes[offset+1] == 0x42) {
					mode = JIS208;
					offset += 2;
					continue;
				}
			}
		}
		if (mode == ASCII) {
			if (b < 128) chars[charOffset++] = (char)b;
		} else if (mode == ROMAN) {
			if (b == 0x5c) chars[charOffset++] = 0xa5;
			else if (b == 0x7e) chars[charOffset++] = 0x203e;
			else if (b < 128) chars[charOffset++] = (char)b;
		} else if (mode == JIS201) {
			if (b >= 0x21 && b <= 0x5f)
				chars[charOffset++] = (char)((0xff61 - 0x21) + b);
		} else if (b >= 0x21 && b <= 0x7e) {
			int b2;
			if ((b2 = bytes[offset++] & 0xff) >= 0x21 && b2 <= 0x7e)
				chars[charOffset++] = CharacterConverter_EUC_JP.jis208.charAt((b - 0x21) * 94 + b2 - 0x21);
			else chars[charOffset++] = 0xfffd;
		} else {
			if (b >= 0x20) offset++;
			chars[charOffset++] = 0xfffd;
		}
	}
	if (isModal) {
		byteMode = mode;
	}
	return offset;
}

public byte[] convert(char[] value, int offset, int count) {
	int mode = ASCII;
	if (isModal) {
		// assumes the chars resume from the last call to convert(char[])
		mode = charMode;
	}
	count += offset;
	ByteArrayOutputStream out = new ByteArrayOutputStream(count);
	for (int i=offset; i<count; i++) {
		char ch = value[i];
		if (ch < 128) {
			if (mode != ASCII) {
				out.write(ESC_ASCII, 0, ESC_ASCII.length);
				mode = ASCII;
			}
			out.write(ch);
		} else if (ch == 0xa5 || ch == 0x203e) {
			if (mode != ROMAN) {
				out.write(ESC_ROMAN, 0, ESC_ROMAN.length);
				mode = ROMAN;
			}
			if (ch == 0xa5) out.write(0x5c);
			else out.write(0x7e);
		} else {
			int index = BinarySearch.binarySearch(CharacterConverter_EUC_JP.keys, value[i]);
			if (index == -1) {
				if (mode != ASCII) {
					out.write(ESC_ASCII, 0, ESC_ASCII.length);
					mode = ASCII;
				}
				out.write('?');
			} else {
				int cp = CharacterConverter_EUC_JP.values.charAt(index);
				byte high = (byte)(cp >> 8);
				if (high < 0 && high >= (byte)0xa0) {
					if (mode != JIS208) {
						out.write(ESC_JIS208, 0, ESC_JIS208.length);
						mode = JIS208;
					}
					out.write(high - 0x80);
					out.write(cp - 0x80);
				} else if (high == (byte)0x8e) {
					if (mode != JIS201) {
						out.write(ESC_JIS201, 0, ESC_JIS201.length);
						mode = JIS201;
					}
					out.write(cp - 0x80);
				} else {
					if (mode != ASCII) {
						out.write(ESC_ASCII, 0, ESC_ASCII.length);
						mode = ASCII;
					}
					out.write('?');
				}
			}
		}
	}
	if (isModal) {
		charMode = mode;
	} else {
		if (mode != ASCII) {
			out.write(ESC_ASCII, 0, ESC_ASCII.length);
		}
	}
	return out.toByteArray();
}

public byte[] getClosingBytes() {
	if (charMode != ASCII) {
		return ESC_ASCII;
	}
	return EMPTY;
}

}
