//////////////////////////////////////////////////////////////////////////////
//
// PocketHAL
// Copyright 2004 by Thierry Tremblay
//
//////////////////////////////////////////////////////////////////////////////

#include "GameBase.h"



namespace PHAL
{

//////////////////////////////////////////////////////////////////////////////
//
// GameBase
//
//////////////////////////////////////////////////////////////////////////////


void GameBase::ConstructL()
{
    BaseConstructL();

    m_view = new (ELeave) SymbianView;
#if defined(PHAL_SYMBIAN_SERIE60)
    m_view->SetMopParent(this);
#endif
    m_view->ConstructL( ClientRect(), this );

    AddToStackL( m_view );

    if (!Run())
    {
        User::Leave( KErrGeneral );
    }
}



GameBase::~GameBase()
{
	if (m_idleTimer)
    {
        m_idleTimer->Cancel();
	    delete m_idleTimer;
    }

    if (m_view)
    {
        RemoveFromStack( m_view );
        delete m_view;
    }
}




void GameBase::HandleCommandL( TInt aCommand )
{
    switch ( aCommand )
    {
    case EEikCmdExit:
        {
            OnShutdown();
            Exit();
        }
        break;
        
    default:
        break;      
    }
}



void GameBase::HandleForegroundEventL( TBool aForeground )
{
    if (aForeground)
    {
#if defined(PHAL_SYMBIAN_SERIE60)
        SetKeyBlockMode( ENoKeyBlock );
#endif
        m_idleTimer->Start( TCallBack( IdleTimerCallback, this) );
        m_lastYield = GetClock();

        OnResume();
    }
    else
    {
		m_idleTimer->Cancel();
#if defined(PHAL_SYMBIAN_SERIE60)
        SetKeyBlockMode( EDefaultBlockMode );
#endif
        OnSuspend();
    }

    Superclass::HandleForegroundEventL( aForeground );
}



TInt GameBase::IdleTimerCallback( TAny* AppUI )
{
    GameBase* game = static_cast<GameBase*>( AppUI );

    if (!game->OnGameLoop())
    {
        game->OnShutdown();
        game->Exit();
        return 0;
    }
    else
    {
        // Yield once in a while (so that we don't starve the Windows Server and cause panic #11)
        uint32_t now = GetClock();
        if (now - game->m_lastYield >= 2500)
        {
            game->m_lastYield = now;

            // Let the system run a bit
            User::After(0);
        }

        return 1;
    }
}



bool GameBase::Initialize( const Config& config )
{
    m_idleTimer = CIdle::New( CActive::EPriorityIdle );

    return true;
}



RWindow& GameBase::GetWindow() const
{
    RWindow* window = static_cast<RWindow*>( m_view->DrawableWindow() );
    return *window;
}



bool GameBase::Run()
{
    return true;
}



void GameBase::ResetInactivity() const
{
    // Reset inactivity timeout
    TRawEvent event; 
    event.Set(TRawEvent::EActive); 
    UserSvr::AddEvent(event); 
}




//////////////////////////////////////////////////////////////////////////////
//
// SymbianView
//
//////////////////////////////////////////////////////////////////////////////

void SymbianView::ConstructL( const TRect&, GameBase* game )
{
    // Keep game pointer around
    m_game = game;

    // Create a window for this application view
    CreateWindowL();

    // Set the windows size
	SetExtentToWholeScreen();

    // Enable drag events
    EnableDragEvents();

    // Activate the window, which makes it ready to be drawn
    ActivateL();
}



TKeyResponse SymbianView::OfferKeyEventL( const TKeyEvent& event, TEventCode type )
{
    switch (type)
    {
    case EEventKeyDown:
        m_game->OnKeyDown( event.iScanCode );
        break;

    case EEventKeyUp:
        m_game->OnKeyUp( event.iScanCode );
        break;

    case EEventKey:
        break;

    default:
        (void)0;
    }

    return EKeyWasConsumed;
}



void SymbianView::HandlePointerEventL( const TPointerEvent& event )
{
    switch (event.iType)
    {
    case TPointerEvent::EButton1Down:
        m_game->OnStylusDown( Point( event.iPosition.iX, event.iPosition.iY ) );
        break;

    case TPointerEvent::EButton1Up:
        m_game->OnStylusUp( Point( event.iPosition.iX, event.iPosition.iY ) );
        break;

    case TPointerEvent::EDrag:
        m_game->OnStylusMove( Point( event.iPosition.iX, event.iPosition.iY ) );
        break;

    default:
        (void)0;
    }
}




} // end of namespace PHAL
